///-*-C++-*-//////////////////////////////////////////////////////////////////
//
// Hoard: A Fast, Scalable, and Memory-Efficient Allocator
//        for Shared-Memory Multiprocessors
// Contact author: Emery Berger, http://www.cs.utexas.edu/users/emery
//
// Copyright (c) 1998, 1999, The University of Texas at Austin.
//
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Library General Public License as
// published by the Free Software Foundation, http://www.fsf.org.
//
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
//////////////////////////////////////////////////////////////////////////////

#include <limits.h>
#include <string.h>

#include "config.h"

#include "heap.h"
#include "threadheap.h"
#include "processheap.h"


threadHeap::threadHeap (void)
  : pHeap (0)
{}


// malloc (sz):
//   inputs: the size of the object to be allocated.
//   returns: a pointer to an object of the appropriate size.
//   side effects: allocates a block from a superblock;
//                 may call sbrk() (via makeSuperblock).

void * threadHeap::malloc (const size_t size)
{
  size_t sz = (size == 0) ? 1 : size;
  const int sizeclass = sizeClass (sz);

  lock (sizeclass);

  block * b = NULL;

  while (!b) {

    // We usually have memory locally (when u <= a),
    // so we skip checking to see if this is true
    // and just look for space in the superblock list.
    
    superblock * sb = findAvailableSuperblock (sizeclass, b);
    
    if (!sb) {
      
      // We don't have memory locally.
      // (This suggests that u == a.)
      // Try to get more from the process heap.
      
      assert (pHeap);
      
      sb = pHeap->acquire ((int) sizeclass, this);
      
      int allocated = 0;
      
      if (!sb) {
	// We didn't get any memory from the process heap.
	// We'll have to allocate our own superblock.
	sb = superblock::makeSuperblock (sizeclass);
#if HEAP_FRAG_STATS
	pHeap->setAllocated (0, sb->getNumBlocks() * sizeFromClass(sb->getBlockSizeClass()));
#endif
	allocated = sb->getNumBlocks ();
	assert (sb);
      }
      
      // Insert the superblock into our list.
      
      insertSuperblock (sizeclass, sb, 0);
      
      // Get a block from the superblock.
      b = sb->getBlock ();
      
      // If we allocated a superblock,
      // we definitely should get a block.
      // I.e., allocated => b.
      assert (!allocated || b);
    }
  }

  assert (b->isValid());

  b->markAllocated();

#if HEAP_FRAG_STATS
  b->setRequestedSize (align(size));
  pHeap->setAllocated (align(size), 0);
#endif

  // Update the stats.
  
  incUStats (sizeclass);

  unlock (sizeclass);
  
  // Skip past the block header,
  // and return the pointer.
  
  return (void *) (b + 1);
}


// free (ptr, pheap):
//   inputs: a pointer to an object allocated by malloc(), above.
//   side effects: returns the block to the object's superblock;
//                 updates the thread heap's statistics;
//                 may release the superblock to the process heap.


void threadHeap::free (void * ptr)
{

  // Return if ptr is 0.
  // This is the behavior prescribed by the standard.
  if (ptr == 0) {
    return;
  }

  // Find the block and superblock corresponding to this ptr.

  block * const b = (block *) ptr - 1;
  assert (b->isValid());
  b->markFree();

  superblock * const sb = b->getSuperblock();
  assert (sb);

#if HEAP_FRAG_STATS
  pHeap->setDeallocated (b->getRequestedSize(), 0);
#endif

  const int sizeclass = sb->getBlockSizeClass();

  // Return the block to the superblock,
  // find the heap that owns this superblock
  // and update its statistics.

  hoardHeap * owner;

  // Lock only the owner of this superblock.
  // Since the owner might change between executing
  // getOwner() and locking the owner's heap,
  // we need this loop to make sure we lock the right heap.

  while (1) {
    owner = sb->getOwner();
    if (owner) {
      owner->lock (sizeclass);
      if (sb->getOwner() != owner) {
	owner->unlock (sizeclass);
      } else {
	break;
      }
    }
  }

  sb->putBlock (b);

  // If the owner is the process heap itself,
  // we're done.

  if (owner == pHeap) {
    owner->unlock (sizeclass);
    return;
  }

  // Otherwise, we need to decrement U.
  int u;
  int a;
  owner->decUStats (sizeclass, u, a);

  // Release a superblock, if necessary.

  static const int numberOfBytesSizeClass = sizeClass(NUMBER_OF_BYTES);

  if (sizeclass > numberOfBytesSizeClass) {
    // We ALWAYS release "big" superblocks.
    owner->decStats (sizeclass,
		     sb->getNumBlocks() - sb->getNumAvailable(),
		     sb->getNumBlocks());

    owner->removeSuperblock (sb, sizeclass);

    // Give the superblock back to the process heap.
    pHeap->lock (sizeclass);
    pHeap->release (sb);
    pHeap->unlock (sizeclass);

    owner->unlock (sizeclass);
    return;
  }

  const int threshold
    = 2 * (MAX (1, NUMBER_OF_BYTES / sizeFromClass (sizeclass)));

  if (a > u + threshold) {
    
    // We've crossed the magical threshold.
    // Find the "best" superblock and give it
    // to the process heap.

    superblock * const maxSb = owner->removeMaxSuperblock (sizeclass);

    if (maxSb) {

      // Update the statistics.

      owner->decStats (sizeclass,
		       maxSb->getNumBlocks() - maxSb->getNumAvailable(),
		       maxSb->getNumBlocks());

      // Give the superblock back to the process heap.
      pHeap->lock (sizeclass);
      pHeap->release (maxSb);
      pHeap->unlock (sizeclass);
    }
  }

  owner->unlock (sizeclass);
}
