///-*-C++-*-//////////////////////////////////////////////////////////////////
//
// Hoard: A Fast, Scalable, and Memory-Efficient Allocator
//        for Shared-Memory Multiprocessors
// Contact author: Emery Berger, http://www.cs.utexas.edu/users/emery
//
// Copyright (c) 1998, 1999, The University of Texas at Austin.
//
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Library General Public License as
// published by the Free Software Foundation, http://www.fsf.org.
//
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
//////////////////////////////////////////////////////////////////////////////

/*
  superblock.h
  ------------------------------------------------------------------------
  The superblock class controls a number of blocks (which are
  allocatable units of memory).
  ------------------------------------------------------------------------
  @(#) $Id: superblock.h,v 1.17 1999/10/29 13:35:08 emery Exp $
  ------------------------------------------------------------------------
  Emery Berger                    | <http://www.cs.utexas.edu/users/emery>
  Department of Computer Sciences |             <http://www.cs.utexas.edu>
  University of Texas at Austin   |                <http://www.utexas.edu>
  ========================================================================
*/

#ifndef _SUPERBLOCK_H_
#define _SUPERBLOCK_H_

#include "config.h"

#include <assert.h>
#include <new.h>
#include <stdio.h>
#include <stdlib.h>

#include "arch-specific.h"
#include "block.h"

class hoardHeap; // forward declaration


class superblock {

public:

  // Construct a superblock for a given size class,
  // and set the owner to heap o.
  superblock (int numblocks,
	      int sizeclass,
	      hoardHeap * owner);

  ~superblock (void)
    {}

  // Make (allocate) a superblock for a given size class.
  static superblock * makeSuperblock (int sizeclass);

  // Find out who allocated this superblock.
  inline hoardHeap * getOwner (void) const;

  // Set the superblock's owner.
  inline void setOwner (hoardHeap * o);

  // Get a block from the superblock.
  inline block * getBlock (void);

  // Put a block back in the superblock.
  inline void putBlock (block * b);

  // How many blocks are available?
  inline int getNumAvailable (void) const;

  // How many blocks are there, in total?
  inline const int getNumBlocks (void) const;

  // What size class are blocks in this superblock?
  inline const size_t getBlockSizeClass (void) const;

  // Insert this superblock before the next one.
  inline void insertBefore (superblock * nextSb);

  // Remove this superblock from its linked list.
  inline void remove (void);
  
  // Return the next pointer (to the next superblock in the list).
  inline superblock * const getNext (void) const;

  // Return the prev pointer (to the previous superblock in the list).
  inline superblock * const getPrev (void) const;

#if HEAP_FRAG_STATS
  // Return the amount of waste in every allocated block.
  int getMaxInternalFragmentation (void);
#endif


private:

  // Disable copying and assignment.

  superblock (const superblock&);
  const superblock& operator= (const superblock&);

  double _pad[CACHE_LINE / sizeof(double)];	// Cache pad.

  const int 	_sizeClass;	// The size class of blocks in the superblock.
  const int 	_numBlocks;	// The number of blocks in the superblock.
  int		_numAvailable;	// The number of blocks available.
  block *	_freeList;	// A pointer to the first free block.
  hoardHeap * 	_owner;		// The heap who owns this superblock.
  superblock * 	_next;		// The next superblock in the list.
  superblock * 	_prev;		// The previous superblock in the list.

  // The data for the superblock starts here.
};


hoardHeap * superblock::getOwner (void) const
{
  hoardHeap * o = _owner;
  return o;
}


void superblock::setOwner (hoardHeap * o) 
{
  // Either the current owner is NULL or the new owner will be.
  assert (!_owner || !o);
  _owner = o;
}


block * superblock::getBlock (void)
{
  // Pop off a block from this superblock's freelist,
  // if there is one available.
  if (_freeList == NULL) {
    // The freelist is empty.
    return NULL;
  }
  assert (getNumAvailable() > 0);
  block * b = _freeList;
  _freeList = _freeList->getNext();
  // Make sure the block is double-word aligned.
  assert ((((unsigned int) b) & 7U) == 0);
  _numAvailable--;
  return b;
}


void superblock::putBlock (block * b)
{
  // Push a block onto the superblock's freelist.
#if HEAP_DEBUG
  assert (b->isValid());
#endif
  assert (b->getSuperblock() == this);
  assert (getNumAvailable() < getNumBlocks());
  b->setNext (_freeList);
  _freeList = b;
  _numAvailable++;
}

int superblock::getNumAvailable (void) const
{
  return _numAvailable;
}


const int superblock::getNumBlocks (void) const 
{
  return _numBlocks;
}


const size_t superblock::getBlockSizeClass (void) const 
{
  return _sizeClass;
}


superblock * const superblock::getNext (void) const
{
  return _next; 
}

superblock * const superblock::getPrev (void) const 
{
  return _prev; 
}


void superblock::insertBefore (superblock * nextSb) {
  // Insert this superblock before the next one (nextSb).
  assert (nextSb != this);
  _next = nextSb;
  if (nextSb) {
    _prev = nextSb->_prev;
    nextSb->_prev = this;
  }
}


void superblock::remove (void) {
  // Remove this superblock from a doubly-linked list.
  if (_next) {
    _next->_prev = _prev;
  }
  if (_prev) {
    _prev->_next = _next;
  }
  _prev = NULL;
  _next = NULL;
}



#endif // _SUPERBLOCK_H_
