///-*-C++-*-//////////////////////////////////////////////////////////////////
//
// Hoard: A Fast, Scalable, and Memory-Efficient Allocator
//        for Shared-Memory Multiprocessors
// Contact author: Emery Berger, http://www.cs.utexas.edu/users/emery
//
// Copyright (c) 1998, 1999, The University of Texas at Austin.
//
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Library General Public License as
// published by the Free Software Foundation, http://www.fsf.org.
//
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
//////////////////////////////////////////////////////////////////////////////

/*
  superblock.cpp
  ------------------------------------------------------------------------
  The superblock class controls a number of blocks (which are
  allocatable units of memory).
  ------------------------------------------------------------------------
  @(#) $Id: superblock.cpp,v 1.11 1999/10/04 16:56:22 emery Exp $
  ------------------------------------------------------------------------
  Emery Berger                    | <http://www.cs.utexas.edu/users/emery>
  Department of Computer Sciences |             <http://www.cs.utexas.edu>
  University of Texas at Austin   |                <http://www.utexas.edu>
  ========================================================================
*/

#include <new.h>
#include <string.h>

#include "arch-specific.h"
#include "config.h"
#include "heap.h"
#include "superblock.h"


superblock::superblock (int numBlocks,	// The number of blocks in the sb.
			int szclass,	// The size class of the blocks.
			hoardHeap * o)	// The heap that "owns" this sb.
  :
    _sizeClass (szclass),
    _numBlocks (numBlocks),
    _numAvailable (0),
    _freeList (NULL),
    _owner (o),
    _next (NULL),
    _prev (NULL)
{
  assert (_numBlocks >= 1);

  // Determine the size of each block.
  const int blksize =
    hoardHeap::align (sizeof(block) + hoardHeap::sizeFromClass(_sizeClass));

  assert ((blksize & hoardHeap::ALIGNMENT_MASK) == 0);

  // Set the first block to just past this superblock header.
  block * b
    = (block *) hoardHeap::align ((unsigned int) (this + 1));

  // Initialize all the blocks,
  // and insert the block pointers into the linked list.
  for (int i = 0; i < _numBlocks; i++) {
    // Make sure the block is on a double-word boundary.
    assert (((unsigned int) b & hoardHeap::ALIGNMENT_MASK) == 0);
    new (b) block (this);
    assert (b->getSuperblock() == this);
    putBlock (b);
    b = (block *) ((char *) b + blksize);
  }
}


superblock * superblock::makeSuperblock (int sizeclass)
{
  // Determine how many blocks will go in this superblock.
  const size_t s = hoardHeap::sizeFromClass (sizeclass);
  int numBlocks = MAX (1, (hoardHeap::NUMBER_OF_BYTES / s));
  assert (s > 0);

  // We need to extend the break value to get more memory.

  // First, round up blksize.
  const int blksize =
    hoardHeap::align (sizeof(block) + s);

  // The amount of memory needed = 
  const long moreMemory =
    hoardHeap::align (sizeof(superblock)  //   one superblock (header)
    + blksize * numBlocks);	  	  // + n blocks (headers + data).

  char * buf = (char *) hoardSbrk (moreMemory + hoardHeap::ALIGNMENT - 1);

  // Make sure that we actually got the memory.
  if (buf == (char *) -1) {
    perror ("heap error");
    return 0;
  }

  // Make sure this is double-word aligned.
  buf = (char *) hoardHeap::align ((unsigned int) buf);
  assert ((((unsigned int) buf) & hoardHeap::ALIGNMENT_MASK) == 0);

  // Instantiate the new superblock in the allocated buffer.
  superblock * sb = new (buf) superblock (numBlocks, sizeclass, NULL);
  return sb;
}
