///-*-C++-*-//////////////////////////////////////////////////////////////////
//
// Hoard: A Fast, Scalable, and Memory-Efficient Allocator
//        for Shared-Memory Multiprocessors
// Contact author: Emery Berger, http://www.cs.utexas.edu/users/emery
//
// Copyright (c) 1998, 1999, The University of Texas at Austin.
//
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Library General Public License as
// published by the Free Software Foundation, http://www.fsf.org.
//
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
//////////////////////////////////////////////////////////////////////////////

/*
  processheap.h
  ------------------------------------------------------------------------
  We use one processHeap for the whole program.
  ------------------------------------------------------------------------
  @(#) $Id: processheap.h,v 1.14 1999/10/27 20:43:19 emery Exp $
  ------------------------------------------------------------------------
  Emery Berger                    | <http://www.cs.utexas.edu/users/emery>
  Department of Computer Sciences |             <http://www.cs.utexas.edu>
  University of Texas at Austin   |                <http://www.utexas.edu>
  ========================================================================
*/

#ifndef _PROCESSHEAP_H_
#define _PROCESSHEAP_H_

#include "config.h"

#include <stdio.h>

#include "arch-specific.h"
#include "heap.h"
#if USE_PRIVATE_HEAPS
#include "privateheap.h"
#define HEAPTYPE privateHeap
#else
#define HEAPTYPE threadHeap
#include "threadheap.h"
#endif

class processHeap : public hoardHeap {

public:

  processHeap (void);
  ~processHeap (void);

  // Print out statistics information.
  void stats (void);

  // Get a thread heap.
  inline HEAPTYPE& allocator (void);

  // Extract a superblock.
  inline superblock * acquire (const int c, HEAPTYPE * to);

  // Insert a superblock.
  inline void release (superblock * sb);

#if HEAP_FRAG_STATS
  // Declare that we have allocated an object.
  void setAllocated (int requestedSize,
		     int actualSize);

  // Declare that we have deallocated an object.
  void setDeallocated (int requestedSize,
		       int actualSize);

  // Return the number of wasted bytes at the high-water mark
  // (maxAllocated - maxRequested)
  inline int getFragmentation (void);

  int getMaxAllocated (void) {
    return _maxAllocated;
  }

  int getMaxRequested (void) {
    return _maxRequested;
  }
  
#endif

private:

  // Get the thread heap with index i.
  inline HEAPTYPE& getHeap (int i);

  // Prevent copying and assignment.
  processHeap (const processHeap&);
  const processHeap& operator= (const processHeap&);

  // We use this mask for rounding to MAX_THREADS.
  enum { MAX_THREADS_MASK = MAX_THREADS - 1 };

  HEAPTYPE theap[MAX_THREADS];	// The per-thread heaps.

#if HEAP_FRAG_STATS
  int _currentAllocated;
  int _currentRequested;
  int _maxAllocated;
  int _maxRequested;
  int _fragmentation;

  hoardLock _statsLock;		// A lock to protect these statistics.
#endif

};


HEAPTYPE& processHeap::getHeap (int i)
{
  assert (i >= 0);
  assert (i < MAX_THREADS);
  return theap[i];
}


HEAPTYPE& processHeap::allocator (void) {
#if USE_SUBHEAPS
  int i = hoardGetThreadID();
  while (1) {
    if (getHeap (i & MAX_THREADS_MASK).acquireHeap ()) {
      return getHeap (i & MAX_THREADS_MASK);
    }
    i++;
  }
#else      
#ifdef _REENTRANT
#ifdef __linux
  return getHeap ((hoardGetThreadID() / 1024) & MAX_THREADS_MASK);
#else
  const int index = (int) hoardGetThreadID() & MAX_THREADS_MASK;
  return getHeap (index);
#endif // __linux
#endif // _REENTRANT
#endif // USE_SUBHEAPS
}


superblock * processHeap::acquire (const int sizeclass,
				   HEAPTYPE * to)
{
  // This removes the warning when assertions are turned off.
  to = to;

  // We can't acquire a superblock from ourselves.
  assert ((hoardHeap *) to != (hoardHeap *) this);

  // Remove the superblock with the most free space.
  lock (sizeclass);
  superblock * maxSb = removeMaxSuperblock (sizeclass);
  unlock (sizeclass);

  return maxSb;
}


// Put a superblock back into our list of superblocks.
void processHeap::release (superblock * sb)
{
  // Insert the superblock
  // (the last argument is a 1 because it's going onto the process heap).
  insertSuperblock (sb->getBlockSizeClass(), sb, 1);
}


#endif // _PROCESSHEAP_H_

