///-*-C++-*-//////////////////////////////////////////////////////////////////
//
// Hoard: A Fast, Scalable, and Memory-Efficient Allocator
//        for Shared-Memory Multiprocessors
// Contact author: Emery Berger, http://www.cs.utexas.edu/users/emery
//
// Copyright (c) 1998, 1999, The University of Texas at Austin.
//
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Library General Public License as
// published by the Free Software Foundation, http://www.fsf.org.
//
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
//////////////////////////////////////////////////////////////////////////////

#include <string.h>
#include <stdio.h>

#include "config.h"

#if USE_PRIVATE_HEAPS
#include "privateheap.h"
#define HEAPTYPE privateHeap
#else
#define HEAPTYPE threadHeap
#include "threadheap.h"
#endif

#include "processheap.h"

processHeap::processHeap (void)
#if HEAP_FRAG_STATS
  : _currentAllocated (0),
    _currentRequested (0),
    _maxAllocated (0),
    _maxRequested (0),
    _fragmentation (0)
#endif
{
  for (int i = 0; i < MAX_THREADS; i++) {
    theap[i].setpHeap (this);
  }
#if HEAP_FRAG_STATS
  hoardLockInit (_statsLock);
#endif
}


processHeap::~processHeap (void)
{
#if HEAP_STATS
  stats();
#endif
}


void processHeap::stats (void) {
#if HEAP_STATS

  int umax = 0;
  int amax = 0;
  for (int j = 0; j < MAX_THREADS; j++) {
    for (int i = 0; i < SIZE_CLASSES; i++) {
      amax += theap[j].maxAllocated(i) * sizeFromClass (i);
      umax += theap[j].maxInUse(i) * sizeFromClass (i);
    }
  }
  printf ("Amax <= %d, Umax <= %d, blowup <= %f (+ %d)\n",
	  amax, umax, (float) amax / (float) umax, amax - umax);
#if HEAP_FRAG_STATS
  amax = getMaxAllocated();
  umax = getMaxRequested();
  printf ("Actual amax = %d, actual umax = %d, blowup = %f\n",
	  amax, umax, (float) amax / (float) umax);
  int frag = getFragmentation();
  printf ("Internal fragmentation = %d\n", frag);
#endif

#endif // HEAP_STATS
}



#if HEAP_FRAG_STATS
void processHeap::setAllocated (int requestedSize,
				int actualSize)
{
  hoardLock (_statsLock);
  _currentRequested += requestedSize;
  _currentAllocated += actualSize;
  if (_currentRequested > _maxRequested) {
    _maxRequested = _currentRequested;
  }
  if (_currentAllocated > _maxAllocated) {
    _maxAllocated = _currentAllocated;
    _fragmentation = _maxAllocated - _maxRequested;
  }
  hoardUnlock (_statsLock);
}


void processHeap::setDeallocated (int requestedSize,
				  int actualSize)
{
  hoardLock (_statsLock);
  _currentRequested -= requestedSize;
  _currentAllocated -= actualSize;
  hoardUnlock (_statsLock);
}


int processHeap::getFragmentation (void)
{
  return _fragmentation;
}
#endif
