// -*- C++ -*-

#ifndef _PRIVATEHEAP_H_
#define _PRIVATEHEAP_H_

#include <string.h>

#include "config.h"
#include "arch-specific.h"
#include "heap.h"

class processHeap;

class privateHeap : public hoardHeap {

public:

  privateHeap (void);

  // Set our process heap.
  inline void setpHeap (processHeap * p);

  // Memory allocation and deallocation routines.
  void * malloc (const size_t sz);
  inline void * memalign (size_t alignment, size_t sz);
  void free (void * ptr);

  // Find out how large an allocated object is.
  inline static size_t objectSize (void * ptr);

  int lock (void) {
#if USE_SUBHEAPS
    // Return 1 if we get the lock.
    if (hoardTryLock (_lock) != 0) {
      // Somebody else has locked it.
      return 0;
    } else {
      return 1;
    }
#else
    hoardLock (_lock);
    return 0;
#endif
  }

private:

  enum { HEAP_REFILL_SIZE = 1048576 }; // 1 MB

  // Prevent copying and assignment.
  privateHeap (const privateHeap&);
  const privateHeap& operator= (const privateHeap&);

  void * _arena; 	// Current arena of available memory.
  int _arenaRemaining;
  int _inUse;
  int _maxInUse;
  hoardLockType _lock;

  void unlock (void) { hoardUnlock (_lock); }

  block * _blocks[hoardHeap::SIZE_CLASSES];
  processHeap *	pHeap;	// Our process heap.

  double _pad[CACHE_LINE / sizeof(double)];	// Cache pad.
};


void * privateHeap::memalign (size_t alignment,
			     size_t size)
{
  // Calculate the amount of space we need
  // to satisfy the alignment requirements.

  while (alignment != hoardHeap::align (alignment))
    alignment <<= 1;

  size = size + alignment - 1;

  // Now malloc the space up with a little extra
  // (we'll put the block pointer in right behind
  // the allocated space).

  void * ptr = malloc (size + sizeof(block));

  void * newptr = (void *)
    ((((unsigned int) ptr + alignment - 1) / alignment) * alignment);

  // Copy the block from the start of the allocated memory.

  block * b = ((block *) ptr - 1);
  block * p = ((block *) newptr - 1);
  memcpy (p, b, sizeof(block));

  return newptr;
}


size_t privateHeap::objectSize (void * ptr) 
{
  block * b = ((block *) ptr - 1);
  assert (b->isValid());
  return b->getActualSize();
}


void privateHeap::setpHeap (processHeap * p) 
{
  pHeap = p; 
}

#endif // _PRIVATEHEAP_H_

