///-*-C++-*-//////////////////////////////////////////////////////////////////
//
// Hoard: A Fast, Scalable, and Memory-Efficient Allocator
//        for Shared-Memory Multiprocessors
// Contact author: Emery Berger, http://www.cs.utexas.edu/users/emery
//
// Copyright (c) 1998, 1999, The University of Texas at Austin.
//
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Library General Public License as
// published by the Free Software Foundation, http://www.fsf.org.
//
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
//////////////////////////////////////////////////////////////////////////////

/*
  heap.h
  ------------------------------------------------------------------------
  hoardHeap, the base class for threadHeap and processHeap.
  ------------------------------------------------------------------------
  @(#) $Id: heap.h,v 1.18 1999/10/29 13:31:06 emery Exp $
  ------------------------------------------------------------------------
  Emery Berger                    | <http://www.cs.utexas.edu/users/emery>
  Department of Computer Sciences |             <http://www.cs.utexas.edu>
  University of Texas at Austin   |                <http://www.utexas.edu>
  ========================================================================
*/


#ifndef _HEAP_H_
#define _HEAP_H_

#include "config.h"

#include <assert.h>
#include <math.h>

#include "arch-specific.h"
#include "superblock.h"
#include "heapstats.h"

class hoardHeap {

public:

  hoardHeap (void);

  // A superblock contains at least this many bytes.
  enum { NUMBER_OF_BYTES = 32768 - sizeof(superblock) }; 

  // The maximum number of thread heaps we allow.
  // This must be a power of two!
  enum { MAX_THREADS = 64 };

  // The number of size classes.
  // This combined with the SIZE_CLASS_BASE
  // determine the maximum size of an object.
  enum { SIZE_CLASSES = 128 };

  enum { ALIGNMENT = sizeof(double) };	  // Every object is double-word aligned.
  enum { ALIGNMENT_MASK = ALIGNMENT - 1}; // ANDing with this rounds to ALIGNMENT.
  enum { HEAP_MAGIC = 0x0badcafe };	// Used for sanity checking.

  // Update memory in-use and allocated statistics.
  // (*UStats = just update U.)
  inline void incStats (int sizeclass, int updateU, int updateA);
  inline void incUStats (int sizeclass);

  inline void decStats (int sizeclass, int updateU, int updateA);
  inline void decUStats (int sizeclass, int& U, int& A);

  inline void getStats (int sizeclass, int& U, int& A);


#if HEAP_STATS
  // How much is the maximum ever in use for this size class?
  inline int maxInUse (int sizeclass);

  // How much is the maximum memory allocated for this size class?
  inline int maxAllocated (int sizeclass);
#endif

  // Insert a superblock into our list.
  // isProcessHeap should be non-zero if
  // the caller is the process heap.
  void insertSuperblock (int sizeclass,
			 superblock * sb,
			 int isProcessHeap);

  // Remove the superblock with the most free space.
  superblock * removeMaxSuperblock (int sizeclass);

  // Remove a particular superblock.
  void removeSuperblock (superblock *, int sizeclass);

  // Find an available superblock (i.e., with some space in it).
  inline superblock * findAvailableSuperblock (int sizeclass,
					       block *& b);

  // Return the size class for a given size.
  inline static const int sizeClass (const size_t sz);

  // Return the size corresponding to a given size class.
  inline static const int sizeFromClass (const int sizeclass);

  // Align a value.
  inline static size_t align (const size_t sz);

  // Lock (for our sizeclass).
  inline void lock (const int sizeclass);

  // Unlock (for our sizeclass).
  inline void unlock (const int sizeclass);

protected:

#if HEAP_DEBUG
  // For sanity checking.
  const unsigned long _magic;
#else
  #define _magic HEAP_MAGIC
#endif

  // Heap statistics.
  heapStats	_stats[SIZE_CLASSES];

  // Per size-class locks.
  hoardLockType _lock[SIZE_CLASSES];

private:

  // Disable copying and assignment.

  hoardHeap (const hoardHeap&);
  const hoardHeap& operator= (const hoardHeap&);

  // Lists of superblocks.
  superblock *	_superblocks[SIZE_CLASSES];

  // The lookup table for size classes.
  static size_t	_sizeTable[SIZE_CLASSES];

};


void hoardHeap::incStats (int sizeclass, int updateU, int updateA) {
  assert (_magic == HEAP_MAGIC);
  assert (updateU >= 0);
  assert (updateA >= 0);
  _stats[sizeclass].incStats (updateU, updateA);
}



void hoardHeap::incUStats (int sizeclass) {
  assert (_magic == HEAP_MAGIC);
  _stats[sizeclass].incUStats ();
}


void hoardHeap::decStats (int sizeclass, int updateU, int updateA) {
  assert (_magic == HEAP_MAGIC);
  assert (updateU >= 0);
  assert (updateA >= 0);
  _stats[sizeclass].decStats (updateU, updateA);
}


void hoardHeap::decUStats (int sizeclass,
		      int& U,
		      int& A)
{
  assert (_magic == HEAP_MAGIC);
  _stats[sizeclass].decUStats (U, A);
}


void hoardHeap::getStats (int sizeclass, int& U, int& A) {
  assert (_magic == HEAP_MAGIC);
  _stats[sizeclass].getStats (U, A);
}


#if HEAP_STATS
int hoardHeap::maxInUse (int sizeclass) {
  assert (_magic == HEAP_MAGIC);
  return _stats[sizeclass].getUmax();
}


int hoardHeap::maxAllocated (int sizeclass) {
  assert (_magic == HEAP_MAGIC);
  return _stats[sizeclass].getAmax(); 
}
#endif


superblock * hoardHeap::findAvailableSuperblock (int sizeclass,
						 block *& b)
{
  assert (_magic == HEAP_MAGIC);
  assert (sizeclass >= 0);
  assert (sizeclass < SIZE_CLASSES);

  superblock * sb;

  sb = _superblocks[sizeclass];

  // Iterate over the entire superblock list,
  // looking for an available superblock.
  while (sb) {
    assert (sb->getOwner() == this);
    b = sb->getBlock();
    if (b) {
      // We found a superblock with space.
      // Move this superblock to the front of the list
      // (if it isn't there already).
      if (sb != _superblocks[sizeclass]) {
	sb->remove();
	sb->insertBefore (_superblocks[sizeclass]);
	_superblocks[sizeclass] = sb;
      }
      break;
    }
    sb = sb->getNext();
  }

  // Either we didn't find a superblock or we did and got a block.
  assert (!sb || b);
  // Either we didn't get a block or we did and we also got a superblock.
  assert (!b || sb);

  return sb;
}


const int hoardHeap::sizeClass (const size_t sz) {
  // Find the size class for a given object size
  // (the smallest i such that _sizeTable[i] >= sz).
  int sizeclass = 0;
  while (_sizeTable[sizeclass] < sz) 
    {
      sizeclass++;
    }
  return sizeclass;
}


const int hoardHeap::sizeFromClass (const int sizeclass) {
  assert (sizeclass >= 0);
  assert (sizeclass < SIZE_CLASSES);
  return _sizeTable[sizeclass];
}


void hoardHeap::lock (const int sizeclass) 
{
  assert (_magic == HEAP_MAGIC);
  assert (sizeclass >= 0);
  assert (sizeclass < SIZE_CLASSES);
  hoardLock (_lock[sizeclass]);
}


void hoardHeap::unlock (const int sizeclass) {
  assert (_magic == HEAP_MAGIC);
  assert (sizeclass >= 0);
  assert (sizeclass < SIZE_CLASSES);
  hoardUnlock (_lock[sizeclass]);
}


size_t hoardHeap::align (const size_t sz)
{
  // Align sz up to the nearest multiple of ALIGNMENT.
  // This is much faster than using multiplication
  // and division.
  return (sz + ALIGNMENT_MASK) & ~ALIGNMENT_MASK;
}


#endif // _HEAP_H_
