///-*-C++-*-//////////////////////////////////////////////////////////////////
//
// Hoard: A Fast, Scalable, and Memory-Efficient Allocator
//        for Shared-Memory Multiprocessors
// Contact author: Emery Berger, http://www.cs.utexas.edu/users/emery
//
// Copyright (c) 1998, 1999, The University of Texas at Austin.
//
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Library General Public License as
// published by the Free Software Foundation, http://www.fsf.org.
//
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
//////////////////////////////////////////////////////////////////////////////
#include "config.h"

#include "heap.h"
#include "superblock.h"

static const char version[] = "The Hoard memory allocator, version 1.2 (http://www.cs.utexas.edu/users/emery/hoard). Copyright (C) 1998, 1999, The University of Texas at Austin. $Id: heap.cpp,v 1.16 1999/10/27 20:43:18 emery Exp $";

// This must be changed if SIZE_CLASSES, ALIGNMENT, or SIZE_CLASS_BASE changes.
size_t hoardHeap::_sizeTable[hoardHeap::SIZE_CLASSES] = {8UL, 16UL, 24UL, 32UL, 40UL, 48UL, 56UL, 72UL, 80UL, 96UL, 120UL, 144UL, 168UL, 200UL, 240UL, 288UL, 344UL, 416UL, 496UL, 592UL, 712UL, 856UL, 1024UL, 1232UL, 1472UL, 1768UL, 2120UL, 2544UL, 3048UL, 3664UL, 4392UL, 5272UL, 6320UL, 7584UL, 9104UL, 10928UL, 13112UL, 15728UL, 18872UL, 22648UL, 27176UL, 32616UL, 39136UL, 46960UL, 56352UL, 67624UL, 81144UL, 97376UL, 116848UL, 140216UL, 168256UL, 201904UL, 242288UL, 290744UL, 348896UL, 418672UL, 502408UL, 602888UL, 723464UL, 868152UL, 1041784UL, 1250136UL, 1500160UL, 1800192UL, 2160232UL, 2592280UL, 3110736UL, 3732880UL, 4479456UL, 5375344UL, 6450408UL, 7740496UL, 9288592UL, 11146312UL, 13375568UL, 16050680UL, 19260816UL, 23112984UL, 27735576UL, 33282688UL, 39939224UL, 47927072UL, 57512488UL, 69014984UL, 82817976UL, 99381576UL, 119257888UL, 143109472UL, 171731360UL, 206077632UL, 247293152UL, 296751776UL, 356102144UL, 427322560UL, 512787072UL, 615344512UL, 738413376UL, 886096064UL, 1063315264UL, 1275978368UL, 1531174016UL, 1837408768UL, 2204890624UL, 2645868544UL, 3175042304UL, 3810050816UL, 277093728UL, 1191505920UL, 2288800512UL, 3605554176UL, 890691136UL, 2786816256UL};


hoardHeap::hoardHeap (void)
#if HEAP_DEBUG
  : _magic (HEAP_MAGIC)
#endif
{
  for (int i = 0; i < SIZE_CLASSES; i++) {
    // Initialize all superblocks lists to empty.
    _superblocks[i] = NULL;
    // Initialized all the per-sizeclass locks.
    hoardLockInit (_lock[i]);
  }
}


void hoardHeap::insertSuperblock (int sizeclass,
				  superblock * sb,
				  int isProcessHeap)
{
  assert (_magic == HEAP_MAGIC);

  // Insert the superblock at the head of our list.

  sb->setOwner (this);
  sb->insertBefore (_superblocks[sizeclass]);
  _superblocks[sizeclass] = sb;

  if (!isProcessHeap) {
    incStats (sizeclass, sb->getNumBlocks() - sb->getNumAvailable(), sb->getNumBlocks());
  }
}


superblock * hoardHeap::removeMaxSuperblock (int sizeclass)
{
  // We're going to find the superblock
  // with the most available blocks.
  // Any superblock will have at least 0 available,
  // so this will be greater than the maximal_value below.

  int maximal_value = -1;
  superblock * maxSb = NULL;

  assert (_magic == HEAP_MAGIC);

  superblock *& head = _superblocks[sizeclass];

  if (!head) {
    return NULL;
  }

  // Find the superblock with the most available blocks.

  superblock * sb = head;

  // Look through REMOVE_MAX_ITERATIONS.
  // (If negative, this scans the whole list.)
  int niterations = REMOVE_MAX_ITERATIONS;
  while ((niterations != 0) && sb) {
    int value = sb->getNumAvailable();
    if (value > maximal_value) {
      maximal_value = value;
      maxSb = sb;
    }
    sb = sb->getNext();
    --niterations;
  }

  if (!maxSb) {
    return NULL;
  }

  // Splice out maxSb from the list.

  if (maxSb == head) {
    head = maxSb->getNext();
  }
  maxSb->remove();

  maxSb->setOwner (NULL);

  return maxSb;
}


void hoardHeap::removeSuperblock (superblock * sb,
				  int sizeclass)
{
  assert (_magic == HEAP_MAGIC);

  assert (sb->getOwner() == this);

  if (sb == _superblocks[sizeclass]) {
    _superblocks[sizeclass] = sb->getNext();
  }

  sb->remove();
  sb->setOwner (NULL);
}

