///-*-C++-*-//////////////////////////////////////////////////////////////////
//
// Hoard: A Fast, Scalable, and Memory-Efficient Allocator
//        for Shared-Memory Multiprocessors
// Contact author: Emery Berger, http://www.cs.utexas.edu/users/emery
//
// Copyright (c) 1998, 1999, The University of Texas at Austin.
//
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Library General Public License as
// published by the Free Software Foundation, http://www.fsf.org.
//
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
//////////////////////////////////////////////////////////////////////////////

#include "arch-specific.h"

#ifdef WIN32

#include <assert.h>

void hoardCreateThread (hoardThreadType& t,
			void *(*function) (void *),
			void * arg)
{
  t = CreateThread (0, 0, (LPTHREAD_START_ROUTINE) function, (LPVOID) arg, 0, 0);
}

void hoardJoinThread (hoardThreadType& t)
{
  WaitForSingleObject (t, INFINITE);
}

int hoardGetThreadID (void) {
  int tid = GetCurrentThreadId();
  return tid;
}

void hoardLockInit (hoardLockType& mutex) {
  InterlockedExchange (&mutex, 0);
}

void hoardLock (hoardLockType& mutex) {
  // A spin-lock, with a magical 100 tick
  // wait for an update on "mutex".
  while (InterlockedExchange (&mutex, 1) ==  1) {
    int count = 0;
    while (count < 100) {
      count++;
    }
  }
}

void hoardUnlock (hoardLockType& mutex) {
  InterlockedExchange (&mutex, 0);
}

int hoardTryLock (hoardLockType& mutex) {
  if (InterlockedExchange (&mutex, 1) == 0) {
    return 0;
  } else {
    return 1;
  }
}

void * hoardSbrk (long size)
{
  void * newMemory = VirtualAlloc (NULL, size, MEM_COMMIT, PAGE_READWRITE);
  assert (newMemory);
  return newMemory;
}


int hoardGetPageSize (void)
{
  SYSTEM_INFO infoReturn[1];
  GetSystemInfo (infoReturn);
  return (int) (infoReturn -> dwPageSize);
}


#else


#if USE_SPROC
#include <sys/types.h>
#include <sys/wait.h>
#include <unistd.h>
#endif


void hoardCreateThread (hoardThreadType& t,
			void *(*function) (void *),
			void * arg)
{
#if USE_SPROC
  typedef void (*sprocFunction) (void *);
  t = sproc ((sprocFunction) function, PR_SADDR, arg);
#else
  pthread_attr_t attr;
  pthread_attr_init (&attr);
#ifndef __linux
  pthread_attr_setscope (&attr, PTHREAD_SCOPE_SYSTEM); /* bound behavior */
#endif
  pthread_create (&t, &attr, function, arg);
#endif
}

void hoardJoinThread (hoardThreadType& t)
{
#if USE_SPROC
  waitpid (t, 0, 0);
#else
  pthread_join (t, NULL);
#endif
}


int hoardGetThreadID (void) {
#if USE_SPROC
  // This hairiness has the same effect as calling getpid(),
  // but it's MUCH faster since it avoids making a system call
  // and just accesses the sproc-local data directly.
  int pid = (int) PRDA->sys_prda.prda_sys.t_pid;
  return pid;
#else
  return (int) pthread_self();
#endif
}


// If we are using either Intel or SPARC,
// we use our own lock implementation
// (spin then yield). This is much cheaper than
// the ordinary mutex, at least on Linux and Solaris.

#if USER_LOCKS && (defined(i386) || defined(sparc) || defined(__sgi))

#include <sched.h>

#if defined(__sgi)
#include <mutex.h>
#endif


static inline unsigned long InterlockedExchange (unsigned long * oldval,
						 unsigned long newval)
{
#if defined(sparc)
  asm volatile ("swap [%1],%0"
		:"=r" (newval)
		:"r" (oldval), "0" (newval)
		: "memory");

#endif
#if defined(i386)
  asm volatile ("xchgl %0, %1"
		: "=r" (newval)
		: "m" (*oldval), "0" (newval)
		: "memory");
#endif
#if defined(__sgi)
  newval = test_and_set (oldval, newval);
#endif
  return newval;
}

void hoardLockInit (hoardLockType& mutex) {
  InterlockedExchange (&mutex, 0);
}

void hoardLock (hoardLockType& mutex) {
  // A blocking lock (with an initial spin).
  const int spinLimit = 100;
  int i;
  while (1) {
    i = 0;
    while (i < spinLimit && hoardTryLock (mutex)) {
      i++;
    }
    if (i == spinLimit) {
      if (InterlockedExchange (&mutex, 1) ==  1) {
	sched_yield ();
      } else {
	return;
      }
    } else {
      return;
    }
  }
}

void hoardUnlock (hoardLockType& mutex) {
  InterlockedExchange (&mutex, 0);
}

int hoardTryLock (hoardLockType& mutex) {
  if (InterlockedExchange (&mutex, 1) == 0) {
    return 0;
  } else {
    return 1;
  }
}

#else // use non-user-level locks. 

void hoardLockInit (hoardLockType& lock) {
  pthread_mutex_init (&lock, NULL);
}

void hoardLock (hoardLockType& lock) {
  // A blocking lock (with an initial spin).
  const int spinLimit = 100;
  int i = 0;
  while (i < spinLimit && hoardTryLock (lock)) {
    i++;
  }
  if (i == spinLimit) {
    pthread_mutex_lock (&lock);
  }
}

void hoardUnlock (hoardLockType& lock) {
  pthread_mutex_unlock (&lock);
}

// Returns 0 when the lock has been successfully acquired.
int hoardTryLock (hoardLockType& lock) {
  int lockIsBusy;
  lockIsBusy = pthread_mutex_trylock (&lock);
  if (lockIsBusy) {
    return 1;
  } else {
    return 0;
  }
}
#endif


void * hoardSbrk (long size) {
#if defined(__linux) || defined(__sgi)
  // Protect calls to sbrk() {we need this for Linux, but not on Solaris}
  // We don't need to acquire a lock for sbrk
  // under Solaris (it's already MT-safe).

#if USER_LOCKS && (defined(i386) || defined(sparc) || defined(__sgi))
  static hoardLockType sbrkLock = 0;
#else
  static hoardLockType sbrkLock = PTHREAD_MUTEX_INITIALIZER;
#endif

  hoardLock (sbrkLock);
#endif
  void * moreMemory = sbrk (size);
#if defined(__linux) || defined(__sgi)
  hoardUnlock (sbrkLock);
#endif
  return moreMemory;
}


int hoardGetPageSize (void)
{
  return (int) sysconf(_SC_PAGESIZE);
}


#endif
