/*  -*- c -*-  */
/* -------------------------------------------------------------------- *
**  copyright (c) 1995 ipvr stuttgart and thomas harrer
** -------------------------------------------------------------------- *
**
**  libhelp
**
**  a comprehensive hypertext help system for OSF/Motif(tm) applications. 
**  based on libhtmlw from NCSA Mosaic(tm) version 2.4
**
**  written by thomas harrer
**  e-mail: Thomas.Harrer@rus.uni-stuttgart.de
**  
** -------------------------------------------------------------------- *
*h  $Id: path.c,v 1.8 1995/06/28 12:59:30 thomas Exp $
** -------------------------------------------------------------------- *
**
*h  module:		path.c
**
**  contents:		the help path implementation
**
**  interface:		procedures 
**			* get_help_path
**
** -------------------------------------------------------------------- *
**  license and copying issues:
**
**  this software is free; you can redistribute it and/or modify it 
**  under terms similar to the gnu general public license (version 1 
**  or any later version published by the free software foundation). 
**  see the file Licence for more details.
**
**  this program is distributed in the hope that it will be useful,
**  but without any warranty; without even the implied warranty of
**  merchantability or fitness for a particular purpose.  
** -------------------------------------------------------------------- */
/*----------------------------------------------------------------------*
*g  include section
**----------------------------------------------------------------------*/
/* application definitions  */
#include "helpp.h"
#include "path.h"
#include "util.h"
#include "buffer.h"

/* -------------------------------------------------------------------- *
*g  module identification
** -------------------------------------------------------------------- */
#ifdef RCSID
static char rcsid [] =
    "$Id: path.c,v 1.8 1995/06/28 12:59:30 thomas Exp $";
#endif /* RCSID */

/* -------------------------------------------------------------------- *
*p  private prototypes
** -------------------------------------------------------------------- */
static void initialize_help_path (void);

/* -------------------------------------------------------------------- *
*g  module data
** -------------------------------------------------------------------- */
/* the help path  */
static char**	help_path = NULL;      /* the help path (string array).	*/
static int	help_max = 0;	       /* max (strlen (help_path [i])).	*/

/* -------------------------------------------------------------------- *
*p  procedure-name:	path_get_helppath
**
**  purpose:		try to add path prefices in front of the help
**			file string. take the first that results in an
**			accessible help file.
**  initialization:	the help path (array of directory strings) is 
**			initialized of not already.
** -------------------------------------------------------------------- *
**  args:		filename and opt. anchor (file#anchor)
**  return type:	static char*
** -------------------------------------------------------------------- */
char*
path_get_helppath (/* i  */ char* str)
{
    if (str) {

	/* local data.  */
	char* 	pathname;	/* return pointer		*/
	char* 	filename;	/* filename part of str.		*/
	int 	i = 0;		/* loop counter (see below).	*/

	execute ("get_help_path");

	/*
	 *  the help path is stored in an array of strings for ease of 
	 *  access. it must be initialized from the LIBHELPPATH 
	 *  environment variable and from the HELP_PATH praeprocessor 
	 *  macro. it it is not already initialized, we do it first.
	 */
	if (!help_path)  {    
	    initialize_help_path ();
	}	

	/* we strip of the anchor;  */
	filename = get_file (str); /* scope -> returned  or local. */
	if (!filename) return NULL;

	/*
	 *  if the file name is absolute: we just return it 
	 *  (without anchor). there is compressed file transparency 
	 *  for absolute filenames too.
	 */
	if ((filename [0] == '/') || (filename [0] == '.')) {

	    if (0 == access (filename, R_OK)) {
		return filename;
	    } else {
		int len = c_strlen (filename) + 4;
		checked_realloc (filename, len, char);
		c_strcat (filename, ".gz");
		if (0 == access (filename, R_OK)) {
		    return filename;
		} else {
		    checked_free (filename);
		    return NULL;
		}
	    }
	}

	/* we need one buffer for new filenames. ret points to this buffer.  */
	/* begin  */  {
	    int len = c_strlen (filename) + 4;
	    if (help_path[0]) {
		int	hp0_len = c_strlen (help_path[0]);
		if (help_max < hp0_len)  {
		    help_max = hp0_len;
		}
	    }
	    len += help_max;
	    checked_malloc (pathname, len, char); /* scope returned.  */
	}
    
	if (help_path[0]) i = 0; else i = 1;

	/*
	 *  this is the actual help path search using access().
	 *  at this level gzipped files transparency comes into play.
	 */
	while (help_path[i]) {

	    c_strcpy (pathname, help_path[i]);
	    c_strcat (pathname, filename);

	    if (access (pathname, R_OK) == 0)
		break;

	    /* only support for gzip-compressed files.  */
	    c_strcat (pathname, ".gz");
	    if (access (pathname, R_OK) == 0)
		break;

	    i++;
	}
	
	/* if filename is not returned, we free allocation.  */
	checked_free (filename);

	if (help_path[i]) {
	    return pathname;
	} else {
	    /* not found in the help path.  */
	    checked_free (pathname);
	    return NULL;
	}
    } else  {
	return NULL;
    }
    
}

/* -------------------------------------------------------------------- *
*p  procedure-name:	path_add_current
**
**  purpose:		adds the current document path to the 
**			help path.
** -------------------------------------------------------------------- */
void
path_add_current (/* i  */ char* path)
{
    static buffer_t* current = NULL;   /* the current pathname.         */
    execute ("path_add_current");
    
    /* initialize current path buffer */
    if (!current) {
	current = bf_new ();	/* allocated only once. never freeed. */
    }

    if (path) {
	bf_strcpy (current, path);
	help_path[0] = bf_the_buffer (current);
    }
}

/* -------------------------------------------------------------------- *
*p  procedure-name:	path_the_helppath
**
**  purpose:		exports the help path
** -------------------------------------------------------------------- */
char**
path_the_helppath (void)
{
    execute ("the_help_path");
    return help_path;
}

/* -------------------------------------------------------------------- *
*p  procedure-name:	initialize_help_path
**
**  purpose:		initializes the help path array. this array
**			contains the entries of the help path as strings.
** -------------------------------------------------------------------- *
**  side effects:	help_path is initialized and help_max is set
**			to the maximum string length of entries.
** -------------------------------------------------------------------- */
static void
initialize_help_path (void)
{
    int 	i = 0;		       /* loop counter			*/
    int 	count = 0;	       /* number of path entries.	*/
    char* 	path;		       /* path from environment		*/
    char* 	prefix = NULL;	       /* copy of the whole path.	*/

    char* 	token;		       /* for token scanning (strtok).  */
    char 	delim[] = ":";	       /* delimiters dto.		*/

    execute ("initialize_help_path");

    /* if it seems that the help_path is already set up we do nothing.  */
    if (help_path) return;

    path = getenv ("LIBHELPPATH");

    /* first we make a copy of the whole path list by concatenating.  */
    if (!path) {
	checked_strdup (prefix, HELP_PATH); /* scope local */
    } else {
	int size = c_strlen (HELP_PATH) + c_strlen (path) + 2;
	checked_malloc (prefix, size, char); /* scope local */
	c_strcpy (prefix, path);
	c_strcat (prefix, ":");
	c_strcat (prefix, HELP_PATH);
    }
	
    /* we compute the number of path entries.  */
    for (i = 0; *(prefix + i) != '\0'; i++) {
	if (*(prefix + i) == ':')
	    count++;
    }

    /* and allocate as much slots for pathname strings.  */
    checked_malloc (help_path, count + 3, char*); /* must not be freed */

    /* we begin with the first path component.  */
    token = strtok (prefix, delim);

    help_path[0] = NULL;
    i = 1;

    /* and seperate the whole path list.  */
    while (token) {
	int len = c_strlen (token) + 2;
	int end;
	char* entry;
	    
	checked_malloc (entry, len, char); /* scope: global or local */

	/* we save the maximum string lenght for later use.  */
	if (help_max < len) help_max = len;
	    
	/* after some modifications we store the pathname in help_path.  */
	c_strcpy (entry, token);
	end = c_strlen (entry);
	if ((end > 0) && (entry [end - 1] != '/')) {
	    entry [end] = '/';
	    entry [end + 1] = '\0';
	}

	/* entry is either stored in the help path or freed.  */
	if (end != 0) {
	    help_path[i] = entry;
	    i++;
	} else  {
	    checked_free (entry);
	}

	/* now we continue with the next entry in the path.  */
	token = strtok (NULL, delim);
    }
    /* we mark the end of the help path with the NULL string.  */
    help_path[i] = NULL;
    if (prefix)  {
	checked_free (prefix);
    }
}

/* -------------------------------------------------------------------- *
*l  emacs:
**  local variables:
**  mode:		c
**  outline-regexp:	"\*[HGPLT]"
**  gh-language:	"English"
**  comment-column:	32
**  eval:		(outline-minor-mode t)
**  end:
** -------------------------------------------------------------------- */
