/*  -*- c -*-  */
/* -------------------------------------------------------------------- *
**  copyright (c) 1995 thomas harrer
** -------------------------------------------------------------------- *
**
**  libhelp
**
**  a comprehensive hypertext help system for osf/motif(tm) applications. 
**  based on libhtmlw from ncsa mosaic version 2.4
**
**  written by thomas harrer
**  e-mail: Thomas.Harrer@rus.uni-stuttgart.de
**  
** -------------------------------------------------------------------- *
*h  $Id: client.c,v 1.8 1995/06/17 12:02:29 thomas Exp $
** -------------------------------------------------------------------- *
**
*h  module:		server.c
**
**  contents:		libhelp server.
**
** -------------------------------------------------------------------- *
**  license and copying issues:
**
**  this software is free; you can redistribute it and/or modify it 
**  under terms similar to the gnu general public license (version 1 
**  or any later version published by the free software foundation). 
**  see the file Licence for more details.
**
**  this program is distributed in the hope that it will be useful,
**  but without any warranty; without even the implied warranty of
**  merchantability or fitness for a particular purpose.  
** -------------------------------------------------------------------- */
#ifdef RCSID
static char rcsid [] =
    "$Id: client.c,v 1.8 1995/06/17 12:02:29 thomas Exp $";
#endif /* RCSID */

/* -------------------------------------------------------------------- *
*g  include files
** -------------------------------------------------------------------- */
#include <X11/Intrinsic.h>
#include <signal.h>
#include <unistd.h>
#include <sys/wait.h>
#include "helpp.h"
#include "path.h"

/* -------------------------------------------------------------------- *
*g  libhelp resources
**  libhelp resources can be set by the application (the user of the
**  help service) via the function help_set_resource.
**
**  list of resources. add new resources here.
** -------------------------------------------------------------------- */
static resource_t help_resources[] =  {
    /* meeningless in libhlpclient  */
    {help_class_name, help_string,  (XtPointer)"Libhelp"},
    {help_standalone, help_boolean, (XtPointer) 0}, 
    {help_index,      help_string,  (XtPointer) NULL},
    {help_update,     help_string,  (XtPointer) 0},
    /* used by libhlpclient  */
    {help_server, help_int, XMHELP_SERVER},
    {help_end_of_resources, help_boolean, NULL},	
};

/* -------------------------------------------------------------------- *
*g  global variables
** -------------------------------------------------------------------- */
static pid_t	server_pid = 0;
static char* 	name = NULL;

/* -------------------------------------------------------------------- *
*g  prototypes
** -------------------------------------------------------------------- */
static int 	start_server (char*);
static void 	server_get_help (char* url);
static void 	compose_name (pid_t pid);
static void	signal_url (char*);

/* -------------------------------------------------------------------- *
*p  procedure-name:	get_help
**
**  purpose:		wrapper for libhelp's get_help command.
**			starts a help server (xmhelp or Mosaic).
** -------------------------------------------------------------------- */
void
get_help (/* i  */ Widget 	w,
	  /* i  */ XtPointer	client_data,
	  /* i  */ XtPointer	call_data)
{
    if (client_data) {
	char* doc = (char*) client_data;
	char* url = NULL;
    
	if (help_resources[help_server].value == MOSAIC_SERVER) {
	    url = path_get_helppath (doc);
	}

	if (!url) checked_strdup (url, doc);
	server_get_help (url);
	checked_free (url);
    }
}

/* -------------------------------------------------------------------- *
*p  procedure-name:	help_set_resource
**
**  purpose:		sets a internal resource for the help
**			subsystem. Xt resources cannot be used because
**			we hide the implementation of the help
**			system from the application. the application 
**			has no anchor to identify the help system as
**			a shell or a widget or something else.
** -------------------------------------------------------------------- *
**  args:		resource symbol (declared in help.h) and
**			a value (of type XtPointer)
**  precondition:	- 
**  postcondition:	resource value is set if symbol was valid
**			else ignored
** -------------------------------------------------------------------- */
void
help_set_resource (/* i  */ int 	symbol,
 		   /* i  */ XtPointer	value)
{
    int i = 0;
    execute ("help_set_resource");
    while (help_resources[i].resource != help_end_of_resources) {
	if (help_resources[i].resource == symbol) {
	    help_resources[i].value = value;
	}
	i++;
    }
}

/* -------------------------------------------------------------------- *
*p  procedure-name:	server_get_help
**
**  purpose:		handles communication with the server
** -------------------------------------------------------------------- */
static void
server_get_help (char* url)
{
    if (server_pid == 0) {
	start_server (url);
    } else  {
	/* first look if server is still running.  */
	int status;
    	pid_t pid = waitpid (server_pid, &status, WNOHANG | WUNTRACED);
	if (pid == 0) {
	    signal_url (url);
	} else  {
	    start_server (url);
	}
    }
}

/* -------------------------------------------------------------------- *
*p  procedure-name:	start_server
**
**  purpose:		starts the server if it is deleted by the user
** -------------------------------------------------------------------- */
static int
start_server (char* url)
{
    if ((url) && (*url)) {
	pid_t 	pid;
	char* 	server;
	char* 	doc;

	if (help_resources[help_server].value == MOSAIC_SERVER) {
	    server = "Mosaic";
	    doc = path_get_helppath (url); /* scope local. */
	    if (!doc) {
		checked_strdup (doc, url);
	    }
	} else  {
	    /* xmhelp is always the default.  */
	    server = "xmhelp";
	    checked_strdup (doc, url); /* scope local */
	}

	switch (pid = fork()) {
	  case -1:
	    fprintf (stderr, "fork failed, cannot start server\n");
	    return;
	    break;

	  case 0:		/* cild process (xmhelp or Mosaic). */
	    if (-1 == execlp (server, server, doc, NULL)) {
		fprintf (stderr, "exec %s failed\n", server);
		exit (1);
	    }
	    break;

	  default:
	    server_pid = pid;
	    compose_name (pid);
	    checked_free (doc);
	    break; 
	}
	trace (("fork_help called with %s\n", doc));
	
    }
    return 0;
}

/* -------------------------------------------------------------------- *
*p  procedure-name:	compose_name
**
**  purpose:		composes the filename for communication
** -------------------------------------------------------------------- */
static void
compose_name (pid_t pid)
{
    static char* home = NULL;
    
    if (!home) {
	checked_strdup (home, getenv ("HOME"));
	if (help_resources[help_server].value == XMHELP_SERVER) {
	    if (!home) {	
		fprintf (stderr, "libhlpclient: can't get HOME environment, "
		     "but need it\n");
		exit (EXIT_FAILURE);
	    }
	}
    }
	
    if (!name) {
	size_t size = strlen (home) + 32;
	checked_malloc (name, size, char);
    }
    
    if (help_resources[help_server].value == MOSAIC_SERVER) {
	sprintf (name, "/tmp/Mosaic.%d", pid);
    } else  {
	sprintf (name, "%s/.xmhelp.%d", home, pid);
    }
}

/* -------------------------------------------------------------------- *
*p  procedure-name:	signal_url
**
**  purpose:		writes the given url to the communication file
** -------------------------------------------------------------------- */
static void
signal_url (/* i  */ char* url)
{
    if (name) {
	FILE* write = fopen (name, "w");
	if (write) {
	    if (help_resources[help_server].value == MOSAIC_SERVER) {
		fprintf (write, "goto\nfile://localhost%s\n", url);
	    } else  {
		fprintf (write, "%s", url);
	    }
	    kill (server_pid, SIGUSR1);
	    fclose (write);
	}
    } else  {
	fprintf (stderr, "name is not valid\n");
	exit (EXIT_FAILURE);
    }
}

/* -------------------------------------------------------------------- *
*l  emacs:
**  local variables:
**  mode:		c
**  outline-regexp:	"\*[HGPLT]"
**  comment-column:	32
**  eval:		(outline-minor-mode t)
**  end:
** -------------------------------------------------------------------- */
