/*  -*- c -*-  */
/* -------------------------------------------------------------------- *
**  copyright (c) 1995 ipvr stuttgart and thomas harrer
** -------------------------------------------------------------------- *
**
**  libhelp
**
**  a comprehensive hypertext help system for OSF/Motif(tm) applications. 
**  based on libhtmlw from NCSA Mosaic(tm) version 2.4
**
**  written by thomas harrer
**  e-mail: Thomas.Harrer@rus.uni-stuttgart.de
**  
** -------------------------------------------------------------------- *
*h  $Id: help.c,v 1.91 1995/06/28 12:59:30 thomas Exp $
** -------------------------------------------------------------------- *
**
*h  module:		help.c
**
**  contents:		help subsystem entry.
**
**  interface:		functions
** 
**	* get_help (Widget parent, char* anchor, unused)
**
**	* help_set_resource (int symbol, XtPointer value)
**
** -------------------------------------------------------------------- *
**  license and copying issues:
**
**  this software is free; you can redistribute it and/or modify it 
**  under terms similar to the gnu general public license (version 1 
**  or any later version published by the free software foundation). 
**  see the file Licence for more details.
**
**  this program is distributed in the hope that it will be useful,
**  but without any warranty; without even the implied warranty of
**  merchantability or fitness for a particular purpose.  
** -------------------------------------------------------------------- */
/*----------------------------------------------------------------------*
*g  include section
**----------------------------------------------------------------------*/
/* X11 and Motif includes.  */
#include <X11/Intrinsic.h>
#include <X11/cursorfont.h>
#include <Xm/Xm.h>
#include <Xm/Form.h>
#include <Xm/RowColumn.h>
#include <Xm/Label.h>
#include <Xm/PushB.h>
#include <Xm/Protocols.h>
#include <Xm/PanedW.h>
#include <xpm.h>

#ifdef PRIVATE_DEBUG
#include <X11/IntrinsicP.h>
#include <X11/ShellP.h>
#endif /* PRIVATE_DEBUG  */

/* application definitions  */
#include "HTML.h"

/* help widget definitions  */
#include "help.h"
#include "helpp.h"
#include "actionarea.h"
#include "contexthelp.h"
#include "language.h"
#include "image.h"
#include "load.h"
#include "buffer.h"
#include "util.h"
#include "path.h"
#include "bcache.h"

/* -------------------------------------------------------------------- *
*g  module and library identification
** -------------------------------------------------------------------- */
#ifdef RCSID
/* module  */
static char rcsid [] =
    "$Id: help.c,v 1.91 1995/06/28 12:59:30 thomas Exp $";
/* library (values will be read from version.h) */
static char system_name [] = "$Release name: " _system_name_ "$";
static char system_version [] = "$Release version: " _system_version_ "$";
static char system_date [] = "$Release date: " _system_date_ "$";
#endif /* RCSID */

/* -------------------------------------------------------------------- *
*g  private type:	help_structure_t
** -------------------------------------------------------------------- *
**  help widget structure:
**  
**  toplevel - help_shell - pane
**                      _____|______
**                     |            |
**                   help       actionarea (rowcol)
**                     |            |
**               (help_view)    (subwidgets from module actionarea)
**   (subwidget html widget)
**
** -------------------------------------------------------------------- */
typedef struct help_structure_s  {

    Widget 	help_shell;
    Widget 	pane;
    Widget 	help;
    Widget 	action_area;
    Widget	help_view;
    int 	is_up;		/* create if 0, map if 1. */

} help_structure_t;

/* -------------------------------------------------------------------- *
*g  private type:	history_t
** 			history for file-anchor pairs (linked list). 
** -------------------------------------------------------------------- */
typedef struct history_s  {

    char* 	name;		/* filename and anchor.			*/
    char* 	refname;	/* currently the document title.	*/
    int 	element_id;	/* current position in html widget.	*/
    struct 	history_s* next;

} history_t;

/* -------------------------------------------------------------------- *
*t  private type:	icon_t
**			type for icon pixmaps.
** -------------------------------------------------------------------- */
typedef struct icon_s {

    Pixmap	icon_pixmap;	/* BadPixmap if there is no pixmap.	*/
    Pixmap	icon_mask;	/* BadPixmap if there is no mask.	*/

} icon_t;

/* -------------------------------------------------------------------- *
*g  libhelp resources
**  libhelp resources can be set by the application (the user of the
**  help service) via the function help_set_resource.
**
**  list of resources. add new resources here.
** -------------------------------------------------------------------- */
static resource_t help_resources[] =  {
    {help_class_name, help_string,  (XtPointer)"Libhelp"},
    {help_standalone, help_boolean, (XtPointer) 0}, 
    {help_index,      help_string,  (XtPointer) NULL},
    {help_update,     help_string,  (XtPointer) 0},
    /* meeningless in libhelp. provided for libhlpclient.  */
    {help_server,     help_int,     NO_SERVER},	
    {help_end_of_resources, help_boolean, NULL},	
};

/* -------------------------------------------------------------------- *
*g  local prototypes
** -------------------------------------------------------------------- */
/* functions for setting help texts.  */
static void 	set_help_text (Widget, char*, int);
static void	set_file_and_anchor (Widget, char*, int);
static void	set_text_check_scroll (Widget, char*, char*, char*,
                                       int, char*, void*);
/* history handling.  */
static void 	add_history (Widget, char*, char*);
static void 	history_add_current_id (Widget);

/* image wrapper.  */
static ImageInfo* resolve_img_wrapper (Widget, char*, int);

/* callbacks  */
static void 	popdown_help (Widget, XtPointer, XtPointer);
static void 	anchor_cb (Widget, XtPointer, XtPointer);
static void 	push_cb (Widget, XtPointer, XtPointer);
static void 	form_cb (Widget, XtPointer, XtPointer);

/* pushbutton actions  */
static void 	go_last (void);
static void 	move_vertical (Widget, XEvent*, const char*);
static void 	compose_history (void);
static void 	compose_index (void);

/* icon  */
static void	add_icon (Widget);

/* state  */
static void 	set_state (int);

/* update  */
static void	help_flush (void);

/* -------------------------------------------------------------------- *
*g  action (XtActionProc's)
** -------------------------------------------------------------------- */
/* action prototypes  */
static void help_scroll_up (Widget, XEvent*, String*, Cardinal*);
static void help_scroll_down (Widget, XEvent*, String*, Cardinal*);
static void help_scroll_home (Widget, XEvent*, String*, Cardinal*);
static void help_action_index (Widget, XEvent*, String*, Cardinal*);
static void help_action_history (Widget, XEvent*, String*, Cardinal*);
static void help_action_back (Widget, XEvent*, String*, Cardinal*);
static void help_action_dismiss (Widget, XEvent*, String*, Cardinal*);
static void help_cancel (Widget, XEvent*, String*, Cardinal*);
static void help_about (Widget, XEvent*, String*, Cardinal*);
static void help_reload (Widget, XEvent*, String*, Cardinal*);

/* action record  */
static XtActionsRec help_actions [] = {
{ "help_scroll_down", 	 (XtActionProc) help_scroll_down },
{ "help_scroll_up", 	 (XtActionProc) help_scroll_up   },
{ "help_scroll_home", 	 (XtActionProc) help_scroll_home },
{ "help_action_index",	 (XtActionProc) help_action_index },
{ "help_action_history", (XtActionProc) help_action_history },
{ "help_action_back",	 (XtActionProc) help_action_back },
{ "help_action_dismiss", (XtActionProc) help_action_dismiss },
{ "help_cancel",         (XtActionProc) help_cancel },
{ "help_about",          (XtActionProc) help_about },
{ "help_reload",         (XtActionProc) help_reload },
};

/* -------------------------------------------------------------------- *
*g  translations (for the help view).
** -------------------------------------------------------------------- */
static char help_translations[] = 
"a <Key>v:		help_scroll_up()\n"
"m <Key>v:		help_scroll_up()\n"
"<Key>osfUp:		help_scroll_up()\n"
"<Key>osfPageUp:	help_scroll_up()\n"

"c <Key>v:		help_scroll_down()\n"
"<Key>space:		help_scroll_down()\n"
"<Key>osfDown:		help_scroll_down()\n" 
"<Key>osfPageDown:	help_scroll_down()\n"
"<Key>osfBeginLine:	help_scroll_home()\n"
"<Key>osfBeginData:	help_scroll_home()\n"

"<Key>h:		help_action_history()\n"
"<Key>i:		help_action_index()\n"
"<Key>b:		help_action_back()\n"
"<Key>q:		help_action_dismiss()\n"
"<Key>osfCancel:	help_cancel()\n"

/* undocumented features.  */
"c a <Key>?:		help_about()\n"
"c m <Key>?:		help_about()\n"
"c a <Key>r:		help_reload()\n"
"c m <Key>r:		help_reload()\n"
"c <Key>l:		help_reload()\n"
;

/* -------------------------------------------------------------------- *
*g  global data 
** -------------------------------------------------------------------- */
/* Widget structure for help. */
static help_structure_t* 	hs = NULL;  

/* counter to distinguish pages.*/
static unsigned long 		this_html_no = 0;

/* the history list anchor */
static history_t* 		history = NULL;     

/*
 *  history flags: %%% ugly, we should write a history module %%% 
 *  0 means document has history, 1 means no history.  
 */
/* currrent_type: this information for the currently displayed document  */
static int			current_type = 0;
/* this information for the new document that is to be displayd.  */
static int			next_type = 0;

/* libhelp browser has it's own icon.  */
#include <libhelp_icon.xpm>

/* -------------------------------------------------------------------- *
*g ------------- public procs -----------------------------------------
** -------------------------------------------------------------------- */
/* -------------------------------------------------------------------- *
*p  procedure-name:	help_set_resource
**
**  purpose:		sets a internal resource for the help
**			subsystem. Xt resources cannot be used because
**			we hide the implementation of the help
**			system from the application. the application 
**			has no anchor to identify the help system as
**			a shell or a widget or something else.
** -------------------------------------------------------------------- *
**  args:		resource symbol (declared in help.h) and
**			a value (of type XtPointer)
**  precondition:	- 
**  postcondition:	resource value is set if symbol was valid
**			else ignored
** -------------------------------------------------------------------- */
void
help_set_resource (/* i  */ int 	symbol,
 		   /* i  */ XtPointer	value)
{
    int i = 0;
    
    execute ("help_set_resource");
    
    while (help_resources[i].resource != help_end_of_resources) {
	if (help_resources[i].resource == symbol) {
	    help_resources[i].value = (XtPointer) value;
	}
	i++;
    }
}

/* -------------------------------------------------------------------- *
*p  procedure-name:	get_help
**
**  purpose:		provides entry to the help facility.
**			
**			the help system can run as a standalone or as 
**			a child of an application. In the latter case 
**			it creates it's own toplevel shell (and only 
**			pops it down if dismiss is pressed) else it
**			creates it's widget structure in the parents
**			shell.
**			
**			the creation of shells and the internal states
**			of the help system are hided from the application.
**			
** -------------------------------------------------------------------- *
**  args:		XtCallbackProc args:
**			Widget parent (toplevel).
**			XtPointer call_data: (char*)help file and anchor.
**			XtPointer client_data: unused.
**  precondition:	toolkit must be initialized. parent must be created.
**  postcondition:	help window is created or raised.
** -------------------------------------------------------------------- */
void
get_help (/* i  */ Widget	parent,
	  /* i  */ XtPointer	call_data,
	  /* -  */ XtPointer	client_data)
{
    Atom 	del_atom;	/* Atom to catch delete */
    char* 	class_name;	/* application class. */
    int 	is_shell = 1;
    
    /* buttons in the action area.  */
    static action_t actions[] = {
        { action_dismiss, (XtCallbackProc) push_cb },
        { action_index,   (XtCallbackProc) push_cb },
        { action_history, (XtCallbackProc) push_cb },
        { action_back, 	  (XtCallbackProc) push_cb },
        { action_up, 	  (XtCallbackProc) push_cb },
        { action_down, 	  (XtCallbackProc) push_cb },
    };

    execute ("get_help");

    /*
     *  the global variable hs is a pointer to the widget structure.
     *  it is initialized to NULL at module initialization.
     *  if it is NULL we need to create it and the whole help window.
     *  if not, we just need to raise the window and display the
     *  requested help text.
     */
    if (hs) {

	if (help_resources[help_update].value) {
	    help_flush ();
	}

	/* if it is already up: display the desired help text.  */
	set_help_text (hs->help, (char*) call_data, 0);

	XMapRaised (XtDisplay (hs->help_shell), XtWindow (hs->help_shell));
	if (hs->is_up == 0) {
	    XtPopup (hs->help_shell, XtGrabNone);
	    hs->is_up = 1;
	}
	return;
    }

    /*
     *  if the client initializes libhelp with NULL Widget we complain.
     */
    if (!parent) {
	fprintf (stderr, 
		 "get_help needs a valid parent Widget "
		 "(at the 1st call)\n");
	exit (EXIT_FAILURE);
    }
    
    /* initialize the help system  */
    checked_malloc (hs, 1, help_structure_t); /* never freed ! */
    class_name = (char*) help_resources [help_class_name].value;

    /* create a shell if we not use the existing (in a standalone help).  */
    if ((int) help_resources [help_standalone].value == 1) {
	
	hs->help_shell = parent; /* toplevel */
	is_shell = 0;
	
    } else {

	hs->help_shell = XtVaAppCreateShell 
	    ("help", class_name, 
	     topLevelShellWidgetClass, XtDisplay (parent), 
	     XmNdeleteResponse, 	XmDO_NOTHING,
	     XmNiconic, 		(XtArgVal) False,
	     NULL);
    }

    add_icon (hs->help_shell);
    editres_support (hs->help_shell);

    hs->pane = XtVaCreateWidget 
	("pane", xmPanedWindowWidgetClass, hs->help_shell,
	 XmNsashWidth,  1,
	 XmNsashHeight, 1,
	 NULL);  
    context_help (hs->pane, cx_browser);
    
    /* create the HTML widget.  */
    hs->help = XtVaCreateManagedWidget 
	("help_text", htmlWidgetClass, hs->pane,
	 WbNresolveImageFunction, 	(long) resolve_img_wrapper,
	 XmNresizePolicy,    		XmRESIZE_ANY,
	 XmNresizable,       		True,
	 XmNshadowThickness, 		2,
	 XmNtopOffset, 	 		5,
	 XmNbottomOffset, 		5, 
	 XmNleftOffset, 		5, 
	 XmNrightOffset, 		5,
	 NULL);
    context_help (hs->help, cx_browser);
    
    /*
     *  add translations to the help view. we want keyboard actions 
     *  for scrolling up and down.
     */

    /* actions and translations are set in their own block */ {
	XtAppContext app_context;
	
	app_context = XtWidgetToApplicationContext (hs->help_shell);
	
	XtAppAddActions (app_context, help_actions, 
			 XtNumber (help_actions));

	XtVaGetValues (hs->help, 
		       WbNview,	(long)(&hs->help_view),
		       NULL);

	context_help (hs->help_view, cx_browser);
	
	XtOverrideTranslations (hs->help_view, 
				XtParseTranslationTable (help_translations));
    }
    /* begin  */ {
	Widget vsb; 
	Widget hsb;
	
	XtVaGetValues (hs->help, 
		       XmNverticalScrollBar, (XtPointer) (&vsb), 
		       XmNhorizontalScrollBar, (XtPointer) (&hsb), 
		       NULL);

	if (vsb) context_help (vsb, cx_browser);
	if (hsb) context_help (hsb, cx_browser);
    }

    /*
     *  create the action buttons. set the sensitive states for the
     *  buttons with respect to the available data and the state at
     *  startup (back is insensitive, index too if there is no index
     *  specified).
     */
    /* begin   */ {
	int pane_size = 54;
	Widget form = XtVaCreateManagedWidget
	    ("form", xmFormWidgetClass, hs->pane, 
	     XmNpaneMaximum, pane_size,
	     XmNpaneMinimum, pane_size, NULL);
	hs->action_area = 
	    create_actionarea (form, actions, XtNumber (actions));
    }
    
    if ((char*) help_resources[help_index].value == NULL) {
	actions_set_sensitive (action_index, False);
    }

    if (!history) {
	actions_set_sensitive (action_back, False);
    } else if (!history->next) {
	actions_set_sensitive (action_back, False);
    }

    actions_set_sensitive (action_history, False);
    actions_set_sensitive (action_up, False);
    actions_set_sensitive (action_down, False);

    /* Register HTML Anchor callback.  */
    XtAddCallback (hs->help, WbNanchorCallback, 
		   (XtCallbackProc) anchor_cb, (XtPointer) 0);
    /* Register  the forms callback.   */
    XtAddCallback (hs->help, WbNsubmitFormCallback,
		   (XtCallbackProc) form_cb, (XtPointer) 0);

    set_help_text (hs->help, (char*) call_data, 0);

    XtManageChild (hs->pane);
    /* prevent it from deleting.  */

    if (is_shell == 1) {
	XtPopup (hs->help_shell, XtGrabNone);
	hs->is_up = 1;

	del_atom = XmInternAtom (XtDisplay(parent), "WM_DELETE_WINDOW", False);
	XmAddWMProtocolCallback (hs->help_shell, del_atom,
				 (XtCallbackProc) popdown_help, 
				 (XtPointer)hs->help_shell);
    }
    
    return;
}

/* -------------------------------------------------------------------- *
*g ------------- icon procs -------------------------------------------
** -------------------------------------------------------------------- */
/* -------------------------------------------------------------------- *
*p  procedure-name:	add_icon
**
**  purpose:		creates the icon pixmap and stores it into the 
**			shell's resource list
** -------------------------------------------------------------------- */
static void
add_icon (/* i  */ Widget shell)
{
    static icon_t* 	icon = NULL;

    execute ("add_icon");
    
    if (!icon) {
	int ret;
	checked_calloc (icon, 1, icon_t); /* never freed !! */
	icon->icon_mask = BadPixmap;

	ret = XpmCreatePixmapFromData 
	    (XtDisplay (shell), 
	     DefaultRootWindow(XtDisplay(shell)),
	     libhelp_icon_xpm, 
	     &icon->icon_pixmap, &icon->icon_mask, NULL);
	if (ret != 0) {
	    icon->icon_pixmap = BadPixmap;
	    icon->icon_mask = BadPixmap;
	}
    }

    if (icon->icon_pixmap != BadPixmap) {

	Display* 	display = XtDisplay (shell);
	Window 		window;
	Window		root;
	unsigned int	width;
	unsigned int	height;

	int		xy_dummy;
	unsigned int	dummy;	/* dummy variable for unused vals. */

	/*
	 *  the iconPixmap resource is guaranteed to work with depth 1
	 *  pixmap. since we work with colored icons, we need the 
	 *  iconWindow resource.
	 */

	XtVaGetValues (shell, XmNiconWindow, (XtArgVal) &window, NULL);

	if (!window) {
	    /*
	     *  if the window is not already installed, we create one of 
	     *  the same geometry as our icon. (the code is from 
	     *  O'reillys book).
	     */
	    if (!XGetGeometry 
		(display, icon->icon_pixmap, &root, &xy_dummy, &xy_dummy,
		 &width, &height, &dummy, &dummy) 
		|| !(window = XCreateSimpleWindow 
		     (display, root, 0, 0, width, height,
		      (unsigned) 0, CopyFromParent, CopyFromParent))) {
		/*
		 *  after creating of the icons window failed, we  
		 *  fall back to the iconPixmap resource and return.
		 */
		XtVaSetValues (shell, XmNiconPixmap, icon->icon_pixmap, NULL);
		return;
	    }   
	    XtVaSetValues (shell, XmNiconWindow, window, NULL);
	} /* !window  */
	XSetWindowBackgroundPixmap (display, window, icon->icon_pixmap);
	XClearWindow (display, window);
    }
}

/* -------------------------------------------------------------------- *
*g ------------- html handling procs ----------------------------------
** -------------------------------------------------------------------- */
/* -------------------------------------------------------------------- *
*p  procedure-name:	set_help_text
**
**  purpose:		sets the help text and adds the history entry.
** -------------------------------------------------------------------- */
static void
set_help_text (/* i  */ Widget 	w,
	       /* i  */ char* 	ref,
	       /* i  */ int 	element_id)
{    
    char*	title_text;	/* for history text line.  */

    execute ("set_help_text");

    /* explicitly state that this page has a history.  */
    next_type = 0;
    
    /* first we display the reference.  */
    set_file_and_anchor (w, ref, element_id);

    /* we need the title of the html document for history lines */ 
    XtVaGetValues (hs->help, WbNtitleText, (long) &title_text, NULL);

    if (!title_text) {
	title_text = str_no_title;
    }

    /* then we add the reference together with the title to the history. */
    /* begin  */ {
	char* current = bcache_current ();
	if (current) {
	    add_history (w, current, title_text);
	    checked_free (current); /* allocated by load.c  */
	}
    }
}

/* -------------------------------------------------------------------- *
*p  procedure-name:	set_file_and_anchor
**
**  purpose:		sets the help text without adding history.
**			either load the html file or uses it if it is 
**			already in the global buffer.
** -------------------------------------------------------------------- */
static void
set_file_and_anchor (/* i  */ Widget 	w,
		     /* i  */ char* 	ref,
		     /* i  */ int 	element_id)
{    
    /* local data.  */
    buffer_t*	buf = NULL;

    execute ("set_file_and_anchor");

    /* loading takes real time. so we change to watch cursor.  */
    set_state (1);

    /* we read in the text (or a warning). buf is valid after this call. */
    buf = read_html_file (ref);

    /* now we display the document.  */
    this_html_no++;
    set_text_check_scroll (w, bf_the_buffer (buf), "\0", "\0", 
			   element_id, get_anchor (ref), NULL);
}

/* -------------------------------------------------------------------- *
*p  procedure-name:	set_text_check_scroll
**
**  purpose:		wrapper for HTMLSetText that
**			checks the scrollbars and set cursor icons.
** -------------------------------------------------------------------- */
static void
set_text_check_scroll (/* i  */ Widget 	w, 
		       /* i  */ char*	text, 
		       /* i  */ char*	header_text,
		       /* i  */ char*	footer_text, 
		       /* i  */ int 	element_id,
		       /* i  */ char*	target_anchor, 
		       /* i  */ void*	ptr)
{
    execute ("set_text_check_scroll");

    /*
     *  while work is in progress we change our cursor-icon to a 
     *  watch on the help shells window.
     */
    set_state (1);
    
    /*
     *  we need to remember the current position in the document for 
     *  restoring it on history requests. %%% ugly %%%
     */
    if (current_type == 0) history_add_current_id (w);
    current_type = next_type;
    
    /* now we call the html widget interface function.  */
    HTMLSetText (w, text, header_text, footer_text, element_id, 
		 target_anchor, ptr);


    /* begin  */ {

	Widget scroll_bar;
	/*
	 *  the initial state for up and down buttons is insensitive.
	 *  so as long as there is no scrollbar, the buttons remain in
	 *  this state. if there are scrollbars, we change the state
	 *  to sensitive, if the vertical scrollbar is managed.
	 */

	XtVaGetValues (hs->help, XmNverticalScrollBar, 
		       (XtPointer) (&scroll_bar), NULL);

	if (scroll_bar) {
	    if (XtIsManaged (scroll_bar) == True) {
		actions_set_sensitive (action_up,   True);
		actions_set_sensitive (action_down, True);
	    } else {
		actions_set_sensitive (action_up,   False);
		actions_set_sensitive (action_down, False);
	    }
	}
    }
    set_state (0);
}

/* -------------------------------------------------------------------- *
*g ------------- callbacks ----------------------------------------
** -------------------------------------------------------------------- */
/* -------------------------------------------------------------------- *
*p  procedure-name:	popdown_help
**
**  purpose:		pops down help window
** -------------------------------------------------------------------- */
static void
popdown_help (/* i  */ Widget w,
	      /* -  */ XtPointer call_data,     /* ignored  */
	      /* -  */ XtPointer client_data)   /* ignored  */
{
    execute ("popdown_help");

    /* we leave the application here if we run standalone.  */
    if ((int)help_resources[help_standalone].value == 1) {
	exit (0);
    }

    /* else we just popdown help's toplevel shell.  */
    if (hs) {
	XtPopdown (hs->help_shell);
	hs->is_up = 0; 
    }
    return;
}

/* -------------------------------------------------------------------- *
*p  procedure-name:	push_cb
**
**  purpose:		called when button is pressed.
**			depends on the actionarea module.
**			we call the action for the specified button type.
** -------------------------------------------------------------------- */
static void
push_cb (/* i  */ Widget w,
	 /* i  */ XtPointer client_data, /* int: number  */
	 /* i  */ XtPointer call_data)   /* needed for event  */
{
    int action = (int) client_data;
    XmPushButtonCallbackStruct* cbs = 
	(XmPushButtonCallbackStruct*) call_data;

    execute ("push_cb");
    
    set_state (1);

    switch (action) {
      case action_history:
	compose_history ();
	break;
      case action_dismiss:
	popdown_help (w, NULL, NULL);
	break;
      case action_back:	
	go_last ();
	break;
      case action_down:
	move_vertical (w, cbs->event, "PageDownOrRight");
	break;
      case action_up:
	move_vertical (w, cbs->event, "PageUpOrLeft");
	break;
      case action_index: 
	compose_index ();
	break;
      default:
	break;
    }
    set_state (0);

    return;
}

/* -------------------------------------------------------------------- *
*p  procedure-name:	anchor_cb
**
**  purpose:		Callback for pressing anchors
** -------------------------------------------------------------------- *
**  args:		Widget w:     HTML widget that calls the callback.
**			client_data:  
**			call_data:    
**  precondition:	callback must be registert to HTML widget.
**  postcondition:	Anchor is followed up and redisplayed.
**  error handling.:	returns.
** -------------------------------------------------------------------- */
static void
anchor_cb (/* i  */ Widget 	w,
	   /* -- */ XtPointer 	client_data,    /* ignored  */
	   /* i  */ XtPointer 	call_data)      /* HTML data  */
{
    char* 		  ref = NULL;
    WbAnchorCallbackData* data = (WbAnchorCallbackData*) call_data;

    execute ("anchor_cb");
    
    if (data->href) {
	checked_strdup (ref, data->href);
    } else {
	checked_strdup (ref, "Unlinked");
    } /* scope ref -> local  */
    
    /*
     *  anchors from the history list have a special format:
     *  they begin with "__hist" followed by the number of the accessed
     *  history entry (reflects the position in the history list).
     */
    if (0 == c_strncmp (ref, "__hist", 6)) {

	history_t* h = history;
	long nr = strtol ((ref + 6), NULL, 0);
	long i = 0;

	hist_trace (("history ref: %s, nr = %d\n", ref, nr));

	while (h) {
	    if (i == nr) break;
	    i++;
	    h = h->next;
	}
	if (!h) {
	    fputs ("inconsistency in anchor_cb\n", stderr);
	    exit (EXIT_FAILURE);
	}
	set_help_text (hs->help, h->name, h->element_id);
	
    } else {
	set_help_text (hs->help, ref, 0);
    }
    
    checked_free (ref);
    return;
}

/* -------------------------------------------------------------------- *
*p  procedure-name:	form_cb
**
**  purpose:		called if form is returned from html.
**			yet there is no real implementation for forms.
** -------------------------------------------------------------------- */
static void
form_cb (/* i  */ Widget w,
	 /* i  */ XtPointer call_data,
	 /* i  */ XtPointer client_data)
{
    WbFormCallbackData *cd = (WbFormCallbackData *) client_data;
    buffer_t*	buf = bf_new (); /* scope -> local */
    
    execute ("form_cb");
    
    bf_strcpy (buf, 
	       "<title>libhelp internal form support</title>\n"
	       "<h1>" str_libhelp_icon " Fill out forms</h1>\n"
	       "Forms are not implemented in <b>libhelp</b>. We provide"
	       " this internal page for testing forms. <p>");

    if (!cd) {
	bf_strcat (buf, "Form with no data\n");
    } else  {
	int c;
	
	bf_strcat (buf, "<pre>\n");
	if (cd->href) bf_sprint (buf, "href: %s\n", cd->href);
	if (cd->method) bf_sprint (buf, "method: %s\n", cd->method);
	if (cd->enctype) bf_sprint (buf, "enctype: %s\n", cd->enctype);
	if (cd->enc_entity) bf_sprint (buf, "enc_entity: %s\n", 
					cd->enc_entity);
	for (c = 0; c < cd->attribute_count; c++) {
	    bf_sprint (buf, "name: %s", cd->attribute_names[c]);
	    bf_sprint (buf, "value: %s\n", cd->attribute_values[c]);
	}
	bf_strcat (buf, "</pre>\n");
    }

    /* we flag, that this page has no history entry.  */
    next_type = 1;

    /* bring the page to display.  */
    set_text_check_scroll (hs->help, bf_the_buffer (buf), 
			   "\0", "\0", 0, NULL, NULL);	
    bf_free (buf);

    /* we must make it able to go back!  */
    actions_set_sensitive (action_back, True);

}

/* -------------------------------------------------------------------- *
*g  -------------image wrapper ----------------------------------------
** -------------------------------------------------------------------- */
/* -------------------------------------------------------------------- *
*p  procedure-name:	resolve_img_wrapper
**
**  purpose:		calls image_resolve with the correct help path
** -------------------------------------------------------------------- */
static ImageInfo*
resolve_img_wrapper (/* i  */ Widget 	w,
		     /* i  */ char*	src,
		     /* i  */ int	no_load)
{
    execute ("resolve_img_wrapper");
    return image_resolve (hs->help, src, this_html_no);
}

/* -------------------------------------------------------------------- *
*g ------------- history handling----------------------------------------
** -------------------------------------------------------------------- */
/* -------------------------------------------------------------------- *
*p  procedure-name:	add_history
**
**  purpose:		adds a file+anchor pair to the history.
** -------------------------------------------------------------------- */
static void
add_history (/* i  */ Widget 	w,
	     /* i  */ char* 	file_and_anchor,
	     /* i  */ char*	title)
{
    /* to save the name of the current file:  */
    static char* current_file = NULL;
    
    char* 	filename = get_file (file_and_anchor); /* scope: local! */
    char* 	anchor = get_anchor (file_and_anchor); /* just ptr. */
    char* 	title_ref = NULL;
    char* 	address = NULL;
    history_t* 	tmp_hist = NULL;
    int   	len;

    execute ("add_history");

    if (*filename == '\0') {
	checked_free (filename);
	filename = current_file;
    } else  {
	if (current_file) checked_free (current_file);
	current_file = filename;
    } /* filename is now treated as a pointer to current_file. */

    if (anchor && (*anchor != '\0')) {
	/* address is used to compose <file>#<anchor>  */
	len = c_strlen (filename) + c_strlen (anchor) + 2;
	checked_malloc (address, len, char);
	c_strcpy (address, filename);
	c_strcat (address, "#");
	c_strcat (address, anchor);
    } else {
	checked_strdup (address, filename); 
    } /* address scope -> go_last  */

    checked_malloc (tmp_hist, 1, history_t); /* scope -> go_back ! */

    if (title) {
	checked_strdup (title_ref, title);
    } else {
	checked_strdup (title_ref, str_no_title);
    } /* title_ref scope -> go_last.  */
    
    tmp_hist->name = address;
    tmp_hist->refname = title_ref;
    tmp_hist->element_id = 0;
    tmp_hist->next = history;

    history = tmp_hist;
    
    if (history->next) {
	actions_set_sensitive (action_back, True);
    }
    actions_set_sensitive (action_history, True);
}

/* -------------------------------------------------------------------- *
*p  procedure-name:	history_add_current_id
**
**  purpose:		adds the current id to the history
** -------------------------------------------------------------------- */
static void
history_add_current_id (/* i  */ Widget w)
{
    int element_id = HTMLPositionToId (w, 0, 0);
    
    execute ("history_add_current_id");
	
    if (history) {
	history->element_id = element_id;
    }
    return;
}

/* -------------------------------------------------------------------- *
*g ------------- display and motion -------------------------------------
** -------------------------------------------------------------------- */
/* -------------------------------------------------------------------- *
*g  procedure-name:	compose_history
**
**  purpose:		displays the history.
**			we translate the history of visited help anchors
**			into a html document and display it like others.
**			the only difference is, that it does not get a
**			history entry for itself.
** -------------------------------------------------------------------- */
static void 
compose_history (void)
{
    /* initial size for buffer allocation. can grow. */

    history_t* 	h = history;	   /* a pointer to the history.         */
    int 	counter = 0;	   /* unique for each history entry.    */

    execute ("compose_history");
    
   /* the history text has no history entry for itself.  */
    next_type = 1;

    /* if we have no history, we do nothing (rather unlikely).  */
    if (h) {

	static buffer_t* buf = NULL;

	if (!buf) {
	    buf = bf_new ();
	}
	
	/* we first write a header (starting an unnumbered list).  */
	bf_strcpy (buf, 
		   "<title>" str_history "</title>\n"
		   "<h1>" str_history_icon " " str_history 
		   "</h1>\n");
        
	/*
	 *  add a list entry for each history entry. uses the <li> tag.
	 *  the reference (<href=>) is artificial. it contains the string
	 *  __hist<nr>. the anchor_cb must understand this special reference
	 *  as a history reference.
	 */
	while (h) {

	    char* ptr;

	    /* search the anchor  */
	    ptr = h->name;
	    while (*ptr != '\0') {
		if (*ptr == '#') { ptr++; break; }
		ptr++;
	    }
	
	    /* add the anchor as history text.  */
	    bf_strcat (buf, str_icon_index "<a href=\"");
	    bf_dprint (buf, "__hist%d__\">", counter); counter++;
	    
	    if (*ptr == '\0') {
		/* if we have no anchor we use the document title */
		bf_strcat (buf, h->refname);
	    } else {
		char* new_string = NULL;
		char* ptr2;
		
		checked_strdup (new_string, ptr); /* scope -> local  */
	    
		/* replace '_' with ' '. the app. can use '_' or ' '.  */
		ptr2 = new_string;
		while (*ptr2 != '\0') {
		    if (*ptr2 == '_') *ptr2 = ' ';
		    ptr2++;
		}
		bf_strcat (buf, new_string);
		checked_free (new_string);
	    }

	    bf_strcat (buf, "</a><br>");

	    /* we go on with the next entry  */
	    h = h->next;
	}

	/* ok, we add the current version number and date.  */
	bf_strcat (buf, "\n<p>" str_libhelp_bar 
		   "\n<address>libhelp " 
		   _system_version_ " (" _system_date_ ")\n");

	
	/* display the new composed history.  */
	set_text_check_scroll (hs->help, bf_the_buffer (buf), 
			       "\0", "\0", 0, NULL, NULL);

	/* allow the user to get back with buttons.  */
	actions_set_sensitive (action_back, True);
    
	/* we do not free buf cause we use it in all calls.  */
    }
}
	
/* -------------------------------------------------------------------- *
*p  procedure-name:	compose_index
**
**  purpose:		composes the index.
**			sets the sensitivity of back button.
** -------------------------------------------------------------------- */
static void
compose_index (void)
{
    execute ("compose_index");
    
    if (help_resources[help_index].value) {

	char* string = NULL;
	checked_strdup (string, (char*) help_resources[help_index].value);
	/* scope -> local  */

	if (history) {
	    /* we check history because there could be */
	    /* an error (no valid history) in the first request. */
	    actions_set_sensitive (action_back, True);
	}
	
	next_type = 1;		/* %%% ugly %%% */
	set_file_and_anchor (hs->help, string, 0);
	checked_free (string);
    }
}

/* -------------------------------------------------------------------- *
*p  procedure-name:	go_last
**
**  purpose:		goes down one history step. 
**
**  implementation:	uses global history linked list.
** -------------------------------------------------------------------- */
static void
go_last (void)
{
    history_t* tmp_hist;

    execute ("go_last");
    
    if (history) {
    
	/*
	 *  the history has no own history entry. we just go back to the 
	 *  top of history. if the current link is no history, we 
	 *  really go back one item.
	 */
	if (current_type == 1) {

	    tmp_hist = history;

	} else {

	    tmp_hist = history->next;
	    if (!tmp_hist) return;

	    if (history->name) checked_free (history->name);
	    if (history->refname) checked_free (history->refname);
	    checked_free (history);
	    history = NULL;
	}

	hist_trace (("name: %s, id: %d\n", 
		     tmp_hist->name, tmp_hist->element_id));
    
	next_type = 0;
	set_file_and_anchor (hs->help, tmp_hist->name, tmp_hist->element_id);

	history = tmp_hist;

	/* reset the sensitivity state of the buttons.  */
	if (!history) {
	    actions_set_sensitive (action_back, False);
	} else if (!history->next) {
	    actions_set_sensitive (action_back, False);
	}
    }
    return;
}

/* -------------------------------------------------------------------- *
*p  procedure-name:	set_state
**
**  purpose:		changes the cursor icon
** -------------------------------------------------------------------- *
**  args:		busy_state: 0 = not busy, 1 = busy.
**  precondition:	help_shell must be set up and managed.
**  return type:	private void
** -------------------------------------------------------------------- */
static void
set_state (/* i  */int	busy_state)
{
#define BUSY_CURSOR XC_watch

    /* permanent data  */
    static int 		is_initialized = 0;
    static int 		is_busy = 0;
    static Cursor 	cursor;

    execute ("set_state");
    
    if (hs) {

	Widget 	shell = hs->help_shell;

	if ((shell) && XtIsRealized (shell)) {

	    Display* 	display = XtDisplay (shell);
	    Window 	window = XtWindow (shell);

	    /* we create the busy cursor once.  */
	    if (is_initialized == 0) {
		cursor = XCreateFontCursor (display, BUSY_CURSOR);
		is_busy = 0;
		is_initialized = 1;
	    }

	    /* and set the cursor according to busy_state argument.  */
	    if (busy_state != is_busy) {

		is_busy = busy_state;
		if (is_busy) {
		    XDefineCursor (display, window, cursor);
		} else {
		    XUndefineCursor (display, window);
		}
		XFlush (display);
	    }
	}
    }
}

/* -------------------------------------------------------------------- *
*p  procedure-name:	move_vertical
**
**  purpose:		goes one page up or down.
**
**  implementation:	uses the XmScrollBar actions (command arg).
**			possible actions are:
**			* PageUpOrLeft
**			* PageDownOrRight
**			* TopOrBottom
** -------------------------------------------------------------------- */
static void
move_vertical (/* i  */ Widget w,
	       /* i  */ XEvent* event,
	       /* i  */ const char* command)
{
    Widget sb;
    String params[1];
      
    execute ("move_vertical");
    
    params[0] = "0";

    XtVaGetValues (hs->help, XmNverticalScrollBar, (long)(&sb), NULL);
    if ((sb) && (XtIsManaged (sb) == True)) {
	XtCallActionProc (sb, command, event, params, 1);
    }
} 

/* -------------------------------------------------------------------- *
*g ------------- XtActionProcs ----------------------------------------
** -------------------------------------------------------------------- */
/* -------------------------------------------------------------------- *
*p  procedure-name:	help_scroll_up
**
**  purpose:		scrolling action (wrapper for move_vertical)
** -------------------------------------------------------------------- */
static void
help_scroll_up (/* i  */ Widget 	w,
		/* i  */ XEvent* 	event,
		/* i  */ String*	string,
		/* i  */ Cardinal*	c)
{
    execute ("help_scroll_up");
    move_vertical (w, event, "PageUpOrLeft");
}

/* -------------------------------------------------------------------- *
*p  procedure-name:	help_scroll_down
**
**  purpose:		action wrapper for move_vertical
** -------------------------------------------------------------------- */
static void
help_scroll_down (/* i  */ Widget 	w,
		  /* i  */ XEvent* 	event,
		  /* i  */ String*	string,
		  /* i  */ Cardinal*	c)
{
    execute ("help_scroll_down");
    move_vertical (w, event, "PageDownOrRight");
}

/* -------------------------------------------------------------------- *
*p  procedure-name:	help_scroll_home
**
**  purpose:		action wrapper for move_vertical
** -------------------------------------------------------------------- */
static void
help_scroll_home (/* i  */ Widget 	w,
		  /* i  */ XEvent* 	event,
		  /* i  */ String*	string,
		  /* i  */ Cardinal*	c)
{
    execute ("help_scroll_home");
    move_vertical (w, event, "TopOrBottom");
}

/* -------------------------------------------------------------------- *
*p  procedure-name:	help_action_index
**
**  purpose:		action wrapper for compose_index
** -------------------------------------------------------------------- */
static void
help_action_index (/* i  */ Widget 	w,
		  /* i  */ XEvent* 	event,
		  /* i  */ String*	string,
		  /* i  */ Cardinal*	c)
{
    execute ("help_action_index");
    compose_index();
}

/* -------------------------------------------------------------------- *
*p  procedure-name:	help_action_history
**
**  purpose:		action wrapper compose_history
** -------------------------------------------------------------------- */
static void
help_action_history (/* i  */ Widget 	w,
		     /* i  */ XEvent* 	event,
		     /* i  */ String*	string,
		     /* i  */ Cardinal*	c)
{
    execute ("help_action_history");
    compose_history ();
}
/* -------------------------------------------------------------------- *
*p  procedure-name:	help_action_back
**
**  purpose:		action wrapper for go last
** -------------------------------------------------------------------- */
static void
help_action_back (/* i  */ Widget 	w,
		  /* i  */ XEvent* 	event,
		  /* i  */ String*	string,
		  /* i  */ Cardinal*	c)
{
    execute ("help_action_back");
    go_last ();
}

/* -------------------------------------------------------------------- *
*p  procedure-name:	help_action_dismiss
**
**  purpose:		action wrapper for popdown help
** -------------------------------------------------------------------- */
static void
help_action_dismiss (/* i  */ Widget 	w,
		     /* i  */ XEvent* 	event,
		     /* i  */ String*	string,
		     /* i  */ Cardinal*	c)
{
    execute ("help_action_dismiss");
    popdown_help (w, NULL, NULL);
}

/* -------------------------------------------------------------------- *
*p  procedure-name:	help_cancel
**
**  purpose:		action wrapper for popdown help
** -------------------------------------------------------------------- */
static void
help_cancel (/* i  */ Widget 	w,
	     /* i  */ XEvent* 	event,
	     /* i  */ String*	string,
	     /* i  */ Cardinal*	c)
{
    execute ("help_action_cancel");

    if ((int) help_resources [help_standalone].value != 1) {
	popdown_help (w, NULL, NULL);
    }
}

/* -------------------------------------------------------------------- *
*p  procedure-name:	help_about
**
**  purpose:		print internal identification
** -------------------------------------------------------------------- */
static void
help_about (/* i  */ Widget 	w,
	    /* i  */ XEvent* 	event,
	    /* i  */ String*	string,
	    /* i  */ Cardinal*	c)
{
    buffer_t*	buf = bf_new (); /* scope -> local */
    char** 	help_path = path_the_helppath (); /* just ptr. */
    char* 	current = bcache_current (); /* scope -> local */
    
    execute ("help_about");
    
    bf_strcpy (buf, 
	       "<title>libhelp internal about</title>\n"
	       "<h1>" str_libhelp_icon " Information</h1>\n"
	       "\n<h3>release identification</h3><blockquote>"
	       "<Release name: " _system_name_
	       "<br>Release version: " _system_version_
	       "<br>Release date: " _system_date_ "</blockquote>");

    /* current document:  */
    if (current) {
	bf_strcat (buf, "<h3>current document</h3>");
	bf_sprint (buf, "<code>%s</code>", current);
	checked_free (current);
    }

    /* libhelp memory: only if mallocdebug is linked.  */

#ifdef MD_MALLOC
    /* begin  */ {
	int overhead = md_overhead ();
	int mem_usage = md_memory_usage () + overhead;
	bf_strcat (buf, "<h3>overall allocation:</h3><blockquote>");
	bf_dprint (buf, "number of allocations: %d", md_number_of_allocs ());
	bf_dprint (buf, "<br>used memory: %d k", (mem_usage / 1024));
	bf_dprint (buf, "<br>debug overhead: %d k\n", (overhead / 1024));
	bf_strcat (buf, "</blockquote>");
    }
#endif     /* MD_MALLOC  */

    /* we add information about buffer usage.  */
    bf_strcat (buf, "<h3>text buffer usage:</h3><blockquote>");
    bf_dprint (buf, "buffers: %d", bf_buffer_usage ());
    bf_dprint (buf, "<br>page size: %d k", bf_page_size () / 1024);
    bf_dprint (buf, "<br>buffer memory: %d k", bf_memory_usage () / 1024);
    bf_dprint (buf, "<br>used memory: %d k", bf_memory_filled () / 1024); 
    bf_strcat (buf, "</blockquote>");

    /* we add information about the image cache.  */
    bf_strcat (buf, "<h3>image cache:</h3><blockquote>");
    bf_dprint (buf, "default slots: %d", the_cache_default_size ());
    bf_dprint (buf, "<br>used slots: %d", the_cache_size ());
    bf_dprint (buf, "<br>memory: %d k", the_cache_mem () / 1024);
    bf_dprint (buf, "<br>colors per image: %d", the_colors ());
    bf_strcat (buf, "</blockquote>");

    if (help_path) {
	int i = 0;
	bf_strcat (buf, "<h3>help path:</h3><blockquote><pre>\n");
	while (help_path[i]) 
	    bf_sprint (buf, "%s\n", help_path[i++]);
	bf_strcat (buf, "</pre></blockquote>\n");
    }	
    

    bf_strcat (buf, "<p>" str_libhelp_bar "\n");

    /* we flag, that this page has no history entry.  */
    next_type = 1;

    /* bring the page to display.  */
    set_text_check_scroll (hs->help, bf_the_buffer (buf), 
			   "\0", "\0", 0, NULL, NULL);
    bf_free (buf);

    /* we must make it able to go back!  */
    actions_set_sensitive (action_back, True);

}
/* -------------------------------------------------------------------- *
*p  procedure-name:	help_flush
**
**  purpose:		flushes all buffers.
** -------------------------------------------------------------------- */
static void
help_flush (void)
{
    char* text = NULL;
    checked_strdup (text, "<!-- empty comment -->"); /* scope: local */
    next_type = 1;
    set_text_check_scroll (hs->help, text, "\0", "\0", 0, NULL, NULL);
    checked_free (text);

    /*
     *  the HTML widget is now prepared to reload our text. now it's
     *  save to flush the imagecache and the buffercache. note: this
     *  only affects cached documents. other buffers (e.g. the history
     *  buffer) are not freed.
     */
    bcache_flush ();
    icache_flush ();
}

/* -------------------------------------------------------------------- *
*p  procedure-name:	help_reload
**
**  purpose:		flushes all buffers and images and reloads the
** 			current document.
** -------------------------------------------------------------------- */
static void
help_reload (/* i  */ Widget 	w,
	     /* i  */ XEvent* 	event,
	     /* i  */ String*	string,
	     /* i  */ Cardinal*	c)
{
    /* the filename of the current html document.  */
    char* ref = bcache_current (); /* scope: local */

    /* the position in the current html document.  */
    int id = HTMLPositionToId (hs->help, 0, 0);

    execute ("help_reload");

    /*
     *  the HTML widget wants to free the pixmaps before setting the
     *  reloaded text. but it cant, after we flushed the image cache.
     *  so we must give it a change to free the pixmaps before
     *  we free our caches. we do this with an empty text.
     */

    if (ref) {
	help_flush ();
	/*
	 *  we've saved the old current document's ref and id.
	 *  we can redisplay it right now.
	 */
	next_type = 1;
	set_file_and_anchor (hs->help, ref, id);
	current_type = 0;
	checked_free (ref);
    } 
}

/* -------------------------------------------------------------------- *
*l  emacs:
**  local variables:
**  mode:		c
**  outline-regexp:	"\*[HGPLT]"
**  gh-language:	"English"
**  comment-column:	32
**  eval:		(outline-minor-mode t)
**  end:
** -------------------------------------------------------------------- */
