/*
 * pushbuttons.c -- Wem das simple Aussehen der Motif PushButtons schon lange
 *                  auf den Geist ging, kann dem tristen Anblick nun mittels
 *                  dieses Programm-Moduls entgehen.
 *                  Hiermit kann jeder einem PushButton ein richtig nettes
 *                  Aussehen verleihen, indem statt eines simplen Textes
 *                  ab sofort ein schoenes Bildchen angezeigt wird.
 *                  Standardmaessig sind schon einmal die oft benoetigten
 *                  Bilder fuer Ok, Abbruch, Hilfe usw. dabei.
 *		    Daneben kommen aber auch noch ordinaere Label-Widgets
 *		    in den Genuss eines bildhaften Aussehens...
 *
 * Version 1.02 vom 03.05.1994
 * Aktueller Stand dieses Moduls:
 *   03.05.1994	    Cache-Lookup ueberarbeitet, so dass verhindert wird, 
 *		    dass zwischendurch unnoetigerweise Pixmaps fuer die
 *		    Zustaende 'armed' und 'insensitive' erzeugt und dann
 *		    bei einem Treffer im Cache wieder geloescht werden.
 *		    Da dabei anfallende Kommunikation mit dem X-Server
 *		    entfaellt nun.
 *   28.04.1994	    Gadgets und Widgets werden unterstuetzt.
 *
 * (c) 1994 Harald Albrecht
 * Institut fuer Geometrie und Praktische Mathematik
 * albrecht@igpm.rwth-aachen.de
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program (see the file COPYING for more details);
 * if not, write to the Free Software Foundation, Inc., 675 Mass Ave, 
 * Cambridge, MA 02139, USA.
 *
 */

#include <Xm/Label.h>
#include <Xm/PushB.h>
#ifdef USEGADGETS
#include <Xm/LabelG.h>
#include <Xm/PushBG.h>
#endif
#include <xpm.h>
#include <pushbuttons.h>
#include <resource.h>


/* ---------------------------------------------------------------------------
 * Hier folgen nun die Pixmaps im Xpm-Format aller vordefinierten neuen
 * Befehlsschaltflaechen. Man beachte bitte die kuenstlerische Gestaltung
 * der einzelnen Schalter...
 */
#include "EmptyPB.xpm"		    /* empty push button */
#include "OkayPB.xpm"		    /* Ok */
#include "OkayPBa.xpm"
#include "CancelPB.xpm"		    /* Cancel */
#include "CancelPBa.xpm"
#include "AbortPB.xpm"		    /* Abort */
#include "AbortPBa.xpm"
#include "IgnorePB.xpm"		    /* Ignore */
#include "IgnorePBa.xpm"
#include "YesPB.xpm"		    /* Yes */
#include "YesPBa.xpm"
#include "NoPB.xpm"		    /* No */
#include "NoPBa.xpm"
#include "RetryPB.xpm"		    /* Retry */
#include "RetryPBa.xpm"
#include "HelpPB.xpm"		    /* Help */
#include "HelpPBa.xpm"
/* ---------------------------------------------------------------------------
 * Die folgenden Pixmaps werden in Message-Boxen benoetigt und stellen die
 * wichtigsten Meldungen dar: Information, Warnung, Frage, Fehler sowie
 * "bitte warten...".
 */
#include "InfoSign.xpm"
#include "ExclamationMark.xpm"
#include "QuestionMark.xpm"
#include "HandSign.xpm"
#include "TimeSign.xpm"

#define NOPIXMAP 0

/* ---------------------------------------------------------------------------
 * Damit der X-Server nicht mit unzaehligen Duplikaten von Pixmaps ueber-
 * flutet wird, implementieren wir hier flugs gleich noch einen Pixmap-Cache,
 * der Duplikate verhindert.
 * Der Cache ist zwar nicht eben sonderlich schnell, aber fuer die meisten
 * Zwecke reicht's vollkommen.
 */
typedef struct _CacheEntry CacheEntry;
struct _CacheEntry {
    CacheEntry *Next;
    Display           *Dsp;	    /* Display, auf dem d. Pixmap erscheint */
    int               Depth;	    /* Tiefe des Pixmaps                    */
    Pixel	      Background;   /* Benutzte Hintergrundfarbe            */
    char              **XpmData;    /* Das Pixmap im Xpm-Format             */
    int               Type;	    /* Art des Eintrags                     */
    Pixmap            Pixels;	    /* Kennung fuer die Pixmap im X-Server  */
    int               RefCount;	    /* Wie oft diese Pixmap benutzt wird... */
}; /* struct _CacheEntry */

/* Diese Konstanten legen fest, was denn nun wirklich hier sich in dem Eintrag
 * befindet.
 * TYPE_ORIGINAL: Dieser Eintrag wurde entweder aus dem vorgegebenen Pixmap
 *   oder aus den Daten in der Ressourcen-Datei erzeugt. Im letzteren Falle
 *   ist zusaetzlich auch noch das Bit TYPE_FROMRESOURCE gesetzt. Auf diese
 *   Weise koennen zum gleichen Pixmap auch noch Buttons existieren, die
 *   statt dessen eine Vorgabe aus den Ressourcen bekamen.
 * TYPE_ARMED: Dieses Pixmap stammt von einem 'geladenen' Bildchen. Auch hier
 *   kann wieder TYPE_FROMRESOURCE gesetzt sein.
 * TYPE_INSENSITIVE: dto....
 */
#define TYPE_ORIGINAL		0
#define TYPE_ARMED		1
#define TYPE_INSENSITIVE	2
#define TYPE_FROMRESOURCE	0x80

static CacheEntry *PixmapCache = NULL; /* The Cache himself. */


/* ---------------------------------------------------------------------------
 * Suche im Pixmap-Cache, ob das durch die uebergebenen Parameter naeher
 * spezifizierte Pixmap sich bereits im Cache befindet. Bei einem Treffer wird
 * dann dieses Pixmap zurueckgeliefert, damit die Resourcen des X-Servers
 * effektiver genutzt werden koennen. Sollte das Pixmap noch nicht vorraetig
 * sein, so wird es erst erzeugt und dann in den Cache gepackt.
 *
 * Parameter:
 *   Dsp		Display, auf dem das Pixmap angezeigt werden soll.
 *   Scr		Screen, dto.
 *   Depth		erforderliche Tiefe des Pixmaps
 *   XpmData		Pixmap im portablen Xpm-Format
 *   XpmAttr		Informationen fuer die Umwandlung von Xpm-Daten in
 *			Pixmaps
 *   ResourceXpmData	alternativ kann das Pixmap auch statt aus dem C-Code
 *			aus den Daten in einem Zwischenspeicher erzeugt 
 *			werden.
 *   Type               Ermoeglicht es, auch ein Pixmap anstatt der Xpm-
 *                      Daten vorzugeben. Aber auch dann wird trotzdem noch
 *			der XpmData-Parameter benoetigt, um es zu ermoeg-
 *			lichen, dass keine doppelten Kopien des gleichen
 *			vorgegebenen Pixmaps entstehen.
 *   Default		Hier laesst sich bei Bedarf auch noch ein Pixmap
 *                      vorgeben.
 *   JustLookup		Hier kann angegeben werden, dass dieser Aufruf erst
 *			einmal nur einen Versuch darstellt. Falls True, und
 *			das Pixmap konnte gefunden werden, dann wird dieses
 *			Pixmap in Beschlag genommen (Referenzzaehler wird
 *			erhoeht). Ansonsten bei einem Misserfolg und dieses
 *			Flag ist False kehrt die Routine unverrichteter Dinge
 *			zum Aufrufer zurueck. Der muss dann in einem zweiten
 *			Anlauf das erforderliche Pixmap erzeugen und hier
 *			abliefern.
 *
 * Ergebnis:
 *   NULL, falls das Pixmap noch nicht im Cache ist und ein Fehler bei der
 *   Umwandlung der uebergebenen Daten in ein Pixmap auftrat. Ansonsten ist
 *   es die Kennung des entsprechenden Pixmaps.
 */
static Pixmap AddXpmToCache(Display *Dsp, Screen *Scr, 
                            int Depth, Pixel Background, 
                            char **XpmData, XpmAttributes *XpmAttr,
			    char *ResourceXpmData, 
                            int Type, Pixmap Default, 
			    Boolean JustLookup)
{
    CacheEntry *pCache = PixmapCache;
    
    if ( ResourceXpmData ) Type = Type | TYPE_FROMRESOURCE;
    while ( (pCache != NULL) &&
            ( (pCache->Dsp        != Dsp       ) || 
              (pCache->Depth      != Depth     ) ||
	      (pCache->Background != Background) ||
              (pCache->XpmData    != XpmData   ) ||
              (pCache->Type       != Type      )
            ) ) pCache = pCache->Next;
    if ( pCache == NULL ) {
        /* 
	 * Zu den Quelldaten im Xpm-Format existiert bislang noch kein 
         * bereits im X-Server-Format vorliegendes Pixmap. Wir muessen hier
         * daher jetzt dieses erst erzeugen und fuegen es dann bei dieser 
         * Gelegenheit auch gleich in den Cache ein. Falls der Aufrufer
	 * nur 'mal 'reinschauen wollte, so entfellt der Schritt mit dem
	 * Erzeugen der Pixmaps.
         */
	int    Error;
	Pixmap Pixels;
        
	if ( JustLookup ) return NOPIXMAP;
	if ( Default == NOPIXMAP ) {
	    /* Nur wenn kein Pixmap vom Aufrufer vorgegeben wurde, muessen
	     * wir uns hier darum kuemmern, dass ein Pixmap erstellt wird.
	     * Dabei kann das Pixmap entweder direkt im C-Code eingebunden
	     * sein, oder aber aus einem Buffer (vorzugsweise die Ressourcen-
	     * Datenbank) stammen.
	     */
	    if ( ResourceXpmData )
		Error = XpmCreatePixmapFromBuffer(
		          Dsp, RootWindowOfScreen(Scr),
		          ResourceXpmData, &Pixels, NULL, XpmAttr);
	    else 
		Error = XpmCreatePixmapFromData(
		          Dsp, RootWindowOfScreen(Scr),
		          XpmData, &Pixels, NULL, XpmAttr);
	    if ( Error != XpmSuccess ) return NOPIXMAP;
	    XpmFreeAttributes(XpmAttr);
	} else {
	    /* Ansonsten nehmen wir die Vorgabe. */
	    Pixels = Default;
	}

	pCache = (CacheEntry *) malloc(sizeof(CacheEntry));
	if ( pCache == NULL ) {
	    /* Urgs -- kein Speicher mehr ??? */
	    XFreePixmap(Dsp, Pixels);
	    return NOPIXMAP;
	}
	pCache->Dsp        = Dsp;
	pCache->Depth      = Depth;
	pCache->Background = Background;
	pCache->XpmData    = XpmData;
	pCache->Type       = Type;
	pCache->Pixels     = Pixels;
	pCache->RefCount   = 1;
         
	/* Nun noch einketten in die Cache-Liste */
	pCache->Next = PixmapCache;
	PixmapCache = pCache;
        
	return Pixels;
     } else {
         /* Ansonsten haben wir bereits ein passendes Pixmap auf Halde
          * liegen, dass wir nun verwenden.
	  */
	++(pCache->RefCount);
	return pCache->Pixels;
     }
} /* AddXpmToCache */

/* ---------------------------------------------------------------------------
 * Nachdem ein Button abgeraeumt wird, kann auch im Cache nachgeschaut werden,
 * ob das dazugehoerige Pixmap u.U. entfernt werden kann. Dieses ist immer 
 * dann der Fall, wenn kein anderer Button mehr Ansprueche auf das Pixmap
 * erhebt.
 * 
 * Die Parameter gestalten sich uebersichtlich:
 *   Dsp		Diese beiden Parameter genuegen beim Loeschen
 *   Pixels		bereits, um ein Pixmap denn wirklich eindeutig
 *			zu identifizieren. Das Display muessen wir auch
 *			nur deswegen wissen, da zwei unterschiedliche
 *			Pixmaps auf zwei unterschiedlichen Displays
 *			moeglicherweise von den beiden unterschiedlichen
 *			X-Servern aus purem Zufall die gleiche Kennung
 *			erhalten. (Man glaubt ja gar nicht, was manchmal
 *			so alles passiert...)
 */
static void FreeXpmFromCache(Display *Dsp, Pixmap Pixels)
{
    CacheEntry *pCache = PixmapCache;
    CacheEntry *pPrev  = NULL;
    
    while ( (pCache != NULL) &&
            ( (pCache->Dsp    != Dsp   ) ||
              (pCache->Pixels != Pixels)
            ) ) { pPrev = pCache; pCache = pCache->Next; }
    if ( pCache == NULL ) {
        /* Hoppla!! Das darf aber nicht passieren... da hat irgendjemand
         * die Pixmaps durcheinander gebracht!
         */
        XtWarningMsg("PixmapCache", "pixmapcache", "UnknownPixmap",
                     "Pixmap to be freed not found in pixmap cache",
                     NULL, 0);
    } else {
        if ( --(pCache->RefCount) == 0 ) {
            /* Zeit, das Pixmap aus dem Cache wieder zu entfernen!
             * Damit werden dann im X-Server wieder Ressourcen frei.
             */
            XFreePixmap(Dsp, pCache->Pixels);
            if ( pPrev == NULL )
                /* Der allererste Eintrag ist zu loeschen... */
                PixmapCache = pCache->Next;
            else
                /* Es ist irgendwo mittendrin ... */
                pPrev->Next = pCache->Next;
            free( (char *) pCache);
        }
    }
} /* FreeXpmFromCache */


/* ---------------------------------------------------------------------------
 * Dieses ist ein XmNdestroyCallback, der immer dann aufgerufen wird, wenn
 * das Widget wieder aus dem Speicher entfernt wird. In diesem Falle
 * muessen wir auch noch die Pixmaps wieder entfernen, damit sie nicht
 * im Server zurueckbleiben.
 */
static void DestroyCallback(Widget w, XtPointer ClientData,
                            XtPointer CallData)
{
    Display *Dsp;
    Pixmap  NormalPixmap, ArmedPixmap, InsensitivePixmap;
    Boolean IsButton;
    
    /* Sollte doch tatsaechlich zuvor der Speicher ausgegangen sein,
     * uebergehen wir hier das Aufraeumen. Eigentlich ist diese
     * Abfrage sowieso das reinste defensive Programmieren...das
     * Programm wird sicherlich schon lange vorher bei Speicher-
     * mangel abgeschmiert sein!
     */
    if ( ClientData == NULL ) return;

    IsButton = XmIsPushButton(w) || XmIsPushButtonGadget(w);
    NormalPixmap      = ((Pixmap *) ClientData)[0];
    InsensitivePixmap = ((Pixmap *) ClientData)[1];
    Dsp = XtDisplay(w);
    FreeXpmFromCache(Dsp, NormalPixmap);
    FreeXpmFromCache(Dsp, InsensitivePixmap);
    if ( IsButton ) {
	ArmedPixmap = ((Pixmap *) ClientData)[2];
        FreeXpmFromCache(Dsp, ArmedPixmap);
    }
    free((char *) ClientData);
} /* DestroyCallback */


/* ---------------------------------------------------------------------------
 * Richte eine Befehlsschaltflaeche mit den angegebenen Bildern ein.
 *
 * Parameter:
 *   PushButton		Der PushButton (oder Gadget), fuer den die Pixmaps
 *			zu erstellen sind.
 *   Normal		Zeiger auf die Daten eines Pixmaps (im Xpm-Format),
 *			welches die Schaltflaeche im Normalzustand darstellt.
 *   Armed		Zeiger auf Daten eines Pixmaps, welches die Schalt-
 *			flaeche im gedrueckten Zustand darstellt. Sollte hier
 *			aus Faulheit NULL uebergeben werden, so wird statt
 *			dessen das Pixmap fuer den Normalzustand benutzt.
 *			Allerdings duerften die Anwender darueber ueberhaupt
 *			nicht gluecklich sein.
 *   Insensitive	Zeiger auf die Daten eines Pixmaps, das die Schalt-
 *			flaeche im nicht benutzbaren Zustand zeigt. Wird hier
 *			NULL uebergeben, so erzeugt diese Routine ein ge-
 *			eignetes Bild durch Kombination des Pixmaps fuer den
 *			Normalzustand mit einen Raster.
 *
 * Ergebnis:
 *   keines. Wenn alles in Ordnung ging, dann sind die entsprechenden Pixmaps
 *   erzeugt und der Schaltflaeche mitgeteilt worden. Damit wird es dann ent-
 *   sprechend bunter auf dem Bildschirm.
 *
 */
static
void CreateButtonPixmaps(Widget PushButton,
                                char **Normal, char **Armed, 
                                char **Insensitive)
{
    Display        *Dsp;
    Screen         *Scr;
    int            Depth;
    Pixel          Background;
    int            PixmapX, PixmapY;
    unsigned int   PixmapHeight, PixmapWidth, PixmapBorder, PixmapDepth;
    Window         RootWindow;
    Pixmap         NormalPixmap, ArmedPixmap, InsensitivePixmap;
    XpmAttributes  XpmAttr;
    Boolean        RecomputeSize;
    char           *ResourceData;
    Boolean        HasResourceData;
    Pixmap         *PixmapReminder;
    XpmColorSymbol XpmTransparentColor[1] = 
		       {{ NULL, "none", 0 }};
    
    /* Hierbei muessen wir etwas vorsichtig zu Werke gehen, denn es
     * kann sein, dass hier ein Gadget statt eines richtigen Widgets
     * uebergeben wurde. In diesem Fall erfahren wir Hintergrundfarbe
     * und die Tiefe des Bildspeichers (oder besser Visuals) nur vom
     * Vater und nicht vom Gadgets. Da es keine Composite Gadgets gibt, 
     * muss damit der Vater dieses Gadets ein Nachkomme von Core sein.
     * Also existiert folglich dort ein Hintergrundfarben-Ressource.
     */
    Dsp = XtDisplayOfObject(PushButton);
    Scr = XtScreenOfObject(PushButton);
    XtVaGetValues(XtIsSubclass(PushButton, coreWidgetClass) ?
                      PushButton : XtParent(PushButton), 
                  XmNdepth,      &Depth, 
		  XmNbackground, &Background, 
		  NULL);
    
    /* Erzeuge aus den noch im lesbaren Format vorliegenden Grafikdaten
     * die benoetigten Pixmaps.
     * Zuerst ist das Pixmap fuer den Normalzustand (ungedrueckt) an der
     * Reihe...
     */
    ResourceData = XmLoadStringResource(PushButton, "normalFace", 
                                        NULL, NULL);
    HasResourceData = ResourceData != NULL;
    XpmTransparentColor[0].pixel = Background;
    XpmAttr.valuemask    = XpmColorSymbols | XpmCloseness | XpmDepth;
    XpmAttr.colorsymbols = XpmTransparentColor;
    XpmAttr.numsymbols   = 1;
    XpmAttr.closeness    = 65535;
    XpmAttr.depth        = Depth;
    NormalPixmap = AddXpmToCache(Dsp, Scr, Depth, Background, 
                                 Normal, &XpmAttr, ResourceData, 
				 TYPE_ORIGINAL, NOPIXMAP, False);
    if ( NormalPixmap == NOPIXMAP ) return;
    
    /* Dannach knoepfen wir uns den gedrueckten Zustand vor... */
    ResourceData = XmLoadStringResource(PushButton, "armedFace", 
                                        NULL, NULL);
    if ( (Armed && !HasResourceData) || 
         (ResourceData && HasResourceData) ) {
	XpmAttr.valuemask    = XpmColorSymbols | XpmCloseness | XpmDepth;
	XpmAttr.colorsymbols = XpmTransparentColor;
	XpmAttr.numsymbols   = 1;
        XpmAttr.closeness    = 65535;
        XpmAttr.depth        = Depth;
        ArmedPixmap = AddXpmToCache(Dsp, Scr, Depth, Background, 
	                            Armed, &XpmAttr, ResourceData, 
				    TYPE_ORIGINAL, NOPIXMAP, False);
        if ( ArmedPixmap == NOPIXMAP ) {
            FreeXpmFromCache(Dsp, NormalPixmap);
            return;
        }
    } else {
	/* Wenn's kein Pixmap fuer diesen Zustand gibt, dann basteln
	 * wir uns eben einen selbst. Dies geschieht, indem wir das
	 * Pixmap fuer den Normalzustand nehmen und dann um die Breite
	 * des Rands um die Schaltflaeche nach unten und nach rechts
	 * verschoben in ein neues Pixmap kopieren.
	 */
        GC        gc;
        XGCValues GCValues;
        Dimension ShadowThickness;
        
	/* Bevor hier ueberhaupt erst einmal irgendein Befehl an den
	 * X-Server ergeht, versuchen wir, ob wir einen Treffer landen
	 * koennen...
	 */
        ArmedPixmap = AddXpmToCache(Dsp, Scr, Depth, Background, 
	                            Normal, &XpmAttr, NULL, 
				    TYPE_ARMED, NOPIXMAP, True);
	if ( ArmedPixmap == NOPIXMAP ) {
	    /* Also doch zeichnen...
	     * Zuerst benoetigen wir einen geeigneten GC, um das Pixmap
	     * erstellen zu koennen, denn die Streifen, die beim Kopieren
	     * nicht mit erfasst werden, muessen ja noch mit der Hinter-
	     * grundfarbe gepinselt werden.
	     */
	    XtVaGetValues(PushButton, 
			  XmNshadowThickness, &ShadowThickness, NULL);
	    GCValues.foreground = Background;
	    gc = XtGetGC(PushButton, GCForeground, &GCValues);
	    
	    XGetGeometry(Dsp, NormalPixmap, &RootWindow,
			 &PixmapX, &PixmapY, &PixmapWidth, &PixmapHeight,
			 &PixmapBorder, &PixmapDepth);
	    ArmedPixmap = XCreatePixmap(Dsp, RootWindowOfScreen(Scr),
					PixmapWidth, PixmapHeight,
					PixmapDepth);
	    XCopyArea(Dsp, NormalPixmap, ArmedPixmap, gc,
		      0, 0, 
		      PixmapWidth - ShadowThickness, 
		      PixmapHeight - ShadowThickness, 
		      ShadowThickness, ShadowThickness);
	    XFillRectangle(Dsp, ArmedPixmap, gc, 0, 0, 
			   PixmapWidth, ShadowThickness);
	    XFillRectangle(Dsp, ArmedPixmap, gc, 0, 0, 
			   ShadowThickness, PixmapHeight);
	    
	    XtReleaseGC(PushButton, gc);
	    if ( AddXpmToCache(Dsp, Scr, Depth, Background, Normal, 
	                       &XpmAttr, NULL, TYPE_ARMED, ArmedPixmap, 
			       False) == NOPIXMAP ) {
		FreeXpmFromCache(Dsp, NormalPixmap);
		XFreePixmap(Dsp, ArmedPixmap);
		return;
	    }
	}
    }
    
    /* Abschliessend noch der insensitive Zustand...das ist aber
     * etwas mehr Arbeit.
     */
    ResourceData = XmLoadStringResource(PushButton, "insensitiveFace", 
                                        NULL, NULL);
    if ( (Insensitive && !HasResourceData) || 
         (ResourceData && HasResourceData) ) {
        /* Wenn etwas vorgegeben wurde, dann nimm' das... */
	XpmAttr.valuemask    = XpmColorSymbols | XpmCloseness | XpmDepth;
	XpmAttr.colorsymbols = XpmTransparentColor;
	XpmAttr.numsymbols   = 1;
        XpmAttr.closeness    = 65535;
        XpmAttr.depth        = Depth;
        InsensitivePixmap = AddXpmToCache(Dsp, Scr, Depth, Background, 
                                          Insensitive, &XpmAttr,
					  ResourceData, 
                                          TYPE_ORIGINAL, NOPIXMAP, False);
        if ( InsensitivePixmap == NOPIXMAP ) {
            FreeXpmFromCache(Dsp, NormalPixmap);
            FreeXpmFromCache(Dsp, ArmedPixmap);
            return;
        }
    } else {
        /* ...ansonsten basteln wir uns eine geeignete Pixmap aus
         * dem Pixmap fuer den Normalzustand zusammen.
         */
        GC        gc;
        XGCValues GCValues;
        Pixmap    Stipple;
        static    char StippleBitmap[8] =
            { 0x55, 0xAA, 0x55, 0xAA, 0x55, 0xAA, 0x55, 0xAA };
            
        InsensitivePixmap = AddXpmToCache(Dsp, Scr, Depth, Background, 
	                                  Normal, &XpmAttr, NULL, 
					  TYPE_INSENSITIVE, 
                                          InsensitivePixmap, True);
	if ( InsensitivePixmap == NOPIXMAP ) {
	    /* Zuerst benoetigen wir das Raster, welches ueber das Pixmap
	     * fuer den insensitiven Zustand gelegt wird.
	     */
	    Stipple = XCreateBitmapFromData(Dsp, RootWindowOfScreen(Scr),
					    StippleBitmap, 8, 8);
	    /* ...dann besorgen wir uns einen geeigneten GC, mit dessen
	     * Hilfe wir das Pixmap malen koennen.
	     */
	    GCValues.foreground = Background;
	    GCValues.fill_style = FillStippled;
	    GCValues.stipple    = Stipple;
	    gc = XtGetGC(PushButton, 
			 GCForeground | GCFillStyle | GCStipple,
			 &GCValues);
	    XGetGeometry(Dsp, NormalPixmap, &RootWindow,
			 &PixmapX, &PixmapY, &PixmapWidth, &PixmapHeight,
			 &PixmapBorder, &PixmapDepth);
	    InsensitivePixmap = XCreatePixmap(Dsp, RootWindowOfScreen(Scr),
					      PixmapWidth, PixmapHeight,
					      PixmapDepth);
	    XCopyArea(Dsp, NormalPixmap, InsensitivePixmap, gc,
		      0, 0, PixmapWidth, PixmapHeight, 0, 0);
	    XFillRectangle(Dsp, InsensitivePixmap, gc, 
			   0, 0, PixmapWidth, PixmapHeight);
	    
	    XtReleaseGC(PushButton, gc);
	    XFreePixmap(Dsp, Stipple);
	    if ( AddXpmToCache(Dsp, Scr, Depth, Background, Normal, 
	                       &XpmAttr, NULL, TYPE_INSENSITIVE, 
			       InsensitivePixmap, False) == NOPIXMAP ) {
		FreeXpmFromCache(Dsp, NormalPixmap);
		FreeXpmFromCache(Dsp, ArmedPixmap);
		XFreePixmap(Dsp, InsensitivePixmap);
		return;
	    }
	}
    }
    
    /* Wenn nun alles in Ordnung ging, koennen wir die Bilderchen dem
     * Button ueberreichen, damit's ab sofort auf dem Bildschirm viel
     * bunter zugeht.
     */
    XtVaGetValues(PushButton, 
                  XmNrecomputeSize, &RecomputeSize, NULL);
    XtVaSetValues(PushButton, XmNrecomputeSize, True, NULL);
    XtVaSetValues(PushButton,
                  XmNlabelType,              XmPIXMAP,
                  XmNlabelPixmap,            NormalPixmap,
                  XmNarmPixmap,              ArmedPixmap,
                  XmNlabelInsensitivePixmap, InsensitivePixmap,
                  XmNarmColor,               Background,
                  NULL);
    XtVaSetValues(PushButton, XmNrecomputeSize, RecomputeSize, NULL);

    /* Damit spaeter keine Daten zurueckbleiben, muessen wir noch einen
     * Callback installieren, der nach getaner Arbeit alles wieder
     * aufraeumt, damit kein Schmutz liegen bleibt. Dazu uebergeben wir
     * dem Callback einen Datenblock, indem vermerkt ist, welche Pixmaps
     * wegzuraeumen sind... Leider koennen wir hier nicht beim Aufraeumen
     * XmNlabelPixmap benutzen, da hierdrin das ArmedPixmap steht, wenn
     * der Pushbutton mit der Tastatur ausgeloest wird und in Folge
     * dessen der Callback ausgeloest wird. Ist das ein Mist!!!!
     */
    PixmapReminder = (Pixmap *) malloc(sizeof(Pixmap)*3);
    if ( PixmapReminder != NULL ) {
	PixmapReminder[0] = NormalPixmap;
	PixmapReminder[1] = InsensitivePixmap;
	PixmapReminder[2] = ArmedPixmap;
    }
    XtAddCallback(PushButton, XmNdestroyCallback,
                  (XtCallbackProc) DestroyCallback, 
		  (XtPointer) PixmapReminder);
} /* CreateButtonPixmaps */

/* ---------------------------------------------------------------------------
 * Hier folgen nun zwei Routinen, mit deren Hilfe sich ein PushButton-Widget
 * oder ein -Gadget einrichten laesst, das dann statt eines Textes Bilder
 * anzeigt.
 */
Widget XmCreatePixmapPushButton(Widget Parent, String Name,
                                char **Normal, char **Armed, 
                                char **Insensitive,
                                ArgList Args, Cardinal ArgCount)
{
    Widget ButtonWidget;
    
    ButtonWidget = XmCreatePushButton(Parent, Name, Args, ArgCount);
    if ( ButtonWidget == NULL ) return NULL;
    CreateButtonPixmaps(ButtonWidget, Normal, Armed, Insensitive);
    return ButtonWidget;
} /* XmCreatePixmapPushButton */

#ifdef USEGADGETS
Widget XmCreatePixmapPushButtonGadget(Widget Parent, String Name,
                                char **Normal, char **Armed, 
                                char **Insensitive,
                                ArgList Args, Cardinal ArgCount)
{
    Widget ButtonWidget;
    
    ButtonWidget = XmCreatePushButtonGadget(Parent, Name, Args, ArgCount);
    if ( ButtonWidget == NULL ) return NULL;
    CreateButtonPixmaps(ButtonWidget, Normal, Armed, Insensitive);
    return ButtonWidget;
} /* XmCreatePixmapPushButtonGadget */
#endif

/* ---------------------------------------------------------------------------
 * Richte eine Befehlsschaltflaeche (mikroweichisch fuer "push button") mit
 * einem der vorgegebenen Bilder ein.
 *
 * Parameter:
 *   Parent		Elterliches Widget, zu dem die Befehlsschaltflaeche ge-
 *			hoeren soll.
 *   Name		Name, auf den das Widget hoeren soll.
 *   FaceType		Eine der XmPUSHBUTTON_xxx-Konstanten, die festlegt,
 *			welches Bild auf der Befehlschaltflaeche erscheinen
 *			soll.
 *   Args		Die Parameter.
 *   ArgCount		Anzahl der uebergebenen Parameter.
 *
 * Ergebnis:
 *   liefert bei Erfolg das Widget, also den PushButton, zurueck.
 */
static Widget CreateStandardPushButton(Widget Parent, String Name, 
                                       int FaceType,
                                       ArgList Args, Cardinal ArgCount, 
				       Boolean Gadget)
{
    char **Normal, **Armed, **Insensitive;
    
    /* Ueberpruefe zunaechst den FaceType Parameter auf seinen Definitions-
     * bereich. Falls ausserhalb, wird immer XmPUSHBUTTON_NONE angenommen...
     * Selbst schuld!
     */
    if ( (FaceType < XmPUSHBUTTON_NONE) || (FaceType > XmPUSHBUTTON_HELP))
        FaceType = XmPUSHBUTTON_NONE;
    /* Jetzt muessen wir uns die passenden Pixmaps heraussuchen, um damit
     * die Schaltflaeche zu erzeugen.
     */
    Insensitive = NULL;
    switch ( FaceType ) {
        case XmPUSHBUTTON_NONE:
            Normal      = EmptyPB_xpm; /* Hier gibt es nur ein einziges */
            Armed       = NULL;        /* Bildchen, die restlichen      */
            break;                     /* werden automatisch erzeugt.   */
        case XmPUSHBUTTON_OK:
            Normal      = OkayPB_xpm;
            Armed       = OkayPBa_xpm;
            break;
        case XmPUSHBUTTON_CANCEL:
            Normal      = CancelPB_xpm;
            Armed       = CancelPBa_xpm;
            break;
        case XmPUSHBUTTON_ABORT:
            Normal      = AbortPB_xpm;
            Armed       = AbortPBa_xpm;
            break;
        case XmPUSHBUTTON_IGNORE:
            Normal      = IgnorePB_xpm;
            Armed       = IgnorePBa_xpm;
            break;
        case XmPUSHBUTTON_YES:
            Normal      = YesPB_xpm;
            Armed       = YesPBa_xpm;
            break;
        case XmPUSHBUTTON_NO:
            Normal      = NoPB_xpm;
            Armed       = NoPBa_xpm;
            break;
        case XmPUSHBUTTON_RETRY:
            Normal      = RetryPB_xpm;
            Armed       = RetryPBa_xpm;
            break;
        case XmPUSHBUTTON_HELP:
            Normal      = HelpPB_xpm; 
            Armed       = HelpPBa_xpm; 
            break;
    }
#ifdef USEGADGETS
    if ( Gadget )
	return XmCreatePixmapPushButtonGadget(Parent, Name,
					      Normal, Armed, Insensitive,
					      Args, ArgCount);
#endif
    return XmCreatePixmapPushButton(Parent, Name,
                                    Normal, Armed, Insensitive,
                                    Args, ArgCount);
} /* CreateStandardPushButton */

/* ---------------------------------------------------------------------------
 * Hier sind wiederum zwei Routinen, mit deren Hilfe wahlweise Widgets oder
 * Gadgets erzeugt werden koennen. Die Schaltflaechen erhalten dabei in diesem
 * Fall eines der vorgegebenen Bilder.
 */
Widget XmCreateStandardPushButton(Widget Parent, String Name, 
                                  int FaceType,
                                  ArgList Args, Cardinal ArgCount)
{
    return CreateStandardPushButton(Parent, Name, FaceType, 
                                    Args, ArgCount, False);
} /* XmCreateStandardPushButton */

#ifdef USEGADGETS
Widget XmCreateStandardPushButtonGadget(Widget Parent, String Name, 
                                        int FaceType,
                                        ArgList Args, Cardinal ArgCount)
{
    return CreateStandardPushButton(Parent, Name, FaceType, 
                                    Args, ArgCount, True);
} /* XmCreateStandardPushButtonGadget */
#endif

/* ---------------------------------------------------------------------------
 * Erzeuge ein statisches Textfeld, das aber statt Textes nun Bilder an-
 * zeigt. Auch hier laesst sich wieder - wie auch schon bei den Schalt-
 * flaechen - wahlweise das anzuzeigende Bild in einer Ressourcendatei
 * festlegen.
 */
static void CreatePixmapLabel(Widget LabelWidget,
                              char **Normal, char **Insensitive)
{
    Display        *Dsp;
    Screen         *Scr;
    int            Depth;
    int            PixmapX, PixmapY;
    unsigned int   PixmapHeight, PixmapWidth, PixmapBorder, PixmapDepth;
    Window         RootWindow;
    Pixmap         NormalPixmap, ArmedPixmap, InsensitivePixmap;
    XpmAttributes  XpmAttr;
    Boolean        RecomputeSize;
    Pixel          Background;
    char           *ResourceData;
    Boolean        HasResourceData;
    Pixmap         *PixmapReminder;
    XpmColorSymbol XpmTransparentColor[1] = 
		       {{ NULL, "none", 0 }};
    
    
    /* Auch hier muessen wir wieder aufpassen, ob es ein Widget
     * oder ein Gadget ist! Ansonsten gelten die bereits weiter
     * oben gemachten Aussagen.
     */ 
    Dsp = XtDisplayOfObject(LabelWidget);
    Scr = XtScreenOfObject(LabelWidget);
    XtVaGetValues(XtIsSubclass(LabelWidget, coreWidgetClass) ?
                      LabelWidget : XtParent(LabelWidget), 
                  XmNdepth,      &Depth, 
		  XmNbackground, &Background, 
		  NULL);
    
    /* Erzeuge aus den noch im lesbaren Format vorliegenden Grafikdaten
     * die benoetigten Pixmaps.
     * Zuerst ist das Pixmap fuer den Normalzustand an der Reihe.
     */
    ResourceData = XmLoadStringResource(LabelWidget, "normalFace", 
                                        NULL, NULL);
    HasResourceData = ResourceData != NULL;
    XpmTransparentColor[0].pixel = Background;
    XpmAttr.valuemask    = XpmColorSymbols | XpmCloseness | XpmDepth;
    XpmAttr.colorsymbols = XpmTransparentColor;
    XpmAttr.numsymbols   = 1;
    XpmAttr.closeness    = 65535;
    XpmAttr.depth        = Depth;
    NormalPixmap = AddXpmToCache(Dsp, Scr, Depth, Background, 
                                 Normal, &XpmAttr, ResourceData, 
                                 TYPE_ORIGINAL, NOPIXMAP, False);
    if ( NormalPixmap == NOPIXMAP ) return;
    
    /* Abschliessend noch der insensitive Zustand...das ist aber
     * etwas mehr Arbeit.
     */
    ResourceData = XmLoadStringResource(LabelWidget, "insensitiveFace", 
                                        NULL, NULL);
    if ( (Insensitive && !HasResourceData) || 
         (ResourceData && HasResourceData) ) {
        /* Wenn etwas vorgegeben wurde, dann nimm' das... */
        XpmAttr.valuemask = XpmCloseness | XpmDepth;
        XpmAttr.closeness = 65535;
        XpmAttr.depth     = Depth;
        InsensitivePixmap = AddXpmToCache(Dsp, Scr, Depth, Background, 
                                          Insensitive, &XpmAttr,
					  ResourceData, 
                                          TYPE_ORIGINAL, NOPIXMAP, False);
        if ( InsensitivePixmap == NOPIXMAP ) {
            FreeXpmFromCache(Dsp, NormalPixmap);
            return;
        }
    } else {
        /* ...ansonsten basteln wir uns eine geeignete Pixmap aus
         * dem Pixmap fuer den Normalzustand zusammen.
         */
        GC        gc;
        XGCValues GCValues;
        Pixmap    Stipple;
        static    char StippleBitmap[8] =
            { 0x55, 0xAA, 0x55, 0xAA, 0x55, 0xAA, 0x55, 0xAA };
        
        InsensitivePixmap = AddXpmToCache(Dsp, Scr, Depth, Background, 
	                                  Normal, &XpmAttr, NULL, 
					  TYPE_INSENSITIVE, 
                                          InsensitivePixmap, True);
	if ( InsensitivePixmap == NOPIXMAP ) {
	    /* Zuerst benoetigen wir das Raster, welches ueber das Pixmap
	     * fuer den insensitiven Zustand gelegt wird.
	     */
	    Stipple = XCreateBitmapFromData(Dsp, RootWindowOfScreen(Scr),
					    StippleBitmap, 8, 8);
	    /* ...dann besorgen wir uns einen geeigneten GC, mit dessen
	     * Hilfe wir das Pixmap malen koennen.
	     */
	    GCValues.foreground = Background;
	    GCValues.fill_style = FillStippled;
	    GCValues.stipple    = Stipple;
	    gc = XtGetGC(LabelWidget, 
			 GCForeground | GCFillStyle | GCStipple,
			 &GCValues);
	    XGetGeometry(Dsp, NormalPixmap, &RootWindow,
			 &PixmapX, &PixmapY, &PixmapWidth, &PixmapHeight,
			 &PixmapBorder, &PixmapDepth);
	    InsensitivePixmap = XCreatePixmap(Dsp, RootWindowOfScreen(Scr),
					      PixmapWidth, PixmapHeight,
					      PixmapDepth);
	    XCopyArea(Dsp, NormalPixmap, InsensitivePixmap, gc,
		      0, 0, PixmapWidth, PixmapHeight, 0, 0);
	    XFillRectangle(Dsp, InsensitivePixmap, gc, 
			   0, 0, PixmapWidth, PixmapHeight);
	    
	    XtReleaseGC(LabelWidget, gc);
	    XFreePixmap(Dsp, Stipple);
	    if ( AddXpmToCache(Dsp, Scr, Depth, Background, Normal, 
	                       &XpmAttr, NULL, TYPE_INSENSITIVE, 
			       InsensitivePixmap, False) == NOPIXMAP ) {
		FreeXpmFromCache(Dsp, NormalPixmap);
		XFreePixmap(Dsp, InsensitivePixmap);
		return;
	    }
	}
    }
    
    /* Wenn nun alles in Ordnung ging, koennen wir die Bilderchen dem
     * Button ueberreichen, damit's ab sofort auf dem Bildschirm viel
     * bunter zugeht.
     */
    XtVaGetValues(LabelWidget, 
                  XmNrecomputeSize, &RecomputeSize, NULL);
    XtVaSetValues(LabelWidget, XmNrecomputeSize, True, NULL);
    XtVaSetValues(LabelWidget,
                  XmNlabelType,              XmPIXMAP,
                  XmNlabelPixmap,            NormalPixmap,
                  XmNlabelInsensitivePixmap, InsensitivePixmap,
                  NULL);
    XtVaSetValues(LabelWidget, XmNrecomputeSize, RecomputeSize, NULL);

    /* Damit spaeter keine Daten zurueckbleiben, muessen wir noch einen
     * Callback installieren, der nach getaner Arbeit alles wieder
     * aufraeumt, damit kein Schmutz liegen bleibt. Auch hier gilt noch
     * einmal das zuvor gesagte (sie PushButtons).
     */
    PixmapReminder = (Pixmap *) malloc(sizeof(Pixmap)*2);
    if ( PixmapReminder != NULL ) {
	PixmapReminder[0] = NormalPixmap;
	PixmapReminder[1] = InsensitivePixmap;
    }
    XtAddCallback(LabelWidget, XmNdestroyCallback,
                  (XtCallbackProc) DestroyCallback, 
		  (XtPointer) PixmapReminder);
} /* CreatePixmapLabel */

/* ---------------------------------------------------------------------------
 * Und noch einmal zwei Routinen, um Label-Widgets oder -Gadgets mit einem
 * Bilchen 'drauf zu erzeugen.
 */
Widget XmCreatePixmapLabel(Widget Parent, String Name, 
                           char **Normal, char **Insensitive,
                           ArgList Args, Cardinal ArgCount)
{
    Widget LabelWidget;
    
    LabelWidget = XmCreateLabel(Parent, Name, Args, ArgCount);
    if ( LabelWidget == NULL ) return NULL;
    CreatePixmapLabel(LabelWidget, Normal, Insensitive); 
    return LabelWidget;
} /* XmCreatePixmapLabel */

#ifdef USEGADGETS
Widget XmCreatePixmapLabelGadget(Widget Parent, String Name, 
                                 char **Normal, char **Insensitive,
                                 ArgList Args, Cardinal ArgCount)
{
    Widget LabelWidget;
    
    LabelWidget = XmCreateLabelGadget(Parent, Name, Args, ArgCount);
    if ( LabelWidget == NULL ) return NULL;
    CreatePixmapLabel(LabelWidget, Normal, Insensitive); 
    return LabelWidget;
} /* XmCreatePixmapLabelGadget */
#endif

/* ---------------------------------------------------------------------------
 * Erzeuge ein Label mit einem der bereits vordefinierten Bilder, als da
 * waeren: Information, Frage, Warnung und Fehler.
 * 
 * Parameter:
 *   Parent	    Elternwidget, zu dem das Bild-Widget gehoeren soll.
 *   Name	    Name des Widgets (Doppelnamen wie Schmitz-Mueller-
 *		    Luedenscheidt werden mit einem Core-Dump geahndet).
 *   FaceType	    Gibt an, welches Bild angezeigt werden soll.
 *   Args	    Weitere Ressourcen-Parameter
 *   ArgCount	    Anzahl der in Args uebergebenen Ressourcen-Einstel-
 *		    lungen.
 */
static Widget CreateStandardLabel(Widget Parent, String Name, int FaceType,
                                  ArgList Args, Cardinal ArgCount, 
				  Boolean Gadget)
{
    char **Normal;
    
    /* Ueberpruefe zunaechst den FaceType Parameter auf seinen Definitions-
     * bereich. Falls ausserhalb, wird immer XmLABELFACE_HAND angenommen...
     * Selbst schuld!
     */
    if ( (FaceType < XmLABELFACE_INFO) || (FaceType > XmLABELFACE_TIME))
        FaceType = XmLABELFACE_HAND;
    /* Jetzt muessen wir uns die passenden Pixmaps heraussuchen, um damit
     * das Label zu erzeugen.
     */
    switch ( FaceType ) {
        case XmLABELFACE_INFO:        Normal = InfoSign_xpm;        break;
	case XmLABELFACE_QUESTION:    Normal = QuestionMark_xpm;    break;
	case XmLABELFACE_EXCLAMATION: Normal = ExclamationMark_xpm; break;
	case XmLABELFACE_HAND:        Normal = HandSign_xpm;        break;
	case XmLABELFACE_TIME:        Normal = TimeSign_xpm;        break;
    }
#ifdef USEGADGETS
    if ( Gadget )
	return XmCreatePixmapLabelGadget(Parent, Name, 
				         Normal, NULL, Args, ArgCount);
#endif
    return XmCreatePixmapLabel(Parent, Name, 
                               Normal, NULL, Args, ArgCount);
} /* CreateStandardLabel */

/* ---------------------------------------------------------------------------
 * Zum Abschluss noch einmal zwei Routinen, um Label-Widgets oder -Gadgets
 * zu erzeugen, die eines der vorgegebenen Bilder tragen. Damit waeren wir
 * dann auch endlich am Ende dieses Programm-Moduls angekommen -- es ist
 * ja doch laenger geworden, als ich eigentlich dachte. Aber was soll's.
 * Universalitaet fordert halt ihren Tribut.
 */
Widget XmCreateStandardLabel(Widget Parent, String Name, int FaceType,
                             ArgList Args, Cardinal ArgCount)
{
    return CreateStandardLabel(Parent, Name, FaceType, 
                               Args, ArgCount,  False);
} /* XmCreateStandardLabel */

#ifdef USEGADGETS
Widget XmCreateStandardLabelGadget(Widget Parent, String Name, int FaceType,
                             ArgList Args, Cardinal ArgCount)
{
    return CreateStandardLabel(Parent, Name, FaceType, 
                               Args, ArgCount,  True);
} /* XmCreateStandardLabelGadget */
#endif

/* Ende von pushbuttons.c */

/* ----------------------------------------------------------------------
 * Da fange ich doch gleich mal an, eine neue Funktion 
 * hintendranzufuegen. Es handelt sich um eine kleine Abwandlung von
 * CreateStandardPushButton, die aber keine Schaltflaeche neu erzeugt, 
 * sondern die Bilder fuer eine bereits existierende eintraegt.
 * Hmm, diese Aenderung macht mich inkompatibel mit der Orginal ButtonFace
 * library. Aber ich haette gerne die Funktionalitaet, um die 
 * Tristigkeit des FileSelectionDialog etwas aufzubessern.
 * Thomas Harrer.
 */

void 
XmAddStandardButtonPixmaps (Widget PushButton, int FaceType)
{
    char** Normal;
    char** Armed;
    char** Insensitive;
    
    /* Ueberpruefe zunaechst den FaceType Parameter auf seinen Definitions-
     * bereich. Falls ausserhalb, wird immer XmPUSHBUTTON_NONE angenommen...
     * Selbst schuld!
     */
    if ( (FaceType < XmPUSHBUTTON_NONE) || (FaceType > XmPUSHBUTTON_HELP))
        FaceType = XmPUSHBUTTON_NONE;
    /* Jetzt muessen wir uns die passenden Pixmaps heraussuchen, um damit
     * die Schaltflaeche zu erzeugen.
     */
    Insensitive = NULL;
    switch ( FaceType ) {
        case XmPUSHBUTTON_NONE:
            Normal      = EmptyPB_xpm; /* Hier gibt es nur ein einziges */
            Armed       = NULL;        /* Bildchen, die restlichen      */
            break;                     /* werden automatisch erzeugt.   */
        case XmPUSHBUTTON_OK:
            Normal      = OkayPB_xpm;
            Armed       = OkayPBa_xpm;
            break;
        case XmPUSHBUTTON_CANCEL:
            Normal      = CancelPB_xpm;
            Armed       = CancelPBa_xpm;
            break;
        case XmPUSHBUTTON_ABORT:
            Normal      = AbortPB_xpm;
            Armed       = AbortPBa_xpm;
            break;
        case XmPUSHBUTTON_IGNORE:
            Normal      = IgnorePB_xpm;
            Armed       = IgnorePBa_xpm;
            break;
        case XmPUSHBUTTON_YES:
            Normal      = YesPB_xpm;
            Armed       = YesPBa_xpm;
            break;
        case XmPUSHBUTTON_NO:
            Normal      = NoPB_xpm;
            Armed       = NoPBa_xpm;
            break;
        case XmPUSHBUTTON_RETRY:
            Normal      = RetryPB_xpm;
            Armed       = RetryPBa_xpm;
            break;
        case XmPUSHBUTTON_HELP:
            Normal      = HelpPB_xpm; 
            Armed       = HelpPBa_xpm; 
            break;
    }
    CreateButtonPixmaps(PushButton, Normal, Armed, Insensitive);
}


