/*
 * msgbox.c -- Die allseits beliebte Message-Box fuer Standardfragen
 *             usw. Einfache Rueckfragen des Programms lassen sich so
 *             mit einem einzigen Befehl namens "MessageBox()" er-
 *	       ledigen, ohne den sonst erforderlichen Aufwand (Schalt-
 *	       flaechen einfuegen, Layout usw.)
 *
 * Version 1.02 vom 03.05.1994
 * Aktueller Stand dieses Moduls:
 *   03.05.1994	    Die Einzeilen-Nachrichten-Box ist implementiert.
 * 
 * (c) 1994 Harald Albrecht
 * Institut fuer Geometrie und Praktische Mathematik
 * albrecht@igpm.rwth-aachen.de
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program (see the file COPYING for more details);
 * if not, write to the Free Software Foundation, Inc., 675 Mass Ave, 
 * Cambridge, MA 02139, USA.
 *
 */

#include <Xm/Xm.h>
#include <Xm/Protocols.h>
#include <Xm/Form.h>
#include <Xm/DialogS.h>
#include <Xm/Separator.h>
#include <Xm/Label.h>
#include <Xm/Frame.h>


#include "msgbox.h"
#include "pushbuttons.h"
#include "Center.h"

#include <stdio.h>

/* --------------------------------------------------------------------
 * Dieser Callback wird immer dann aktiviert, sobald jemand versucht, 
 * mit Hilfe des Fenstermanagers so einfach 'mal eben die Dialogbox zu
 * schliessen. Der nette Trick an der Sache ist, dass dieser Protokoll-
 * Callback mit einem ClientData-Zeiger registriert wird, der das-
 * jenige Widget darstellt, welches ausgeloest werden soll. Hat der
 * Dialog aber keine Schaltflaeche, der sich die Close-Funktion des
 * Window-Managers zuweisen laesst, so wird dieses Ansinnen des Be-
 * nutzers einfach ignoriert. Leider ist es bislang noch nicht moeg-
 * lich, den Close-Eintrag im Systemmenue des Dialogfensters in einem
 * solchen Fall zu dimmen, um so anzuzeigen, dass dieser Menue-Eintrag
 * momentan nicht verfuegbar ist.
 */
static void DialogCleanupCB(Widget widget, XtPointer ClientData, 
                            XtPointer CallData)
{
    if ( ClientData != NULL ) {
	/* Loese den entsprechenden Button aus, der die Funktion des
	 * Close-Menue-Eintrags im Fenstermenue wiederspiegelt. Aber
	 * auch nur dann, wenn ein Button diese Funktion uebernimmt.
	 * Ausserdem setzt diese Routine voraus, dass das betroffene
	 * Widget eine Action-Prozedur names ArmAndActivate() be-
	 * sitzt. Wir uebergeben dieser Routine im uebrigen einen
	 * NULL-Event!!! Das sollte aber nicht weiter stoeren, da
	 * die Actionroutine in diesem Fall sowieso keine Annahmen
	 * ueber den Event treffen sollte.
	 */
	XtCallActionProc((Widget) ClientData, "ArmAndActivate", 
	                 NULL, NULL, 0);
    }
} /* DialogCleanupCB */


/* --------------------------------------------------------------------
 * Mittels dieses Timer-Callbacks kann immer dann, wenn's eben unge-
 * faehrlich ist, der Tastaturfokus auf eine bestimmte gewuenschte
 * Taste innerhalb der Messagebox verschoben werden. Der Wecker fuer
 * diesen Callback wird immer dann aufgezogen, wenn der betroffene
 * Dialog zum ersten Mal den Tastaturfokus bekommt. Welches Widget den
 * Fokus erhalten soll, legt ClientData fest. Dieser Zeiger enhaelt
 * dasjenige Widget, auf dem sich dann der Tastaturfokus niederlassen
 * soll.
 */
typedef struct _FocusData FocusData;
struct _FocusData {
    Widget    Form;
    XtPointer ClientData;
    Widget    FocusWidget;
}; /* struct _FocusData */
static void FocusCallback(Widget w, XtPointer ClientData, 
                                    XtPointer CallData);

static void TimerCallback(XtPointer ClientData, XtIntervalId *TimerID)
{
/*    fprintf(stderr, "Callback: %08X\n", ((FocusData *) ClientData)->Form);*/
    XtRemoveCallback(((FocusData *) ClientData)->Form, 
                     XmNfocusCallback, 
                     (XtCallbackProc) FocusCallback, 
		     ((FocusData *) ClientData)->ClientData);
    XmProcessTraversal((Widget) ((FocusData *) ClientData)->FocusWidget, 
                       XmTRAVERSE_CURRENT);
    free((char *) ClientData);
} /* TimerCallback */

/* --------------------------------------------------------------------
 * Dieser Callback sorgt dafuer, dass, wenn der Dialog erstmalig den
 * Fokus erhaelt, ein Wecker aufgezogen wird. Sobald der Wecker 
 * klingelt, kann dann der Tastaturfokus auf ein in ClientData vorge-
 * gebenes Widget gesetzt werden. Da dieses aber nur einmal passieren
 * darf, schmeisst sich der Callback dann auch gleich selbst wieder
 * heraus.
 */
static void FocusCallback(Widget w, XtPointer ClientData, 
                                    XtPointer CallData)
{
    FocusData *pFocusData;
    
    pFocusData = (FocusData *) malloc(sizeof(FocusData));
    if ( pFocusData ) {
	pFocusData->Form        = w;
	pFocusData->ClientData  = ClientData;
	pFocusData->FocusWidget = (Widget) ClientData;
	XtAppAddTimeOut(XtWidgetToApplicationContext(w), 
                        0, (XtTimerCallbackProc) TimerCallback, 
		        (XtPointer) pFocusData);
    }
    	    
/*    XtAppAddTimeOut(XtWidgetToApplicationContext((Widget) ClientData), 0,
                    (XtTimerCallbackProc) TimerCallback, ClientData);
    fprintf(stderr, "Callback: %08X\n", w);
    XtRemoveCallback(w, XmNfocusCallback, 
                     (XtCallbackProc) FocusCallback, ClientData); */
} /* FocusCallback */


/* --------------------------------------------------------------------
 * Ermittele anhand einer Kennung das entsprechende Widget (hier nur
 * Schaltflaechen) einer Mitteilungsbox. Existiert das entsprechende
 * Kind nicht, so liefert die Funktion NULL zurueck. Diese Funktion
 * wird alleine schon deshalb benoetigt, um die erforderlichen Call-
 * backs fuer die Schaltflaechen eines Dialogs installieren zu
 * koennen.
 * 
 * Parameter:
 *   w		    Zeiger auf die MsgBox, deren Kind wir suchen.
 *   WhichChild	    Eine der XmPUSHBUTTON_xxx-Konstanten, die naeher
 *		    spezifiziert, welches Kind gesucht werden soll.
 *
 * Ergebnis:
 *   Widget-Kennung der zu suchenden Schaltflaeche oder NULL, wenn
 *   es nichts zu finden gab.
 */
Widget XmMsgBoxGetChild(Widget w, int WhichChild)
{
    char *Name;
    
    switch ( WhichChild ) {
	case XmPUSHBUTTON_OK:	    Name = "*center.ok"; break;
	case XmPUSHBUTTON_CANCEL:   Name = "*center.cancel"; break;
	case XmPUSHBUTTON_ABORT:    Name = "*center.abort"; break;
	case XmPUSHBUTTON_RETRY:    Name = "*center.retry"; break;
	case XmPUSHBUTTON_IGNORE:   Name = "*center.ignore"; break;
	case XmPUSHBUTTON_YES:	    Name = "*center.yes"; break;
	case XmPUSHBUTTON_NO:	    Name = "*center.no"; break;
	case XmPUSHBUTTON_HELP:	    Name = "*center.help"; break;
	default:		    return NULL;
    }
    return XtNameToWidget(w, Name);
} /* XmMsgBoxGetChild */

/* --------------------------------------------------------------------
 * Erzeuge einen Mitteilungsdialog mit einen Symbol, Text und den
 * Schaltflaechen, ueber die der Benutzer dem Programm kundtun kann, 
 * was er denn von der Mitteilung haelt... Nach dem Aufruf dieser
 * Funktion hat's man dann aber auch...so richtig lebensfaehig ist
 * der Dialog in diesem Moment noch nicht. Man muss erst mit
 * XmMsgBoxGetChild() noch die passenden Callbacks fuer die Schalt-
 * flaechen installieren, ehe es so richtig losgehen kann.
 * 
 * Parameter:
 *   Parameter	    Dasjenige Eltern-Widget, zu dem dieser Dialog ge-
 *		    hoeren soll.
 *   Name	    Name, auf den der Dialog hoert.
 *   Caption	    Dieser Text erscheint in der Titelzeile (also in
 *		    der Dekoration) des Mitteilungsdialogs.
 *   Text	    Hier ist der Text anzugeben, der als Mitteilung im
 *		    Dialog erscheint.
 *   Flags	    Hierueber wird festgelegt, welche Schaltflaechen
 *		    und welches Symbol in der Messagebox angezeigt
 *		    werden.
 *   Args	    wie schon so oft erwaehnt...
 *   ArgCount	    dito.
 *
 * Ergebnis:
 *   Widgetkennung des Dialogs. Dieses ist in guter (?!) Motif-Tradi-
 *   tion eben nicht die Shell, in welcher der Dialog steckt, sondern
 *   das Dialogwidget selbst. In unserem Fall handelt es sich hierbei
 *   um ein Form-Widget. Damit laesst sich doch noch am bequemsten das
 *   Layout von Dialogboxen erzeugen...
 */
Widget XmCreateMsgBox(Widget Parent, char *Name, 
                      char *Caption, char *Text, 
		      int Flags, 
		      ArgList Args, Cardinal ArgCount)
{
    Atom     WM_Delete_Window;
    Arg      args[10];
    int      n, DefaultIndex;
    XmString TextString;
    Widget   DialogShell, Form, IconCenter, IconFrame, IconPicture, 
             TextFrame, TextLabel, Separator, Center, 
	     CancelWidget, Button, Childs[5];
    
    /* Zuerst einmal benoetigen wir natuerlich eine Shell, in die
     * hinein wir unseren Dialog verpflanzen koennen.
     */
    DialogShell = XmCreateDialogShell(Parent, Name, Args, ArgCount);
    TextString = XmStringCreate(Caption, XmSTRING_DEFAULT_CHARSET);

    /* Nun richten wir in die Shell, die ja alleine noch nicht sehr
     * nuetzlich ist, auch noch ein Form-Widget ein. Diese dient dann
     * als Container fuer die diversen Widgets.
     */
    n = 0;
    TextString = XmStringCreate(Caption, XmSTRING_DEFAULT_CHARSET);
    XtSetArg(args[n], XmNdialogStyle, XmDIALOG_FULL_APPLICATION_MODAL); n++;
    XtSetArg(args[n], XmNautoUnmanage, False); n++;
    XtSetArg(args[n], XmNnoResize, True); n++;
    XtSetArg(args[n], XmNdialogTitle, TextString); n++;
    Form = XtCreateWidget(Name, xmFormWidgetClass, DialogShell, args, n);
    XmStringFree(TextString);

    /* So. Jetzt geht die Arbeit aber erst so richtig los...
     * Hier sind jetzt die einzelnen Elemente der Messagebox nachein-
     * ander einzurichten. Dann ma los!
     */
    /* Zuerst richten wir das Sinnbild ein, das anzuzeigen ist. */
    IconCenter = XtVaCreateManagedWidget(
        "iconbox", xmCenterWidgetClass, Form, 
	XmNleftAttachment,    XmATTACH_FORM, 
	XmNleftOffset,	      8, 
	XmNtopAttachment,     XmATTACH_FORM, 
	XmNtopOffset,	      8, 
	XmNbottomOffset,      8, 
	XmNorientation,       XmVERTICAL, 
	XmNhorizontalBorder,  0, 
	XmNverticalBorder,    0, 
	XmNverticalSpacing,   0, 
	NULL);
    IconFrame = XtVaCreateManagedWidget(
	"iconframe", xmFrameWidgetClass, IconCenter, 
	XmNshadowType,	      XmSHADOW_ETCHED_IN, 
	NULL);
    IconPicture = XmCreateStandardLabel(IconFrame, "icon", 
                                        Flags &MB_ICONMASK, NULL, 0);
    XtManageChild(IconPicture);

    /* Nun kommt der auszugebende Text 'dran...
     * Hier benoetigen wir XmStringCreateLtoR, weil nur diese
     * Bibliotheksfunktion Ruecksicht auf \n nimmt und einen
     * Zeilenumbruch einfuegt.
     */
    TextFrame = XtVaCreateManagedWidget(
	"textframe", xmFrameWidgetClass, Form,
	XmNleftAttachment,    XmATTACH_WIDGET, 
	XmNleftWidget,        IconCenter, 
	XmNleftOffset,        8,   
	XmNrightAttachment,   XmATTACH_FORM, 
	XmNrightOffset,       8, 
	XmNleftOffset,        8, 
	XmNtopAttachment,     XmATTACH_FORM, 
	XmNtopOffset,	      8, 
	XmNbottomOffset,      8, 
	NULL);
    TextString = XmStringCreateLtoR(Text, XmSTRING_DEFAULT_CHARSET);
    TextLabel = XtVaCreateManagedWidget(
        "text", xmLabelWidgetClass, TextFrame,
	XmNlabelString,       TextString,  
	XmNalignment,         XmALIGNMENT_BEGINNING, 
	XmNmarginLeft,	      4, 
	XmNmarginRight,	      4, 
	NULL);
    XmStringFree(TextString);
    
    /* Jetzt kommt der untere Bereich an die Reihe. Hier tummeln
     * sich die Buttons...
     */
    Separator = XtVaCreateManagedWidget(
	"separator", xmSeparatorWidgetClass, Form, 
	XmNleftAttachment,    XmATTACH_FORM, 
	XmNrightAttachment,   XmATTACH_FORM, 
	XmNtopOffset,	      8, 
	XmNbottomOffset,      8, 
	XmNseparatorType,     XmSHADOW_ETCHED_IN,
	XmNorientation,       XmHORIZONTAL,
	XmNmargin,            0, 
	NULL);
    Center = XtVaCreateManagedWidget(
	"center", xmCenterWidgetClass, Form, 
	XmNleftAttachment,    XmATTACH_FORM, 
	XmNleftOffset,	      8, 
	XmNrightAttachment,   XmATTACH_FORM, 
	XmNrightOffset,	      8, 
	XmNbottomAttachment,  XmATTACH_FORM, 
	XmNbottomOffset,      8, 
	XmNorientation,       XmHORIZONTAL,
	XmNhorizontalBorder,  0, 
	XmNverticalBorder,    0, 
	XmNhorizontalSpacing, 16, 
	NULL);
	
    
    XtVaSetValues(IconCenter,
                  XmNbottomAttachment, XmATTACH_WIDGET, 
		  XmNbottomWidget,     Separator, 
		  NULL);
    XtVaSetValues(TextFrame,
                  XmNbottomAttachment, XmATTACH_WIDGET, 
		  XmNbottomWidget,     Separator, 
		  NULL);
    XtVaSetValues(Separator,
                  XmNbottomAttachment, XmATTACH_WIDGET, 
		  XmNbottomWidget,     Center, 
		  NULL);

    /* Nun kommt noch einmal so richtig Arbeit auf uns zu...
     * denn es muessen die angeforderten Schaltflaechen eingerichtet
     * werden.
     */
    CancelWidget = NULL; n = 0;
    switch ( Flags & MB_BUTTONMASK ) {
	case MB_OKCANCEL:
	    Button = XmCreateStandardPushButton(
		Center, "ok", XmPUSHBUTTON_OK, NULL, 0);
	    XtManageChild(Button); Childs[n++] = Button;
	    Button = XmCreateStandardPushButton(
		Center, "cancel", XmPUSHBUTTON_CANCEL, NULL, 0);
	    XtManageChild(Button); Childs[n++] = Button;
	    CancelWidget = Button;
	    break;
	case MB_YESNO:
	    Button = XmCreateStandardPushButton(
		Center, "yes", XmPUSHBUTTON_YES, NULL, 0);
	    XtManageChild(Button); Childs[n++] = Button;
	    Button = XmCreateStandardPushButton(
		Center, "no", XmPUSHBUTTON_NO, NULL, 0);
	    XtManageChild(Button); Childs[n++] = Button;
	    CancelWidget = Button;
	    break;
	case MB_YESNOCANCEL:
	    Button = XmCreateStandardPushButton(
		Center, "yes", XmPUSHBUTTON_YES, NULL, 0);
	    XtManageChild(Button); Childs[n++] = Button;
	    Button = XmCreateStandardPushButton(
		Center, "no", XmPUSHBUTTON_NO, NULL, 0);
	    XtManageChild(Button); Childs[n++] = Button;
	    Button = XmCreateStandardPushButton(
		Center, "cancel", XmPUSHBUTTON_CANCEL, NULL, 0);
	    XtManageChild(Button); Childs[n++] = Button;
	    CancelWidget = Button;
	    break;
	case MB_RETRYABORT:
	    Button = XmCreateStandardPushButton(
		Center, "retry", XmPUSHBUTTON_RETRY, NULL, 0);
	    XtManageChild(Button); Childs[n++] = Button;
	    Button = XmCreateStandardPushButton(
		Center, "abort", XmPUSHBUTTON_ABORT, NULL, 0);
	    XtManageChild(Button); Childs[n++] = Button;
	    CancelWidget = Button;
	    break;
	case MB_RETRYABORTIGNORE:
	    Button = XmCreateStandardPushButton(
		Center, "retry", XmPUSHBUTTON_RETRY, NULL, 0);
	    XtManageChild(Button); Childs[n++] = Button;
	    Button = XmCreateStandardPushButton(
		Center, "abort", XmPUSHBUTTON_ABORT, NULL, 0);
	    XtManageChild(Button); Childs[n++] = Button;
	    CancelWidget = Button;
	    Button = XmCreateStandardPushButton(
		Center, "ignore", XmPUSHBUTTON_IGNORE, NULL, 0);
	    XtManageChild(Button); Childs[n++] = Button;
	    break;
	case MB_OK:
	default:
	    Button = XmCreateStandardPushButton(
		Center, "ok", XmPUSHBUTTON_OK, NULL, 0);
	    XtManageChild(Button); Childs[n++] = Button;
    }
    if ( Flags & MB_HELP ) {
	Button = XmCreateStandardPushButton(
	    Center, "help", XmPUSHBUTTON_HELP, NULL, 0);
	XtManageChild(Button); Childs[n++] = Button;
    }

    /* Damit das ganze auch wirklich benutzerfreundlich wird, muessen
     * wir noch die Moeglichkeit abfangen, dass der Anwender im System-
     * menue des Dialogs "Close" anklickt. In einem solchen Fall darf
     * das Fenster nicht einfach geloescht werden -- dies duerfen erst
     * wir hier selbst entscheiden.
     */
    WM_Delete_Window = XmInternAtom(XtDisplay(DialogShell), 
                                    "WM_DELETE_WINDOW", FALSE);
    XtVaSetValues(DialogShell, XmNdeleteResponse, XmDO_NOTHING, NULL);
    XmAddWMProtocolCallback(DialogShell, WM_Delete_Window, 
                            DialogCleanupCB, (caddr_t) CancelWidget);

    if ( CancelWidget )
	XtVaSetValues(Form, XmNcancelButton, CancelWidget, NULL);

    /* ...und weil das noch nicht benutzerfreundlich genug ist,
     * bieten wir hier auch noch die Moeglichkeit, den Tastaturfokus
     * gezielt auf einen Button zu setzen.
     */
    DefaultIndex = (Flags & MB_DEFAULTMASK) >> MB_DEFAULTSHIFT;
    if ( DefaultIndex >= n ) DefaultIndex = 0;
/*    fprintf(stderr, "Form: %08X\n", Form);*/
    XtAddCallback(Form, XmNfocusCallback, 
                  (XtCallbackProc) FocusCallback, 
		  (XtPointer) Childs[DefaultIndex]);
    XtVaSetValues(Form, XmNdefaultButton, Childs[DefaultIndex], NULL);
    
    return Form;
} /* XmCreateMsgBox */


/* --------------------------------------------------------------------
 * Der folgende Callback behandelt die Situation, dass der Anwender
 * eine der Schaltflaechen in einer MessageBox anklickte. Dann wird
 * hierin der momentan laufenden Nachrichtenschleife mitgeteilt, dass
 * jetzt mit der MsgBox Schluss ist und die Box geschlossen werden
 * kann. Dazu erhaelt dieser Callback in pClientData Informationen, wie
 * er die momentan laufende Mitteilungsbox schliessen kann.
 * Die direkt folgende Datenstruktur uebertraegt diese Informationen
 * zu den Schaltflaechen.
 */
typedef struct _ButtonInfo ButtonInfo;
struct _ButtonInfo {
    Widget  MsgBox;
    Boolean *ContinueFlagPtr;
    int     *ReturnCodePtr;
    int     ReturnCode;
}; /* struct _ButtonInfo */

static void MsgBoxButtonActivateCB(Widget w, XtPointer ClientData, 
				             XtPointer CallData)
{
    ButtonInfo *pData;
    
    pData = (ButtonInfo *) ClientData;
    if ( pData->ReturnCode != XmPUSHBUTTON_HELP ) {
	/* 
	 * Da lediglich einer der "normalen" Buttons gedrueckt wurde,
	 * den Dialog beenden...
	 */
	Pixmap  NormalPixmap, ArmedPixmap, InsensitivePixmap;
        XtVaGetValues(w,
                  XmNlabelPixmap,            &NormalPixmap,
		  XmNarmPixmap,		     &ArmedPixmap, 
                  XmNlabelInsensitivePixmap, &InsensitivePixmap,
                  NULL);
	*(pData->ReturnCodePtr)   = pData->ReturnCode;
	*(pData->ContinueFlagPtr) = False;
    } else {
	/* 
	 * HILFE!!! Der Anwender moechte Hilfe. In diesem Fall rufe
	 * den Hilfe-Callback der Nachrichten-Box auf.
	 */
	if ( XtHasCallbacks(w, XmNhelpCallback) == XtCallbackHasSome ) {
	    XmAnyCallbackStruct CBStruct;
	    
	    CBStruct.reason = XmCR_HELP;
	    CBStruct.event  = ((XmPushButtonCallbackStruct *) CallData)
	                        ->event;
	    XtCallCallbacks(pData->MsgBox, XmNhelpCallback, 
	                    (XtPointer) &CBStruct);
	}
    }
} /* MsgBoxButtonActivateCB */

/* ---------------------------------------------------------------------
 * Bearbeite jeweils einen einzelnen Event und kehre dann schoen
 * brav zurueck. Wehe dem, der hier boese Tricks einbaut! Als Parameter
 * bekommt diese Routine den Applikations-Kontext uebergeben, innerhalb
 * dessen eintreffende Ereignisse bearbeitet werden sollen. Der Rueck-
 * gabewert signalisiert dann, ob die Nachrichtenverarbeitung ueberhaupt
 * noch fortgesetzt werden soll (ok = True, beenden = False).
 */
static Boolean DefaultLoopProc(XtAppContext AppContext)
{
    XEvent       NextEvent;

    XtAppNextEvent(AppContext, &NextEvent);
    XtDispatchEvent(&NextEvent);
    return True;
} /* DefaultLoopProc */

/* ---------------------------------------------------------------------
 * Nun kommt das eigentliche Schmuckstueck dieses Programm-Moduls. Hier-
 * mit laesst sich in einem Funktionsaufruf eine Nachrichten-Box er-
 * stellen, anzeigen, verwaltung und bei der Rueckkehr vom Funktions-
 * aufruf erhaelt der Aufrufer die Kennung derjenigen Schaltflaeche zu-
 * rueck, die der Benutzer anklickte. Fein Sache, oder?!
 * 
 * Parameter:
 *   Parent		Dasjenige Widget, unterhalb dessen diese Nach-
 *			richten-Box angesiedelt sein soll.
 *   Caption		Der in der Titelzeile des Fensters erscheinende
 *			Text.
 *   Text		Die eigentliche Nachricht.
 *   Flags		Genauere Angaben darueber, welche Schaltflachen, 
 *			welches Symbol und welches Verhalten verlangt
 *			werden.
 *   ExternalEventLoop	Hier kann der Aufrufer ggf. eine eigene
 *			Funktion angeben, die die eingehenen Ereignisse
 *			verarbeiten und weiterleiten kann.
 * 
 * Ergebnis:
 *   Die Kennung der ausgewaehlten Schaltflaeche (d.h. eine der vielen
 *   XmPUSHBUTTON_xxx-Werte).
 *
 * Ach ja, ...
 * diese Routine dient nur der Bequemlichkeit, so dass der Benutzer nur
 * noch einen einzigen Befehl benoetigt, um eine Message-Box auf den
 * Bildschirm zu bringen.
 * 
 * Hmmm...das folgende Makro ist in logischer Schlussfolge natuerlich
 * nur aus der puren Tip-Faulheit heraus entstanden...
 * Es initialisiert jeweils fuer eine einzelne Schaltflaeche die dem
 * Ausloese-Callback uebergebenen Daten. Ausserdem wird zugleich auch
 * noch der Callback registriert.
 */
#define SETINFO(RetCode)					    \
    {								    \
	XtAddCallback(Button, XmNactivateCallback,		    \
	              (XtCallbackProc) &MsgBoxButtonActivateCB,	    \
		      (XtPointer) &(Infos[n]));			    \
        Infos[n].MsgBox          = MsgBox;			    \
	Infos[n].ContinueFlagPtr = &ContinueFlag;		    \
	Infos[n].ReturnCodePtr   = &ReturnCode;			    \
	Infos[n].ReturnCode      = RetCode;			    \
	n++;							    \
    }

int XmMessageBox(Widget Parent, char *Caption, char *Text, 
                 int Flags, XmEventLoopProc *ExternalEventLoop)
{
    Widget       MsgBox, Button;
    ButtonInfo   Infos[10];
    int          n = 0;
    Boolean      ContinueFlag;
    int          ReturnCode;
    XtAppContext AppContext;
    
    /* Zuerst einmal die benoetigten Widgets einrichten... */
    MsgBox = XmCreateMsgBox(Parent, "msgbox", Caption, Text, 
                            Flags, NULL, 0);
    if ( !MsgBox ) return 0;
    
    /* Jetzt sind die Schaltflaechen an der Reihe... */
    Button = XmMsgBoxGetChild(MsgBox, XmPUSHBUTTON_OK);
    if ( Button ) SETINFO(XmPUSHBUTTON_OK);    
    Button = XmMsgBoxGetChild(MsgBox, XmPUSHBUTTON_CANCEL);
    if ( Button ) SETINFO(XmPUSHBUTTON_CANCEL);    
    Button = XmMsgBoxGetChild(MsgBox, XmPUSHBUTTON_ABORT);
    if ( Button ) SETINFO(XmPUSHBUTTON_ABORT);    
    Button = XmMsgBoxGetChild(MsgBox, XmPUSHBUTTON_RETRY);
    if ( Button ) SETINFO(XmPUSHBUTTON_RETRY);    
    Button = XmMsgBoxGetChild(MsgBox, XmPUSHBUTTON_YES);
    if ( Button ) SETINFO(XmPUSHBUTTON_YES);    
    Button = XmMsgBoxGetChild(MsgBox, XmPUSHBUTTON_NO);
    if ( Button ) SETINFO(XmPUSHBUTTON_NO);    
    Button = XmMsgBoxGetChild(MsgBox, XmPUSHBUTTON_HELP);
    if ( Button ) SETINFO(XmPUSHBUTTON_HELP);    
    /*
     * Initialisiere nun noch die benoetigten Variablen und dann
     * kann der Dialog auf dem Bildschirm erscheinen...
     */
    ContinueFlag = True;
    ReturnCode = 0;
    AppContext = XtWidgetToApplicationContext(MsgBox);
    if ( ExternalEventLoop == NULL )
	ExternalEventLoop = (XmEventLoopProc *) &DefaultLoopProc;
    XtManageChild(MsgBox);
    /* 
     * Nachdem nun alles brav vorbereitet ist, koennen wir uns
     * an die Verarbeitung der eintreffenden Ereignisse und Nach-
     * richten machen. Diese Schleife wird nur dann abgebrochen, 
     * wenn entweder der externe Nachrichtenverteiler dieses
     * explizit signalisiert oder aber eine Schaltflaeche ange-
     * klickt wurde.
     */
    while ( ContinueFlag ) {
        if ( !(*ExternalEventLoop)(AppContext) ) break;
    }
    /*
     * Fertig. Raeume nun alles wieder schoen weg...
     */
    XtUnmanageChild(MsgBox);
    XtDestroyWidget(XtParent(MsgBox));
    return ReturnCode;
} /* XmMessageBox */


/* Ende von mbox.c */
