/* $Id: fsm.c,v 0.2 1997/11/16 16:48:34 rsmit06 Exp $
 *
 * This file implements the Finite State Machine library (libfsm).
 *
 * Copyright (C) 1997  R.F. Smith
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Log: fsm.c,v $
 * Revision 0.2  1997/11/16 16:48:34  rsmit06
 * Updated fsm_init to reflect schanges in fsm_entry structure.
 * Added some more parameter checking.
 *
 * Revision 0.1  1997/11/16 15:04:50  rsmit06
 * Initial code
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "fsm.h"

/* reports a fatal error and exits */
static void _fsm_error(char *msg);

static char *mem_err = "unable to allocate memory for FSM";
static char *par_err = "parameter error";

fsm_t 
fsm_init(unsigned num, struct fsm_entry *entries, void *fsm_data)
{
  fsm_t new_fsm = 0;
  int i;
  unsigned count;

  /* check for parameter errors */
  if (entries == 0) {
    _fsm_error(par_err);
  }

  /* allocate the structure */
  new_fsm = calloc(1, sizeof(struct _fsm));
  if (new_fsm == 0) {
    _fsm_error(mem_err);
  }

  /* set external data pointer */
  new_fsm->data = fsm_data;

  /* scan for the range of states and inputs */
  new_fsm->min_state = (unsigned)-1;
  new_fsm->min_input = (unsigned)-1;
  for (i=0; i<num; i++) {
    if (entries[i].state < new_fsm->min_state)
      new_fsm->min_state = entries[i].state;
    if (entries[i].state > new_fsm->max_state)
      new_fsm->max_state = entries[i].state;
    if (entries[i].input < new_fsm->min_input)
      new_fsm->min_input = entries[i].input;
    if (entries[i].input > new_fsm->max_input)
      new_fsm->max_input = entries[i].input;
  }
  new_fsm->num_states = new_fsm->max_state - new_fsm->min_state;
  new_fsm->num_inputs = new_fsm->max_input - new_fsm->min_input;
  count = new_fsm->num_states * new_fsm->num_inputs;

  /* allocate a table, setting all elements to 0 */
  new_fsm->table = (struct fsm_trans *) calloc(count, 
                                               sizeof(struct fsm_trans));
  if (new_fsm->table == 0) {
    free(new_fsm);
    new_fsm = 0;
    _fsm_error(mem_err);
  }

  /* fill the table */
  for (i=0; i<num; i++) {
    struct fsm_entry *pentry = entries + i;
    unsigned offset = new_fsm->num_inputs * pentry->state + pentry->input;
    struct fsm_trans *ptrans = new_fsm->table + offset;

    ptrans->new_state = pentry->new_state - new_fsm->min_state;
    ptrans->action = pentry->action;
  }

  /* return the completed FSM */
  return new_fsm;
}

int 
fsm_input(fsm_t fsm, unsigned input)
{
  if (fsm == 0) {
    _fsm_error(par_err);
  }

  if (input > fsm->max_input || input < fsm->min_input) {
    _fsm_error(par_err);
  }

  if (fsm->input_available == 0) {
    fsm->input_available = 1;
    fsm->input_offset = input - fsm->min_input;
  } else {
    return 1; /* previous input hasn't been processed yet */
  }
  return 0; /* OK */
}

int 
fsm_run(fsm_t fsm)
{
  struct fsm_trans *ptrans;
  unsigned offset;

  /* check for parameter errors. */
  if (fsm == 0) {
    _fsm_error(par_err);
  }

  /* check for input */
  if (fsm->input_available == 0) {
    return 1; /* no input available */
  }

  /* get pointer to transition */
  offset = fsm->state_offset * fsm->num_inputs 
           + fsm->input_offset; 
  ptrans = fsm->table+offset;

  /* set new state */
  fsm->state_offset = ptrans->new_state;

  /* carry out action */
  if (ptrans->action) {
    (*ptrans->action)(fsm->data);
  }
  
  /* reset input */
  fsm->input_available = 0;

  return 0; /* OK */
}

void 
fsm_destroy(fsm_t fsm)
{
  /* check for parameter errors. */
  if (fsm == 0) {
    _fsm_error(par_err);
  }

  free(fsm->table);
  free(fsm);
}


static void 
_fsm_error(char *msg)
{
  /* print error message on STDERR */
  fprintf(stderr, "libfsm error: %s!\n", msg);
  /* then exit the program */
  exit(1);
}
