/*
  Library for reading configuration files
 */

#include <stdlib.h>
#include <errno.h>
#include <stdarg.h>
#include <string.h>
#include <stdio.h>

#include "cpriv.h"

static int config_write_GUESS(config_t c, FILE *fp);
static int config_write_LINEVAL(config_t c, FILE *fp);
static int config_write_CAP(config_t c, FILE *fp);
static int config_write_INI(config_t c, FILE *fp);

int config_write(config_t writeconf, char *filename, int style,
                char *keyvalsep)
{
  int i;
  FILE *fp;
  if(filename) {
    if(writeconf->filename)
      free(writeconf->filename);
    writeconf->filename = xstrdup(filename);
  }
  if(keyvalsep) {
    if(writeconf->kvsep)
      free(writeconf->kvsep);
    writeconf->kvsep = xstrdup(keyvalsep);
  }
  if(style != CS_GUESS)
    writeconf->style = style;
  config_debug("Style set to %d from %d\n", style, writeconf->style);

  fp = fopen(writeconf->filename,"w");
  if(!fp)
    return CRET_NOTFOUND;

  switch(writeconf->style) {
  case CS_GUESS:
    i = config_write_GUESS(writeconf, fp);
    break;
  case CS_LINEVAL:
    i = config_write_LINEVAL(writeconf, fp);
    break;
  case CS_CAP:
    i = config_write_CAP(writeconf, fp);
    break;
  case CS_INI:
    i = config_write_INI(writeconf, fp);
    break;
  default:
    i = CRET_INVALID;
    config_debug("Invalid style\n");
  }
  fclose(fp);
  return i;
}


static int config_write_GUESS(config_t c, FILE *fp)
{
#if 0 /* This is total junk right now */
  hash_t h, i, j;
  fputs("GROUP\n", fp);
  for(h = c->groups; h; h = h->next) {
    fprintf(fp,"%s\n",h->key?h->key:"NULL");
    for(i = h->value; i; i = i->next) {
      fputs("            KEY\n", fp);
      fprintf(fp, "%s\n", i->key?i->key:"NULL");
      for(j = i->value; j; j = j->next) {
        fputs("                    GROUP        VALUE\n", fp);
        fprintf(fp, "%s %s\n",
                j->key?j->key:"NULL", (char *)j->value?(char *)j->value:"NULL");
      }
    }
  }
#endif
  return CRET_INVALID;
}
  

static int config_write_LINEVAL(config_t c, FILE *fp)
{
  hash_t h, i, j;
  struct conf_val *v;
  int counter;
  for(h = c->groups; h; h = h->next) {
    if(h->key)
      fprintf(fp,"##Group %s\n",h->key);
    for(i = h->value; i; i = i->next) {
      for(j = i->value; j && j->key; j = j->next); /* Find the NULL key */
      if(!j)
	return CRET_NOTFOUND;
      if(j->value) {
	v = j->value;
	if(v->type != COMMENT_T && v->type < LAST_T) /* Don't print
							out the junk
							for comments */
	  fprintf(fp, "%s%s", i->key?i->key:"NULL", c->kvsep?c->kvsep:" ");
	switch(v->type) {
	case CHARPTR_T:
	  fprintf(fp, "%s\n", v->u.charptr_val); break;
	case COMMENT_T:
	  fprintf(fp, "#%s\n", v->u.charptr_val); break;
	case INT_T:
	  fprintf(fp, "%d\n", v->u.int_val); break;
	case BOOLEAN_T:
	  fprintf(fp, "%s\n", v->u.boolean_val?"true":"false"); break;
	case CHARPTRARRAY_T:
	  if(v->count > 0) {
	    fprintf(fp, "%s\n", v->u.charptrarray_val[0]);
	    for(counter = 1; counter < v->count; counter++)
	      fprintf(fp, "%s%s%s\n",i->key?i->key:"NULL",
		      c->kvsep?c->kvsep:" ", v->u.charptrarray_val[counter]);
	  }
	  break;
	default:
	  fprintf(fp, "\"Unknown type %d\"\n", v->type);
	}
      }
    }
  }
  return CRET_SUCCESS;
}

static int config_write_CAP(config_t c, FILE *fp)
{
  hash_t h, i, j;
  struct conf_val *v;
  int counter, inkey=0;
  for(h = c->groups; h; h = h->next) {
    if(h->key)
      fprintf(fp,"##Group %s\n",h->key);
    for(i = h->value; i; i = i->next) {
      inkey = 0;
      for(j = i->value; j; j = j->next){
	if(j->value) {
	  v = j->value;
	  switch(v->type) {
	  case CHARPTR_T:
	    if(inkey == 0) {
	      fprintf(fp, "%s:\\\n", i->key); inkey = 1;
	    }
	    fprintf(fp, "\t:%s=%s:%s\n", j->key, v->u.charptr_val,
		    j->next?"\\":""); break;
	  case COMMENT_T:
	    fprintf(fp, "#%s\n", v->u.charptr_val); break;
	  case INT_T:
	    if(inkey == 0) {
	      fprintf(fp, "%s:\\\n", i->key); inkey = 1;
	    }
	    fprintf(fp, "\t:%s#%d:%s\n", j->key, v->u.int_val,
		    j->next?"\\":""); break;
	  case BOOLEAN_T:
	    if(inkey == 0) {
	      fprintf(fp, "%s:\\\n", i->key); inkey = 1;
	    }
	    if(v->u.boolean_val) fprintf(fp, "\t:%s:%s\n", j->key,
					 j->next?"\\":""); break;
	  case CHARPTRARRAY_T:
	    if(inkey == 0) {
	      fprintf(fp, "%s:\\\n", i->key); inkey = 1;
	    }
	    if(v->count > 0) {
	      for(counter = 0; counter < (v->count - 1); counter++)
		fprintf(fp, "\t:%s=%s:\\\n",i->key?i->key:"NULL",
			v->u.charptrarray_val[counter]);
	      fprintf(fp, "\t:%s=%s:%s\n",i->key?i->key:"NULL",
		      v->u.charptrarray_val[counter+1],
		      j->next?"\\":"");
	    }
	    break;
	  default:
	    fprintf(fp, "#\"Unknown type %d\"\n", v->type);
	  }
	}
      }
    }
  }
  return CRET_SUCCESS;
}


static int config_write_INI(config_t c, FILE *fp)
{
  hash_t h, i, j;
  struct conf_val *v;
  int counter;
  for(h = c->groups; h; h = h->next) {
    if(h->key)
      fprintf(fp,"[%s]\n",h->key);
    fflush(fp);
    for(i = h->value; i; i = i->next) {
      for(j = i->value; j && j->key; j = j->next); /* Find the NULL key */
      if(!j)
	return CRET_NOTFOUND;
      if(j->value) {
	v = j->value;
	if(v->type != COMMENT_T && v->type < LAST_T) /* Don't print
							out the junk
							for comments */
	  fprintf(fp, "%s%s%s", c->linehead?c->linehead:"", 
		  i->key?i->key:"NULL", c->kvsep?c->kvsep:" ");
	switch(v->type) {
	case CHARPTR_T:
	  fprintf(fp, "%s\n", v->u.charptr_val); break;
	case COMMENT_T:
	  fprintf(fp, ";%s\n", v->u.charptr_val); break;
	case INT_T:
	  fprintf(fp, "%d\n", v->u.int_val); break;
	case BOOLEAN_T:
	  fprintf(fp, "%s\n", v->u.boolean_val?"true":"false"); break;
	case CHARPTRARRAY_T:
	  if(v->count > 0) {
	    fprintf(fp, "%s\n", v->u.charptrarray_val[0]);
	    for(counter = 1; counter < v->count; counter++)
	      fprintf(fp, "%s%s%s%s\n", c->linehead?c->linehead:"",
		      i->key?i->key:"NULL",
		      c->kvsep?c->kvsep:" ", v->u.charptrarray_val[counter]);
	  }
	  break;
	default:
	  fprintf(fp, "\"Unknown type %d\"\n", v->type);
	}
      }
    }
  }
  return CRET_SUCCESS;
}
