/*
  Library for reading configuration files
 */

#include <stdlib.h>
#include <errno.h>
#include <stdarg.h>
#include <string.h>
#include <stdio.h>
#include <limits.h>
#include <regex.h>
#include <ctype.h>
#include <assert.h>

#include "cpriv.h"

static int config_read_GUESS(config_t c, FILE *infile);
static int config_read_LINEVAL(config_t c, FILE *infile);
static int config_read_CAP(config_t c, FILE *infile);
static int config_read_INI(config_t c, FILE *infile);

int config_read(config_t *readconf, char *filename,
		int style, char **column_names)
{
  int i;
  FILE *fp;
  if(!readconf)
    return CRET_INVALID;
  if(!*readconf && (i = config_new(readconf)))
    return i;
  (*readconf)->colnames = column_names;
  (*readconf)->filename = xstrdup(filename);
  (*readconf)->style = style;
  fp = fopen(filename,"r");
  if(!fp) {
    config_destroy(readconf);
    return CRET_NOTFOUND;
  }

  switch(style) {
  case CS_GUESS:
    i = config_read_GUESS(*readconf, fp);
    break;
  case CS_LINEVAL:
    i = config_read_LINEVAL(*readconf, fp);
    break;
  case CS_CAP:
    i = config_read_CAP(*readconf, fp);
    break;
  case CS_INI:
    i = config_read_INI(*readconf, fp);
    break;
  default:
    i = CRET_INVALID;
  }
  fclose(fp);
  if(i != CRET_SUCCESS)
    config_destroy(readconf);
  return i;
}

static int config_read_GUESS(config_t c, FILE *infile)
{
  return CRET_SUCCESS;
}

static int config_read_LINEVAL(config_t c, FILE *infile)
{
  char key[_POSIX2_LINE_MAX + 1];
  char kvsep[_POSIX2_LINE_MAX + 1];
  char line[_POSIX2_LINE_MAX + 1];
  char value[_POSIX2_LINE_MAX + 1];
  hash_t h = NULL, i;
  int type, count;
  int p1, p2, p3, p4, t, maxkvs=0;
  char **tar, **tar2, *ar;
  regmatch_t pmatch[10];
  regex_t lm;

  p1 = regcomp(&lm, "^([a-z0-9_.]+)([ \t:=]+)(.+)$", REG_EXTENDED|REG_ICASE);

  while(fgets(line, _POSIX2_LINE_MAX, infile)) {
    p3 = strlen(line) - 1;
    while(line[p3] == '\n' || line[p3] == '\r') line[p3--] = '\0';
    p1 = regexec(&lm, line, 10, pmatch, 0);
    if(p1)
      config_set(c, NULL, line + 1, NULL, COMMENT_T, line + 1, 1);
    else {
      config_debug("Got line %s\n",line);
      if(!p1) {
	p2 = pmatch[1].rm_eo - pmatch[1].rm_so;
	if(p2 < 0) p2 = 0;
	p3 = pmatch[2].rm_eo - pmatch[2].rm_so;
	if(p2 < 0) p2 = 0;
	p4 = pmatch[3].rm_eo - pmatch[3].rm_so;
	if(p4 < 0) p4 = 0;
	strncpy(key, (char *)((int)line + pmatch[1].rm_so), p2);
	key[p2] = '\0';
	strncpy(kvsep, (char *)((int)line + pmatch[2].rm_so), p3);
	kvsep[p3] = '\0';
	if(h) {
	  for(i = h; i->next && strcasecmp(kvsep, i->next->key); i = i->next);
	  if(i->next) {
	    t = (int)i->next->value;
	    i->next->value = (void *)t++;
	  } else {
	    i->next = malloc(sizeof(struct hash_t_s));
	    i->next->next = NULL;
	    i->next->key = xstrdup(kvsep);
	    i->next->value = (void *)1;
	  }
	} else {
	  h = malloc(sizeof(struct hash_t_s));
	  h->key = xstrdup(kvsep);
	  h->value = (void *)1; h->next = NULL;
	}
	strncpy(value, (char *)((int)line + pmatch[3].rm_so), p4);
	value[p4] = '\0';
	config_debug("Got key %s and value %s|\n", key, value);
	if(!config_get(c, NULL, key, NULL, &type, (void **) &ar, &count)) {
	  tar2 = malloc(sizeof(char *) * ++count);
	  assert(tar2);
	  if(type == CHARPTR_T) {
	    tar2[0] = value;
	    tar2[1] = ar;
	    config_debug("tar2[0] = %s, tar2[1] = %s\n", tar2[0], tar2[1]);
	  } else if(type == CHARPTRARRAY_T) {
	    tar = (char **)ar;
	    memcpy(tar2, tar, sizeof(char *) * (count - 1));
	    tar2[count - 1] = value;
	  } else
	    assert(0); /* We don't dupe anything else right now */
	  config_set(c, NULL, key, NULL, CHARPTRARRAY_T, tar2, count);
	  free(tar2);
	} else
	  config_set(c, NULL, key, NULL, CHARPTR_T, value, 1);
	
	/*	if(/^([a-zA-Z0-9_\-.]+)(\s*[:=]?\s*)(.*)$/) { */
      }
    }
    /*    memset(line, 0, _POSIX2_LINE_MAX); */
  }
  if(h) {
    for(i = h; i; i = i->next)
      if((int)i->value > maxkvs) {
	maxkvs = (int)i->value;
	free(c->kvsep); c->kvsep = xstrdup(i->key);
      }
  } else
    c->kvsep = xstrdup(" ");
  while(h) {
    if(h->key) free(h->key);
    i = h->next; free(h); h = i;
  }
  regfree(&lm);
  return CRET_SUCCESS;
}

static int config_read_CAP(config_t c, FILE *infile)
{
  char line[_POSIX2_LINE_MAX + 1];
  char key[_POSIX2_LINE_MAX + 1];
  char column[_POSIX2_LINE_MAX + 1];
  char value[_POSIX2_LINE_MAX + 1];
  int infl=0, ll, p1, p2, p3; /* ll == line length */
  key[0] = line[0] = column[0] = '\0';
  while(fgets(line, _POSIX2_LINE_MAX, infile) && line[0] == '#') {
    ll = strlen(line) - 1;
    while(line[ll] == '\n' || line[ll] == '\r') line[ll--] = '\0';
    config_set(c, NULL, line + 1, NULL, COMMENT_T, line + 1, 1);
  }
  if(feof(infile))
    return CRET_SUCCESS;
  do {
    ll = strlen(line) - 1;
    while(line[ll] == '\n' || line[ll] == '\r') line[ll--] = '\0';
    if(line[0] == '#')
      config_set(c, NULL, line + 1, NULL, COMMENT_T, line + 1, 1);
    else {
      for(p1 = 0; isblank(line[p1]);p1++);
      if(!line[p1]) continue;
      if(infl) {
	while(line[p1] && line[p1] == ':') p1++;
	for(; p1 < ll; p1++) {
	  p3 = 0;
	  for(p2 = p1;
	      line[p2] && (line[p2] != ':' && line[p2] != '#' &&
			   line[p2] != '=' && line[p2] != '\\');
	      column[p3++] = line[p2++]);
	  column[p3] = '\0';
	  switch(line[p2]) {
	  case ':':
	    config_set(c, NULL, key, column, BOOLEAN_T, (void **)1, 1);
	    p1 = p2;
	    break;
	  case '#':
	    p3 = 0;
	    for(p2++; line[p2] && line[p2] != ':';
		value[p3++] = line[p2++]);
	    value[p3] = '\0';
	    config_set(c, NULL, key, column, INT_T, (void **)atoi(value), 1);
	    p1 = ++p2;
	    break;
	  case '=':
	    p3 = 0;
	    for(p2++; line[p2] && (line[p2] != ':' && line[p2] != '\\');
		value[p3++] = line[p2++]);
	    value[p3] ='\0';
	    config_set(c, NULL, key, column, CHARPTR_T, value, 1);
	    p1 = ++p2;
	    break;
	  default:
	    assert(0);
	  }
	}
	if(line[ll] != '\\') infl = 0;
      } else {
	p3 = 0;
	for(p2 = p1; line[p2] && (line[p2] != ':' && line[p2] != '\\');
	    key[p3++] = line[p2++]);
	key[p3] = '\0';
	infl = 1;
	config_debug("Got key %s\n", key);
      }
    }
  } while(fgets(line, _POSIX2_LINE_MAX, infile));
  return CRET_SUCCESS;
}


static int config_read_INI(config_t c, FILE *infile)
{
  char *group = NULL;
  char key[_POSIX2_LINE_MAX + 1];
  char linehead[_POSIX2_LINE_MAX + 1];
  char kvsep[_POSIX2_LINE_MAX + 1];
  char line[_POSIX2_LINE_MAX + 1];
  char value[_POSIX2_LINE_MAX + 1];
  hash_t h = NULL, i, j = NULL;
  int type=0, count=0;
  int p1, p2, p3, p4, p5, t, maxkvs=0, maxlh=0;
  char **tar, **tar2, *ar;
  config_debug("INI start\n");

  while(fgets(line, _POSIX2_LINE_MAX, infile)) {
    p3 = strlen(line) - 1;
    while(line[p3] == '\n' || line[p3] == '\r') line[p3--] = '\0';
    if(!line[0]) continue;
    if(line[0] == ';')
      config_set(c, group, line + 1, NULL, COMMENT_T, line + 1, 1);
    else if(line[0] == '[') { /* It's a group starter */
      free(group);
      group = malloc(strlen(line) - 1);
      strncpy(group, line + 1, strlen(line) - 2);
      group[strlen(line) - 2] = '\0';
    } else {
      for(p1 = 0; line[p1] && isblank(line[p1]); p1++)
	linehead[p1] = line[p1];
      linehead[p1] = '\0';
      if(!line[p1])
	continue;
      /*      else if(line[p1] == ';')
	config_set(c, NULL, line + 1, NULL, COMMENT_T, line + 1, 1); */
      else {
	for(p2 = p1; line[p2] && line[p2] != '=' && line[p2] != ':'; p2++)
	  key[p2 - p1] = line[p2];
	key[p2 - p1] = '\0';
	p4 = 0;
	if(!line[p2])
	  continue;
	for(p3 = strlen(key) - 1; isblank(key[p3]);) {
	  kvsep[p4++] = key[p3]; key[p3--] = '\0';
	}	
	for(p3 = p2; line[p3] &&
	      (isblank(line[p3]) || line[p3] == '=' || line[p3] == ':'); p3++)
	  kvsep[p4++] = line[p3];
	kvsep[p4] = '\0';
	if(!line[p3]) {
	  value[0] = '\0';
	} else {
	  p5 = 0;
	  for(p4 = p3; line[p4]; p4++) value[p5++] = line[p4];
	  value[p5] = '\0';
	}
	if(j) {
	  for(i = j; i->next && strcasecmp(linehead, i->next->key);
	      i = i->next);
	  if(i->next) {
	    t = (int)i->next->value;
	    i->next->value = (void *)t++;
	  } else {
	    i->next = malloc(sizeof(struct hash_t_s));
	    i->next->next = NULL;
	    i->next->key = xstrdup(kvsep);
	    i->next->value = (void *)1;
	  }
	} else {
	  j = malloc(sizeof(struct hash_t_s));
	  j->key = xstrdup(linehead);
	  j->value = (void *)1; j->next = NULL;
	}
	if(h) {
	  for(i = h; i->next && strcasecmp(kvsep, i->next->key); i = i->next);
	  if(i->next) {
	    t = (int)i->next->value;
	    i->next->value = (void *)t++;
	  } else {
	    i->next = malloc(sizeof(struct hash_t_s));
	    i->next->next = NULL;
	    i->next->key = xstrdup(kvsep);
	    i->next->value = (void *)1;
	  }
	} else {
	  h = malloc(sizeof(struct hash_t_s));
	  h->key = xstrdup(kvsep);
	  h->value = (void *)1; h->next = NULL;
	}
	config_debug("Got key %s and value %s|\n", key, value);
	if(!config_get(c, group, key, NULL, &type, (void **) &ar, &count)) {
	  tar2 = malloc(sizeof(char *) * ++count);
	  assert(tar2);
	  if(type == CHARPTR_T) {
	    tar2[0] = value;
	    tar2[1] = ar;
	    config_debug("tar2[0] = %s, tar2[1] = %s\n", tar2[0], tar2[1]);
	  } else if(type == CHARPTRARRAY_T) {
	    tar = (char **)ar;
	    memcpy(tar2, tar, sizeof(char *) * (count - 1));
	    tar2[count - 1] = value;
	  } else
	    assert(0); /* We don't dupe anything else right now */
	  config_set(c, NULL, key, NULL, CHARPTRARRAY_T, tar2, count);
	  free(tar2);
	} else
	  config_set(c, group, key, NULL, CHARPTR_T, value, 1);
      }
    }
    /*    memset(line, 0, _POSIX2_LINE_MAX); */
  }
  if(h) {
    for(i = h; i; i = i->next)
      if((int)i->value > maxkvs) {
	maxkvs = (int)i->value;
	free(c->kvsep); c->kvsep = xstrdup(i->key);
      }
  } else
    c->kvsep = xstrdup(" ");
  if(j) {
    for(i = j; i; i = i->next)
      if((int)i->value > maxlh) {
	maxlh = (int)i->value;
	free(c->linehead); c->linehead = xstrdup(i->key);
      }
  } else
    c->linehead = xstrdup("");
  while(h) {
    if(h->key) free(h->key);
    i = h->next; free(h); h = i;
  }
  return CRET_SUCCESS;
}
