/*
  Library for reading configuration files
 */

#include <stdlib.h>
#include <errno.h>
#include <stdarg.h>
#include <string.h>
#include <stdio.h>
#include <assert.h>

#include "cpriv.h"

int config_new(config_t *newconf)
{
  *newconf = malloc(sizeof(struct config_t_s));
  if(!*newconf)
    return CRET_MEM;
  (*newconf)->flags = 0;
  (*newconf)->style = CS_GUESS;
  (*newconf)->groups = (hash_t) NULL;
  (*newconf)->filename = (*newconf)->kvsep = (*newconf)->linehead =
    (char *) NULL;
  (*newconf)->colnames = NULL;
  /* malloc(sizeof(struct hash_t_s)); */
    /*  (*newconf)->groups->key = NULL;
  (*newconf)->groups->value = NULL;
  (*newconf)->groups->next = NULL; */
  return CRET_SUCCESS;
}

int config_destroy(config_t *destconf)
{
  hash_t h,i,j,t;
  /* For each item in a linked list,
     - Deallocate key & value if needed
     - deallocate node
     - Go to next node
   */
  if(!destconf)
    return CRET_INVALID;
  if(!*destconf) {
    free(destconf);
    destconf = NULL;
    return CRET_INVALID;
  }
  for(h = (*destconf)->groups; h;) {
    if(h->key)
      free(h->key);
    if(h->value) {
      for(i = h->value; i;) {
	if(i->key)
	  free(i->key);
	if(i->value) {
	  for(j = i->value; j;) {
	    if(j->key)
	      free(j->key);
	    config_val_free(j->value);
	    t = j->next; free(j); j = t;
	  }
	}
	t = i->next; free(i); i = t;
      }
    }
    t = h->next; free(h); h = t;
  }
  if((*destconf)->filename)
    free((*destconf)->filename);
  if((*destconf)->kvsep)
    free((*destconf)->kvsep);
  if((*destconf)->colnames)
    free((*destconf)->colnames);
  if((*destconf)->linehead)
    free((*destconf)->linehead);
  free(*destconf);
  *destconf = NULL;
  destconf = NULL;
  return CRET_SUCCESS;
}

int config_set(config_t c, char *group, char *key, char *column, int type,
	       void *value, int count)
{
  hash_t h, p, vp;
  struct conf_val *newval, *freeval;
  int counter;
  char **chararray = NULL;
  h = c->groups;
  if(!h) {
    c->groups = malloc(sizeof(struct hash_t_s));
    h = c->groups;
    h->key = h->value = h->next = NULL;
  }
  vp = p = NULL;
  /* If h is not NULL, and the two things don't match, go on
     matching means that either BOTH are NULL,
     or BOTH are not null and match */
#define _charptrs_match(x) ((!h->key && !x) \
			    || (x && h->key && !strcasecmp(x,h->key)))
    
  for(;
      h && !_charptrs_match(group);) {
    p =  h;
    h = h->next;
  }

  if(!h) {
    if(p) {
      p->next = malloc(sizeof(struct hash_t_s));
      if(!(p->next))
	return CRET_MEM;
      p = h = p->next;
    } else {
      h = p = c->groups = malloc(sizeof(struct hash_t_s));
    }
    h->key = group?xstrdup(group):NULL;
    vp = h;
    h = h->next = h->value = NULL;
  } else {
    vp = h;
    h = h->value;
  }
  p = NULL;

  for(;
      h && !_charptrs_match(key);) {
    p = h;
    h = h->next;
  }

  if(!h) {
    if(p) {
      p->next = malloc(sizeof(struct hash_t_s));
      if(!(p->next))
	return CRET_MEM;
      p = h = p->next;
    } else {
      h = p = vp->value = malloc(sizeof(struct hash_t_s));
    }
    vp = h;
    h->key = key?xstrdup(key):NULL;
    h = h->next = h->value = NULL;
  } else {
    vp = h;
    h = h->value;
  }
  p = NULL;

  for(;
      h && !_charptrs_match(column);) {
    p = h;
    h = h->next;
  }

  if(!h) {
    if(p)
      h = p->next = malloc(sizeof(struct hash_t_s));
    else
      h = vp->value = malloc(sizeof(struct hash_t_s));
    if(!h)
      return CRET_MEM;
    h->key = column?xstrdup(column):NULL;
    h->next = h->value = NULL;
  }
  freeval = h->value;
  newval = malloc(sizeof(struct conf_val));
  newval->type = type;
  switch(type) {
  case CHARPTR_T: case COMMENT_T:
    newval->u.charptr_val = value?xstrdup(value):NULL; newval->count = 1;
    break;
  case CHARPTRARRAY_T:
    newval->u.charptrarray_val = malloc(sizeof(char *) * count);
    chararray = value;
    for(counter = 0; counter < count; counter++) {
      newval->u.charptrarray_val[counter] = xstrdup(chararray[counter]);
      config_debug("Duping %s| (%s|)\n", newval->u.charptrarray_val[counter],
		   chararray[counter]);
    }
    newval->count = count;
    break;
  case INT_T:
    newval->u.int_val = (int) value; newval->count = 1;
    break;
  case BOOLEAN_T:
    newval->u.boolean_val = (boolean) value; newval->count = 1;
    break;
    /* Assumptions: sizeof(void *) == sizeof(int) */
  default:
	  h->value = NULL; /* So that if we get any errors assigning
			      values, we can know we don't have
			      something weird sitting around */
    free(newval);
    return CRET_INVALID;
  }
  h->value = newval;
  config_val_free(freeval);
  return CRET_SUCCESS;
}

int config_unset(config_t c, char *group, char *key, char *column)
{
  hash_t h, p, vp, t;
  hash_t group_p, key_p, key_vp, col_p, col_vp;
  struct conf_val *val = NULL;
  vp = p = NULL;
  for(h = c->groups;
      h && !_charptrs_match(group);) {
    p =  h;
    h = h->next;
  }
  if(!h)
    return CRET_NOTFOUND;

  group_p = p;
  key_vp = h;

  for(h = h->value;
      h && !_charptrs_match(key);) {
    p = h;
    h = h->next;
  }
  if(!h)
    return CRET_NOTFOUND;

  key_p = p;
  col_vp = h;

  for(h = h->value;
      h && !_charptrs_match(column);) {
    p = h;
    h = h->next;
  }
  col_p = p;
  if(!h)
    return CRET_NOTFOUND;
  val = h->value;

  if(h->key)
    free(h->key);
  config_val_free(h->value);
  if(col_p)
    p = col_p->next = col_p->next->next;
  else {
    t = col_vp;
    p = t->value = h->next;
  }
  if(h)
    free(h);
  if(!p) {
    if(key_p) {
      p = key_p->next;
      key_p->next = key_p->next->next;
    } else {
      t = key_vp; p = t->value;
      t->value = p->next;
    }
    if(p->key)
      free(p->key);
    if(p)
      free(p);
    else {
      if(group_p) {
	p = group_p->next;
	group_p->next = group_p->next->next;
      } else {
	p = c->groups;
	c->groups = c->groups->next;
      }
      if(p->key)
	free(p->key);
      if(p)
	free(p);
    }
  }
  return CRET_SUCCESS;
}

int config_get(config_t c, char *group, char *key, char *column,
	       int *type, void **value, int *count)
{
  hash_t h;
  struct conf_val *val;
  for(h = c->groups;
      h && !_charptrs_match(group);
      h = h->next);
  if(!h)
    return CRET_NOTFOUND;

  for(h = h->value;
      h && !_charptrs_match(key);
      h = h->next);
  if(!h)
    return CRET_NOTFOUND;

  for(h = h->value;
      h && !_charptrs_match(column);
      h = h->next);

  if(!h)
    return CRET_NOTFOUND;
  val = h->value;
  if(!val)
    return CRET_NOTFOUND; /* Weirdness */
  *type = val->type;
  *count = val->count;
  *value = config_val_get(val);
  return CRET_SUCCESS;
}

/* Makes grplist point to a NULL-terminated array of character pointers
   Has a bug - doesn't return the NULL key in there, but NULL group still
   *is* valid for set/unset/get/keys/columns()'s
 */
int config_groups(config_t c, char ***grplist)
{
  int i = 0;
  hash_t h;
  for(h = c->groups; /* First we have to find the size of array to malloc() */
      h; h = h->next) i++;
  i++;
  config_debug("Got %d groups\n", i);
  *grplist = malloc(i * sizeof(char *));
  if(!*grplist)
    return CRET_MEM;
  i = 0;
  for(h = c->groups;
      h; h = h->next) {
    (*grplist)[i++] = xstrdup(h->key?h->key:"NULL");
  }
  (*grplist)[i] = NULL;
  return CRET_SUCCESS;
}


int config_keys(config_t c, char *group, char ***keylist)
{
  int type, b;
  void *a;
  int i = 0;
  hash_t h, g;
  for(h = c->groups; h && !_charptrs_match(group); h = h->next);
  if(!h)
    return CRET_NOTFOUND;
  g = h;
  for(h = g->value; /* First we have to find the size of array to malloc() */
      h; h = h->next) i++;
  i++;
  config_debug("Got %d groups\n", i);
  *keylist = malloc(i * sizeof(char *));
  if(!*keylist)
    return CRET_MEM;
  i = 0;
  /* Get all keys, ignoring the ones that are comments */
  for(h = g->value;
      h; h = h->next)
    if(config_get(c, group, h->key, NULL, &type, &a, &b) && type != COMMENT_T)
      (*keylist)[i++] = xstrdup(h->key?h->key:"NULL");
  
  (*keylist)[i] = NULL;
  config_debug("*keylist[%d] = NULL\n",i);
  return CRET_SUCCESS;
}

int config_columns(config_t c, char *group, char *key, char ***collist)
{
  int i = 0;
  struct conf_val *t;
  hash_t h, g, k;
  for(h = c->groups; h && !_charptrs_match(group); h = h->next);
  if(!h)
    return CRET_NOTFOUND;
  g = h;
  for(h = g->value; h && !_charptrs_match(key); h = h->next);
  if(!h)
    return CRET_NOTFOUND;
  k = h;
  for(h = k->value; /* First we have to find the size of array to malloc() */
      h; h = h->next) i++;
  i++;
  config_debug("Got %d groups\n", i);
  *collist = malloc(i * sizeof(char *));
  if(!*collist)
    return CRET_MEM;
  i = 0;
  for(h = k->value;
      h; h = h->next) {
    t = h->value;
    if(t->type != COMMENT_T)
      (*collist)[i++] = xstrdup(h->key);
    }
  (*collist)[i] = NULL;
  config_debug("*collist[%d] = NULL\n",i);
  return CRET_SUCCESS;
  return CRET_SUCCESS;
}

int config_arrayfree(char ***anarray)
{
  int i;
  config_debug("Doing config_arrayfree\n");
  for(i = 0; (*anarray)[i]; i++) free((*anarray)[i]);
  free(*anarray);
  *anarray = NULL;
  return CRET_SUCCESS;
}

void config_debug(char *format, ...) {
#ifdef DEBUG
  va_list args;
  va_start(args, format);
  vfprintf(stderr,format, args);
  va_end(args);
#endif
}

char *xstrdup(char *astring)
{
  char *t = malloc(strlen(astring) + 1);
  assert(t);
  strcpy(t, astring);
  return t;
}
#if 0
#undef malloc
void *xmalloc(size_t size)
{
  void *p;
  config_debug("Allocated %X\n", (p = malloc(size)));
  return p;
}
#undef free
void xfree(void *ptr)
{
  config_debug("Freeing %X\n", ptr);
  free(ptr);
}
#endif
int config_dump(config_t c, FILE *fp)
{
  hash_t h, i, j;
  struct conf_val *v;
  int counter;
  fputs("GROUP\n", fp);
  for(h = c->groups; h; h = h->next) {
    fprintf(fp,"%s\n",h->key?h->key:"NULL");
    for(i = h->value; i; i = i->next) {
      fputs("            KEY\n", fp);
      fprintf(fp, "%s\n", i->key?i->key:"NULL");
      for(j = i->value; j; j = j->next) {
	fputs("                    COLUMN	VALUE\n", fp);
	v = j->value;
	fprintf(fp, "%s ", j->key?j->key:"NULL");
	switch(v->type) {
	case CHARPTR_T:
	  fprintf(fp, "%s\n", v->u.charptr_val);
	  break;
	case COMMENT_T:
	  fprintf(fp, "# %s\n", v->u.charptr_val);
	  break;
	case INT_T:
	  fprintf(fp, "[%d]\n", v->u.int_val);
	  break;
	case CHARPTRARRAY_T:
	  fputs("", fp);
	  for(counter = 0; counter < v->count; counter++)
	    fprintf(fp, "|%s\n", *(v->u.charptrarray_val)[counter]);
	  break;
	case BOOLEAN_T:
	  if(v->u.boolean_val) fputs("true\n", fp);
	  else fputs("false\n", fp);
	  break;
	default:
	  fprintf(fp, "Unknown type %d\n", v->type);
	}
      }
    }
  }
  return CRET_SUCCESS;
}

void config_val_free(struct conf_val *aval)
{
  int i;
  if(aval) {
    switch(aval->type) {
    case CHARPTR_T: case COMMENT_T:
      free(aval->u.charptr_val);
      aval->u.charptr_val = NULL;
      break;
    case CHARPTRARRAY_T:
      for(i = 0; i < aval->count; i++) {
	free(aval->u.charptrarray_val[i]);
	aval->u.charptrarray_val[i] = NULL;
      }
      free(aval->u.charptrarray_val);
      aval->u.charptrarray_val = NULL;
      break;
    }
    free(aval);
  }
}

void *config_val_get(struct conf_val *aval)
{
  if(!aval)
    return NULL;
  switch(aval->type) {
  case CHARPTR_T: case COMMENT_T:
    return aval->u.charptr_val;
  case INT_T:
    return (void *)aval->u.int_val;
  case CHARPTRARRAY_T:
    return aval->u.charptrarray_val;
  default:
    return 0;
  }
}
