/* $Id: comm.c,v 1.3 2001/02/04 12:18:28 rsmith Exp $ 
 * -*- c -*- 
 * This file contains the comm module. A module for exchanging and
 * processing structured messages between applications.
 * Copyright (C) 2001  R.F. Smith <rsmith@xs4all.nl>
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this code; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * $Log: comm.c,v $
 * Revision 1.3  2001/02/04 12:18:28  rsmith
 * Implemented handling of `delim' in comm_read_text.
 *
 * Revision 1.2  2001/02/04 10:26:15  rsmith
 * Fixed bugs found in testing.
 *
 * Revision 1.1  2001/02/02 19:51:08  rsmith
 * Initial revision
 *
 * */

#include <errno.h>
#include <fcntl.h>
#include <limits.h>
#include <stdarg.h>
#define _GNU_SOURCE
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include "comm.h"

#ifndef NULL
#define NULL (void*)0
#endif

#ifndef RETURN_IF_FAIL
#define RETURN_IF_FAIL(test) if (!(test)) return
#endif

#ifndef RETURN_VAL_IF_FAIL
#define RETURN_VAL_IF_FAIL(test,val) if (!(test)) return ((val))
#endif

#ifndef NDEBUG
static void comm_debug (const char *func, const char *msg);
#define debug(MSG) comm_debug(__FUNCTION__, MSG)
#else
#define debug(MSG) (void)0
#endif

#define BSIZE 1024

typedef struct _node node_t;
struct _node
{
  node_t *next;
  int fd;
  char *writeptr;
  char *buf[BSIZE];
};

static node_t *nlist = 0;

static node_t *findnode (int fd);
static int fillnode (node_t *pnode);

int 
comm_get_int (char **ppstr, int *pres)
{
  long l;
  volatile char *endptr;
  if (ppstr == NULL || *ppstr == NULL || pres == NULL)
    {
      debug ("invalid parameter");
      return -1;        /* invalid parameter */
    }
  l = strtol (*ppstr, (char**)&endptr, 10);
  if (*ppstr == endptr)
    {
      debug ("no valid number found");
      return -2;        /* no valid number found */
    }
  if (l > INT_MAX)
    {
      debug ("value out if range");
      return -3;        /* value out of range */
    }
  *ppstr = (char*)endptr;
  *pres = (int)l;
  return 0;     /* normal return value */
}

int 
comm_get_uint (char **ppstr, unsigned *pres)
{
  unsigned long ul;
  volatile char *endptr;
  if (ppstr == NULL || *ppstr == NULL || pres == NULL)
    {
      debug ("invalid parameter");
      return -1;        /* invalid parameter */
    }
  ul = strtoul (*ppstr, (char**)&endptr, 10);
  if (*ppstr == endptr)
    {
      debug ("no valid number found");
      return -2;        /* no valid number found */
    }
  if (ul > UINT_MAX)
    {
      debug ("value out of range");
      return -3;        /* value out of range */
    }
  *ppstr = (char*)endptr;
  *pres = (unsigned)ul;
  return 0;     /* normal return value */
}

int 
comm_get_dbl (char **ppstr, double *pres)
{
  double d;
  volatile char *endptr;
  if (ppstr == NULL || *ppstr == NULL || pres == NULL)
    {
      debug ("invalid parameter ppstr");
      return -1;        /* invalid parameter */
    }
  d = strtod (*ppstr, (char**)&endptr);
  if (*ppstr == endptr)
    {
      debug ("no valid number found");
      return -2;        /* no valid number found */
    }
  *ppstr = (char*)endptr;
  *pres = d;
  return 0;     /* normal return value */
}

int 
comm_read_text (int fd, const char *delim, int num, tcomm_t handlers[])
{
  node_t *pnode = findnode (fd);
  char *found, *ptr;
  tcomm_t *ph = handlers;
  int n;
  const char *ws = " \t\n";
  
  if (pnode == NULL)
    {
      debug ("memory allocation failed");
      return -1;
    }
  if (fillnode (pnode) < 0)
    {
      debug ("reading from file failed");      
      return -2;
    }
  found = strstr ((char*)pnode->buf, delim);
  if (found)
    {
      *found = '\0';
      ptr = (char*)pnode->buf + strspn ((char*)pnode->buf, ws);
      for (n=0; n<num; n++)
        {
          ph = handlers+n;
          if (strncmp (ptr, ph->msgid, strlen(ph->msgid)) == 0)
            {
              ptr += strlen (ph->msgid);
              ptr += strspn (ptr, ws);
              if (ph->msg_handler)
                ph->msg_handler (ptr);
              pnode->writeptr = (char*)pnode->buf;
              memset (pnode->buf, 0, BSIZE);
              return 0;     /* normal return value */
            }
        }
      pnode->writeptr = (char*)pnode->buf;
      memset (pnode->buf, 0, BSIZE);
      debug ("unknown message");
      return -3;
    }
  return 0;     /* normal return value */
}

int 
comm_write_text (int fd, char *msgid, char *fmt, ...)
{
  int l;
  char buf[BSIZE];
  va_list ap;
  
  l = strlen(msgid);
  if (write (fd, msgid, l) == -1)
    {
      debug ("writing the message-id failed");
      return -1;
    }
  else if (write (fd, " ", 1) == -1)
    {
      debug ("writing the separator failed");
      return -2;
    }
  va_start (ap, fmt);
  l = vsnprintf (buf, BSIZE, fmt, ap);
  va_end (ap);
  if (l == -1)
    {
      debug ("buffer overflow while formatting message");
      return -3;
    }
  else if (write (fd, buf, l) == -1)
    {
      debug ("writing the message failed");
      return -4;
    }
  return 0;     /* normal return value */
}

static node_t *
findnode (int fd)
{
  node_t *pnode = nlist;
  /* Search in existing list. */
  while (pnode && pnode->fd != fd)
    {
      pnode = pnode->next;
    }
  if (pnode == NULL)
    {
      /* Set the file to non-blocking. */
      if (fcntl (fd, F_SETFL, O_NONBLOCK) == 0)
        {
          /* Allocate a new node, add it to the list. */
          pnode = calloc (1, sizeof(node_t));
          if (pnode)
            {
              pnode->fd = fd;
              pnode->writeptr = (char*)pnode->buf;
              pnode->next = nlist;
              nlist = pnode;
            }
        }
    }
  return pnode;
}

static int 
fillnode (node_t *pnode)
{
  int room;
  int rc;
  room = (unsigned)pnode->buf + BSIZE - (unsigned)pnode->writeptr;
  if (room < 1)
    {
      debug ("buffer is full");
      return -1; /* buffer is full */
    }
  room--;
  rc = read (pnode->fd, pnode->writeptr, room);
  if (rc == -1 && errno != EAGAIN)
    {
      debug ("read error");
      return -2; /* read error */
    }
  if (rc > 0)
    pnode->writeptr += rc;
  return 0;     /* normal return value */
}

#ifndef NDEBUG
static void 
comm_debug (const char *func, const char *msg)
{
  printf ("comm.c, %s: %s.\n", func, msg);  
}
#endif


/* EOF $RCSfile: comm.c,v $ */
