/*
This is part of the audio CD player library
Copyright (C)1998 Tony Arcieri

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Library General Public
License as published by the Free Software Foundation; either
version 2 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Library General Public License for more details.

You should have received a copy of the GNU Library General Public
License along with this library; if not, write to the
Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA  02111-1307, USA.
*/

#ifndef _CDAUDIO_H
#define _CDAUDIO_H

/* Used with disc_status */
#define CD_ABSENT                                 0
#define CD_PRESENT                                1

/* Used with disc_info */
#define CDAUDIO_PLAYING                           0
#define CDAUDIO_PAUSED                            1
#define CDAUDIO_COMPLETED                         2
#define CDAUDIO_NOSTATUS                          3

#define MAX_TRACKS                                100
#define MAX_SLOTS                                 100 /* For CD changers */

#define CURRENT_CDDBREVISION                      2
#define CURRENT_CDDBCONFREVISION                  1

/* CDDB defaults */
#define CDDB_DEFAULT_HOSTNAME                     "cddb.cddb.com"
#define CDDB_DEFAULT_PORT                         8880
#define CDDB_MAX_HOSTS                            128

/* CDDB genres, so I can store these as integers */
#define UNKNOWN                                   0
#define BLUES                                     1
#define CLASSICAL                                 2
#define COUNTRY                                   3
#define DATA                                      4
#define FOLK                                      5
#define JAZZ                                      6
#define MISC                                      7
#define NEWAGE                                    8
#define REGGAE                                    9
#define ROCK                                      10
#define SOUNDTRACK                                11

/* Used for keeping track of times */
struct disc_timeval {
   int minutes;
   int seconds;
};

/* Track specific information */
struct track_info {
   struct disc_timeval track_length;
   struct disc_timeval track_pos;
   int track_start;
};

/* Disc information such as current track, amount played, etc */
struct disc_info {
   int disc_status;                            /* Disc status */
   int disc_mode;                              /* Current disc mode */
   struct disc_timeval track_time;             /* Current track time */
   struct disc_timeval disc_time;              /* Current disc time */
   struct disc_timeval disc_length;            /* Total disc length */
   int disc_frame;                             /* Current frame */
   int disc_track;                             /* Current track */
   int disc_totaltracks;                       /* Number of tracks on disc */
   struct track_info track[MAX_TRACKS];        /* Track specific information */
};

/* Invisible volume structure */
struct __volume { 
   int left;
   int right;
};

/* Volume structure */
struct disc_volume {
   struct __volume vol_front;
   struct __volume vol_back;
};
   
/* Track database structure */
struct track_data {
   char track_name[128];                       /* Track name */
   char track_extendedindex;                   /* Number of lines of extended information */
   char track_extended[30][96];                /* Extended information */
};

/* Disc database structure */
struct disc_data {
   unsigned long data_id;                      /* CD id */
   char data_title[128];                       /* Disc title */
   char data_artist[128];                      /* We may be able to extract this */
   int data_extendedindex;                     /* Number of lines of extended information */
   char data_extended[30][96];                 /* Extended information */
   int data_genre;                             /* Disc genre */
   struct track_data data_track[MAX_TRACKS];   /* Track names */
};

/* Summary of a single disc in the changer */
struct disc_summary {
   int disc_status;                            /* Disc status */
   struct disc_timeval disc_length;            /* Length of disc */
   int disc_totaltracks;                       /* Total tracks */
   unsigned long disc_id;                      /* CDDB ID */
   int disc_info[128];                         /* Artist name / Disc name */
};

/* Disc changer structure */
struct disc_changer {
   int changer_slots;
   struct disc_summary changer_disc[MAX_SLOTS];
};

/* CDDB server list structure */
struct cddb_hostelement {
   char host_hostname[256];
   char host_city[256];
   int host_port;
   int host_icbm_latitude;
   int host_icbm_longitude;
};

/* CDDB server list structure */
struct cddb_serverlist {
   int list_len;
   struct cddb_hostelement list_host[256];
};

/* Function prototopes */
void cd_version(char *buffer, int len);
int cd_init(char *device);
int cd_stat(int cd_desc, struct disc_info *disc);
int cd_play_frames(int cd_desc, int startframe, int endframe);
int cd_play_track_pos(int cd_desc, int starttrack, int endtrack, int startpos);
int cd_play_track(int cd_desc, int starttrack, int endtrack);
int cd_play_pos(int cd_desc, int track, int startpos);
int cd_play(int cd_desc, int track);
int cd_track_advance(int cd_desc, struct disc_timeval *time);
int cd_stop(int cd_desc);
int cd_pause(int cd_desc);
int cd_resume(int cd_desc);
int cd_eject(int cd_desc);
int cd_close(int cd_desc);
int cd_set_volume(int cd_desc, struct disc_volume *vol);
int cd_get_volume(int cd_desc, struct disc_volume *vol);
unsigned long cddb_discid(int cd_desc);
char *cddb_genre(int genre);
int cddb_data_mod(char *var, char *value, struct disc_data *data);
int cddb_process_line(char *line, struct disc_data *data);
int cddb_read_data(int desc, struct disc_data *data);
int cddb_generate_new_entry(int cd_desc, struct disc_data *data);
int cddb_read_disc_data(int cd_desc, struct disc_data *data);
int cddb_write_disc_data(int cd_desc, struct disc_data *data);
int cddb_erase_entry(struct disc_data *data);
int cddb_refresh_disc_data(int cd_desc, struct disc_data *data);
int data_process_control_codes(char *outbuffer, char *inbuffer, int len);
int data_format_extended_info(char *outbuffer, struct disc_data *data, int track, int len);
int cd_changer_select_disc(int cd_desc, int disc);
int cd_changer_slots(int cd_desc);
int cd_changer_stat(int cd_desc, struct disc_changer *changer);

#endif
