/* $Id: vector.c,v 1.2 1997/10/28 19:39:56 rsmit06 Exp $
 *
 * This file is part of the affine transforms library (libat).
 * Copyright (C) 1997  R.F. Smith
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Log: vector.c,v $
 * Revision 1.2  1997/10/28 19:39:56  rsmit06
 * Enhanced comments, cleaned up the code.
 *
 * Revision 1.1  1997/09/28 12:33:18  rsmit06
 * Initial revision
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "at.h"

/*
 * Homegenizes a vector: [x,y,z] => [x,y,z,1]
 */
void 
at_hom(vec4_t dest, vec3_t src)
{
  dest[0] = src[0]; dest[1] = src[1]; dest[2] = src[2];
  dest[3] = (real_t)1;
}

/*
 * Transforms [x,y,z,W] into [x/W,y/W,z/W], if W != 0.
 * We call this `de-homogenizing'
 */
void 
at_dehom(vec3_t dest, vec4_t src)
{
  real_t W = src[3];
  /* prevent devide by zero! */
  if (W == (real_t)0) {
    fprintf(stderr, "ERROR in at_dehom: src[3] == 0!\n");
    exit(1);
  }
  dest[0] = src[0]/W; dest[1] = src[1]/W; dest[2] = src[2]/W; 
}

/* 
 * Transforms `v' using the homogeneous coordinates of matrix `m'.
 * 
 *                           |r11 r12 r13 0|
 * [x',y',z',w'] = [x,y,z,w]*|r21 r22 r23 0|
 *                           |r31 r32 r33 0|
 *                           |tx  ty  tz  1|
 */
void 
at_xf4(vec4_t res, mat4_t m, vec4_t v)
{
  /* The pattern is:
   * res[j] = v[0]*m[0][j]+v[1]*m[1][j]+v2*m[2][j]+v[3]*m[3][j];
   */
  res[0] = v[0]*m[0][0]+v[1]*m[1][0]+v[2]*m[2][0]+v[3]*m[3][0];
  res[1] = v[0]*m[0][1]+v[1]*m[1][1]+v[2]*m[2][1]+v[3]*m[3][1];
  res[2] = v[0]*m[0][2]+v[1]*m[1][2]+v[2]*m[2][2]+v[3]*m[3][2];
  res[3] = v[0]*m[0][3]+v[1]*m[1][3]+v[2]*m[2][3]+v[3]*m[3][3];
}

/* 
 * Transforms `v' using `m' without applying the translation
 * component of `m'.
 *                      |r11 r12 r13|
 * [x',y',z'] = [x,y,z]*|r21 r22 r23|
 *                      |r31 r32 r33|
 */
void 
at_xf3(vec3_t res, mat4_t m, vec3_t v)
{
  res[0] = v[0]*m[0][0]+v[1]*m[1][0]+v[2]*m[2][0];
  res[1] = v[0]*m[0][1]+v[1]*m[1][1]+v[2]*m[2][1];
  res[2] = v[0]*m[0][2]+v[1]*m[1][2]+v[2]*m[2][2];
}

/*
 * Calculates the cross product of the vectors `v1' and `v2', and returns it 
 * in vector `res'.
 */
void 
at_cross(vec3_t res, vec3_t v1, vec3_t v2)
{
  res[0] = v1[1]*v2[2] - v1[2]*v2[1];
  res[1] = v1[2]*v2[0] - v1[0]*v2[2];
  res[2] = v1[0]*v2[1] - v1[1]*v2[0];
}

/* 
 * Calculates and returns the dot product of `v1' and `v2'.
 * This is a measure for the angle between vectors.
 * It has the following properties:
 *   v1 . v2 > 0 if angle < 90 deg.
 *   v1 . v2 = 0 if angle = 90 deg.
 *   v1 . v2 < 0 if angle > 90 deg.
 */
real_t 
at_dot(vec3_t v1, vec3_t v2)
{
  return v1[0]*v2[0]+v1[1]*v2[1]+v1[2]*v2[2];
}

/* 
 * Adds vectors: `res' = `v1' + `v2'
 */
void 
at_add(vec3_t res, vec3_t v1, vec3_t v2)
{
  res[0] = v1[0] + v2[0]; res[1] = v1[1] + v2[1];
  res[2] = v1[2] + v2[2];
}

/*
 * Returns the length of `src'.
 */
real_t 
at_len(vec3_t src)
{
  real_t l = (real_t)0;

  l = src[0]*src[0] + src[1]*src[1] + src[2]*src[2];
  if (l == (real_t)0) return l;
  return (real_t)sqrt(l);
}

/*
 * Normalizes the vector `src'; scales it to length 1. 
 * Exits the program if the length of `src' is 0.
 */
void 
at_norm(vec3_t src)
{
  real_t l = at_len(src);
  if (l == (real_t)0) {
    fprintf(stderr, "ERROR in at_norm: length of `src' is 0!\n");
    exit(1);
  }
  src[0] /= l; src[1] /= l; src[2] /= l;
}

