/* $Id: matrix.c,v 1.2 1997/10/28 19:37:20 rsmit06 Exp $
 *
 * This file is part of the affine transforms library (libat)
 * Copyright (C) 1997 by R.F. Smith <rsmit06@ibm.net>
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Library General Public License as published
 * by the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 * $Log: matrix.c,v $
 * Revision 1.2  1997/10/28 19:37:20  rsmit06
 * Fixed bugs in at_view1 and at_rot. Added at_project.
 * Tried to enhance documentation :-)
 *
 * Revision 1.1  1997/09/28 12:32:35  rsmit06
 * Initial revision
 *
 */
#include <math.h>
#include <string.h>
#include "at.h"

/* Converts degrees to radians.
 * Takes a value, which is converted from degrees to radians.
 * Returns the converted value.
 */
real_t 
at_rad (real_t deg)
{
  return (real_t) M_PI*deg/180.0;
}

/* Sets a matrix to the Identity.
 * The single matrix argument is set to the I matrix
 */
void 
at_id (mat4_t m)
{
  m[0][0] = (real_t)1; m[0][1] = m[0][2] = m[0][3] = (real_t)0;
  m[1][1] = (real_t)1; m[1][0] = m[1][2] = m[1][3] = (real_t)0;
  m[2][2] = (real_t)1; m[2][0] = m[2][1] = m[2][3] = (real_t)0;
  m[3][3] = (real_t)1; m[3][0] = m[3][1] = m[3][2] = (real_t)0;
}

/* Copies matrices.
 * Copies the contents of src to dest.
 */
void 
at_cpy (mat4_t dest, mat4_t src)
{
  memcpy (dest, src, sizeof(mat4_t));
}

/* Multiplies matrices
 * The matrix res is set to the product of m1 and m2.
 * res = m1 * m2
 */
void 
at_mul (mat4_t res, mat4_t m1, mat4_t m2)
{
  /* The pattern is:
   * r[i][j] = m1[i][0]*m2[0][j] + m1[i][1]*m2[1][j]
   *         + m1[i][2]*m2[2][j] + m1[i][3]*m2[3][j];
   */

  res[0][0] = m1[0][0]*m2[0][0] + m1[0][1]*m2[1][0]
            + m1[0][2]*m2[2][0] + m1[0][3]*m2[3][0];
  res[0][1] = m1[0][0]*m2[0][1] + m1[0][1]*m2[1][1]
            + m1[0][2]*m2[2][1] + m1[0][3]*m2[3][1];
  res[0][2] = m1[0][0]*m2[0][2] + m1[0][1]*m2[1][2]
            + m1[0][2]*m2[2][2] + m1[0][3]*m2[3][2];
  res[0][3] = m1[0][0]*m2[0][3] + m1[0][1]*m2[1][3]
            + m1[0][2]*m2[2][3] + m1[0][3]*m2[3][3];

  res[1][0] = m1[1][0]*m2[0][0] + m1[1][1]*m2[1][0]
            + m1[1][2]*m2[2][0] + m1[1][3]*m2[3][0];
  res[1][1] = m1[1][0]*m2[0][1] + m1[1][1]*m2[1][1]
            + m1[1][2]*m2[2][1] + m1[1][3]*m2[3][1];
  res[1][2] = m1[1][0]*m2[0][2] + m1[1][1]*m2[1][2]
            + m1[1][2]*m2[2][2] + m1[1][3]*m2[3][2];
  res[1][3] = m1[1][0]*m2[0][3] + m1[1][1]*m2[1][3]
            + m1[1][2]*m2[2][3] + m1[1][3]*m2[3][3];

  res[2][0] = m1[2][0]*m2[0][0] + m1[2][1]*m2[1][0]
            + m1[2][2]*m2[2][0] + m1[2][3]*m2[3][0];
  res[2][1] = m1[2][0]*m2[0][1] + m1[2][1]*m2[1][1]
            + m1[2][2]*m2[2][1] + m1[2][3]*m2[3][1];
  res[2][2] = m1[2][0]*m2[0][2] + m1[2][1]*m2[1][2]
            + m1[2][2]*m2[2][2] + m1[2][3]*m2[3][2];
  res[2][3] = m1[2][0]*m2[0][3] + m1[2][1]*m2[1][3]
            + m1[2][2]*m2[2][3] + m1[2][3]*m2[3][3];

  res[3][0] = m1[3][0]*m2[0][0] + m1[3][1]*m2[1][0]
            + m1[3][2]*m2[2][0] + m1[3][3]*m2[3][0];
  res[3][1] = m1[3][0]*m2[0][1] + m1[3][1]*m2[1][1]
            + m1[3][2]*m2[2][1] + m1[3][3]*m2[3][1];
  res[3][2] = m1[3][0]*m2[0][2] + m1[3][1]*m2[1][2]
            + m1[3][2]*m2[2][2] + m1[3][3]*m2[3][2];
  res[3][3] = m1[3][0]*m2[0][3] + m1[3][1]*m2[1][3]
            + m1[3][2]*m2[2][3] + m1[3][3]*m2[3][3];
}

/* Concatates matrices.
 * Multiplies res with m1 and returns the result in res.
 */
void 
at_cat (mat4_t res, mat4_t m1)
{
  mat4_t tmp;
  at_mul (tmp, res, m1);
  memcpy (res, tmp, sizeof(mat4_t));
}

/* Calculates the determinant of a matrix.
 * Returns the calculated determinant.
 */
real_t 
at_det (mat4_t m)
{
  real_t det = (real_t)0;
  real_t tmp;
  if (m[0][3]) {
    tmp = m[1][0]*m[2][1]*m[3][2] + m[1][1]*m[2][2]*m[3][0] 
        + m[2][0]*m[3][1]*m[1][2] - m[1][2]*m[2][1]*m[3][0]
        - m[1][1]*m[2][0]*m[3][2] - m[2][2]*m[3][1]*m[1][0];
    det -= m[0][3]*tmp;
  }
  if (m[1][3]) {
    tmp = m[0][0]*m[2][1]*m[3][2] + m[0][1]*m[2][2]*m[3][0] 
        + m[2][0]*m[3][1]*m[0][2] - m[0][2]*m[2][1]*m[3][0]
        - m[0][1]*m[2][0]*m[3][2] - m[2][2]*m[3][1]*m[0][0];
    det += m[1][3]*tmp;
  }
  if (m[2][3]) {
    tmp = m[0][0]*m[1][1]*m[3][2] + m[0][1]*m[1][2]*m[3][0] 
        + m[1][0]*m[3][1]*m[0][2] - m[0][2]*m[1][1]*m[3][0]
        - m[0][1]*m[1][0]*m[3][2] - m[1][2]*m[3][1]*m[0][0];
    det -= m[2][3]*tmp;
  }
  if (m[3][3]) {
    tmp = m[0][0]*m[1][1]*m[2][2] + m[0][1]*m[1][2]*m[2][0] 
        + m[1][0]*m[2][1]*m[0][2] - m[0][2]*m[1][1]*m[2][0]
        - m[0][1]*m[1][0]*m[2][2] - m[1][2]*m[2][1]*m[0][0];
    det += m[3][3]*tmp;
  }

  return det;
}

/* Inverts a matrix.  
 * Calculates the determinant of src. If this value is
 * not 0, the inverse of src is returned in dest.
 * Returns 0 if src is invertable. Otherwise returns 1.
 */
int 
at_inv (mat4_t dest, mat4_t src)
{
  real_t d = at_det (src);
  mat4_t tmp;

  if (d == (real_t)0) return 1;

  /* first row */
  tmp[0][0] = src[1][1]*src[2][2]*src[3][3] + src[1][2]*src[2][3]*src[3][1]
            + src[2][1]*src[3][2]*src[1][3] - src[1][3]*src[2][2]*src[3][1]
            - src[1][2]*src[2][1]*src[3][3] - src[2][3]*src[3][2]*src[1][1];

  tmp[0][1] = src[1][0]*src[2][2]*src[3][3] + src[1][2]*src[2][3]*src[3][0]
            + src[2][0]*src[3][2]*src[1][3] - src[1][3]*src[2][2]*src[3][0]
            - src[1][2]*src[2][0]*src[3][3] - src[2][3]*src[3][2]*src[1][0];

  tmp[0][2] = src[1][0]*src[2][1]*src[3][3] + src[1][1]*src[2][3]*src[3][0]
            + src[2][0]*src[3][1]*src[1][3] - src[1][3]*src[2][1]*src[3][0]
            - src[1][1]*src[2][0]*src[3][3] - src[2][3]*src[3][1]*src[1][0];

  tmp[0][3] = src[1][0]*src[2][1]*src[3][2] + src[1][1]*src[2][2]*src[3][0]
            + src[2][0]*src[3][1]*src[1][2] - src[1][2]*src[2][1]*src[3][0]
            - src[1][1]*src[2][0]*src[3][2] - src[2][2]*src[3][1]*src[1][0];

  /* second row */
  tmp[1][0] = src[0][1]*src[2][2]*src[3][3] + src[0][2]*src[2][3]*src[3][1]
            + src[2][1]*src[3][2]*src[0][3] - src[0][3]*src[2][2]*src[3][1]
            - src[0][2]*src[2][1]*src[3][3] - src[2][3]*src[3][2]*src[0][1];

  tmp[1][1] = src[0][0]*src[2][2]*src[3][3] + src[0][2]*src[2][3]*src[3][0]
            + src[2][0]*src[3][2]*src[0][3] - src[0][3]*src[2][2]*src[3][0]
            - src[0][2]*src[2][0]*src[3][3] - src[2][3]*src[3][2]*src[0][0];

  tmp[1][2] = src[0][0]*src[2][1]*src[3][3] + src[0][1]*src[2][3]*src[3][0]
            + src[2][0]*src[3][1]*src[0][3] - src[0][3]*src[2][1]*src[3][0]
            - src[0][1]*src[2][0]*src[3][3] - src[2][3]*src[3][1]*src[0][0];

  tmp[1][3] = src[0][0]*src[2][1]*src[3][2] + src[0][1]*src[2][2]*src[3][0]
            + src[2][0]*src[3][1]*src[0][2] - src[0][2]*src[2][1]*src[3][0]
            - src[0][1]*src[2][0]*src[3][2] - src[2][2]*src[3][1]*src[0][0];

  /* third row */
  tmp[2][0] = src[0][1]*src[1][2]*src[3][3] + src[0][2]*src[1][3]*src[3][1]
            + src[1][1]*src[3][2]*src[0][3] - src[0][3]*src[1][2]*src[3][1]
            - src[0][2]*src[1][1]*src[3][3] - src[1][3]*src[3][2]*src[0][1];

  tmp[2][1] = src[0][0]*src[1][2]*src[3][3] + src[0][2]*src[1][3]*src[3][0]
            + src[1][0]*src[3][2]*src[0][3] - src[0][3]*src[1][2]*src[3][0]
            - src[0][2]*src[1][0]*src[3][3] - src[1][3]*src[3][2]*src[0][0];

  tmp[2][2] = src[0][0]*src[1][1]*src[3][3] + src[0][1]*src[1][3]*src[3][0]
            + src[1][0]*src[3][1]*src[0][3] - src[0][3]*src[1][1]*src[3][0]
            - src[0][1]*src[1][0]*src[3][3] - src[1][3]*src[3][1]*src[0][0];

  tmp[2][3] = src[0][0]*src[1][1]*src[3][2] + src[0][1]*src[1][2]*src[3][0]
            + src[1][0]*src[3][1]*src[0][2] - src[0][2]*src[1][1]*src[3][0]
            - src[0][1]*src[1][0]*src[3][2] - src[1][2]*src[3][1]*src[0][0];

  /* fourth row */
  tmp[3][0] = src[0][1]*src[1][2]*src[2][3] + src[0][2]*src[1][3]*src[2][1]
            + src[1][1]*src[2][2]*src[0][3] - src[0][3]*src[1][2]*src[2][1]
            - src[0][2]*src[1][1]*src[2][3] - src[1][3]*src[2][2]*src[0][1];

  tmp[3][1] = src[0][0]*src[1][2]*src[2][3] + src[0][2]*src[1][3]*src[2][0]
            + src[1][0]*src[2][2]*src[0][3] - src[0][3]*src[1][2]*src[2][0]
            - src[0][2]*src[1][0]*src[2][3] - src[1][3]*src[2][2]*src[0][0];

  tmp[3][2] = src[0][0]*src[1][1]*src[2][3] + src[0][1]*src[1][3]*src[2][0]
            + src[1][0]*src[2][1]*src[0][3] - src[0][3]*src[1][1]*src[2][0]
            - src[0][1]*src[1][0]*src[2][3] - src[1][3]*src[2][1]*src[0][0];

  tmp[3][3] = src[0][0]*src[1][1]*src[2][2] + src[0][1]*src[1][2]*src[2][0]
            + src[1][0]*src[2][1]*src[0][2] - src[0][2]*src[1][1]*src[2][0]
            - src[0][1]*src[1][0]*src[2][2] - src[1][2]*src[2][1]*src[0][0];

  dest[0][0] = tmp[0][0]/d; dest[0][1] = -tmp[1][0]/d; 
  dest[0][2] = tmp[2][0]/d; dest[0][3] = -tmp[3][0]/d;

  dest[1][0] = -tmp[0][1]/d; dest[1][1] = tmp[1][1]/d;
  dest[1][2] = -tmp[2][1]/d; dest[1][3] = tmp[3][1]/d;

  dest[2][0] = tmp[0][2]/d; dest[2][1] = -tmp[1][2]/d;
  dest[2][2] = tmp[2][2]/d; dest[2][3] = -tmp[3][2]/d;

  dest[3][0] = -tmp[0][3]/d; dest[3][1] = tmp[1][3]/d;
  dest[3][2] = -tmp[2][3]/d; dest[3][3] = tmp[3][3]/d;

  return 0;

}

/* Makes a rotation matrix for a rotation by an angle of rad radians around
 * the x-axes. The calculated matrix is returned in dest.
 * To make a coordinate system transform, negate the angle.
 *      
 *     | Y+         These functions assume a coordinate system
 *     |            with the axis arranged as shown left.
 *     |
 *     |______ X+
 *    /
 *   /
 *  / Z+
 */
void 
at_rotx (mat4_t dest, real_t rad)
{
  at_id (dest);
  dest[1][1] = dest[2][2] = (real_t)cos(rad);
  dest[1][2] = (real_t)sin(rad); dest[2][1] = -dest[1][2];
}

/* Makes a rotation matrix for a rotation by an angle of rad radians around
 * the y-axes. The calculated matrix is returned in dest.
 * To make a coordinate system transform, negate the angle.
 */
void 
at_roty (mat4_t dest, real_t rad)
{
  at_id (dest);
  dest[0][0] = dest[2][2] = (real_t)cos (rad);
  dest[2][0] = (real_t)sin (rad); dest[0][2] = -dest[2][0];
}

/* Makes a rotation matrix for a rotation by an angle of rad radians around
 * the z-axes. The calculated matrix is returned in dest.
 * To make a coordinate system transform, negate the angle.
 */
void 
at_rotz (mat4_t dest, real_t rad)
{
  at_id (dest);
  dest[0][0] = dest[2][2] = (real_t)cos (rad);
  dest[0][1] = (real_t)sin (rad); dest[1][0] = -dest[0][1];
}

/* Makes a translation matrix.
 * The matrix for the translation with the vector trans is returned in dest.
 * To make a coordinate system translation matrix, negate the vector.
 */
void 
at_tr (mat4_t dest, vec3_t trans)
{
  at_id (dest);
  dest[3][0] = trans[0]; dest[3][1] = trans[1]; 
  dest[3][2] = trans[2];
}

/* Makes a scaling matrix.  The matrix for the scaling operation is
 * returned in dest. 
 * The vector scale contains the scaling factors in x-,
 * y- and z-direction. If a scaling vector is 0, it is interpreted as 1.
 *
 */
void 
at_sc (mat4_t dest, vec3_t scale)
{
  real_t s;

  at_id (dest);

  if (scale[0] == (real_t)0)
    s = (real_t)1;
  else
    s = scale[0];
  dest[0][0] = s; 

  if (scale[1] == (real_t)0)
    s = (real_t)1;
  else
    s = scale[1];
  dest[1][1] = s; 

  if (scale[2] == (real_t)0)
    s = (real_t)1;
  else
    s = scale[2];
  dest[2][2] = s;
}

/* Makes a viewing matrix, that looks at the origin from a distance d. The
 * projection on the x-y plane of the line from the origin to d is at an
 * angle phi radians from the x-axis. The line is at an angle of theta
 * radians form the x-y plane. The calculated matrix is returned in dest.
 *
 *     | Y+         These functions assume a coordinate system
 *     |            with the axis arranged as shown left.
 *     |
 *     |______ X+          \ Z'+         When transformed, the coordinate
 *    /                     \            system looks like this: X' and Y'
 *   /                       \_____ X'+  define the screen plane, while Z'
 *  / Z+                     |           (the looking direction) points
 *                           |           to the origin of the global
 *                           | Y'+       coordinate system.
 *
 */
void 
at_view1 (mat4_t dest, real_t phi, real_t theta, real_t d)
{
  real_t sin_phi, cos_phi, sin_theta, cos_theta;

  sin_phi = (real_t)sin(phi); cos_phi = (real_t)cos(phi);
  sin_theta = (real_t)sin(theta); cos_theta = (real_t)cos(theta);

  at_id (dest);

  dest[0][0] = sin_phi; dest[0][1] = cos_phi*sin_theta; 
  dest[0][2] = -cos_phi*cos_theta;
  dest[1][0] = -cos_phi; dest[1][1] = -sin_phi*sin_theta;
  dest[1][2] = sin_phi*cos_theta;
  dest[2][1] = -cos_theta; dest[2][2] = -sin_theta;
  dest[3][2] = d;
}

/*
 * Builds a viewing transform matrix and stores it in `dest'. The
 * parameters for the viewing system are as follows:
 * `pos' is the origin of the view coordinate system.
 * `look' is the direction in which the camera looks. 
 * `up' establishes the direction to the top of the screen. 
 *      It is projected onto the plane that has `look' as it's normal.
 *      It maps to *negative* Yview coordinates!
 * If either `look' or `up' has length zero, `dest' is set to the 
 * identity matrix!
 */
void 
at_view3 (mat4_t dest, vec3_t pos, vec3_t look, vec3_t up)
{
  vec3_t Xv, Yv, Zv;
  real_t c;

  /* 
   * check the parameters. set dest matrix to identity and
   * return if one of the given vectors has length 0.
   */
  if(at_len(look)==(real_t)0 || at_len(up)==(real_t)0) {
    at_id(dest);
    return;
  }

  /* set Zview vector */
  memcpy(Zv, look, sizeof(vec3_t));
  at_norm(Zv);

  /* calculate Yview vector from `up' vector. */
  c = (-up[0]*look[0]-up[1]*look[1]-up[2]*look[2])/
      (look[0]*look[0]+look[1]*look[1]+look[2]*look[2]);
  Yv[0] = -(up[0]+c*look[0]);
  Yv[1] = -(up[1]+c*look[1]);
  Yv[2] = -(up[2]+c*look[2]);
  at_norm(Yv);

  /* calculate Xview vector from Yv and Zv */
  at_cross(Xv, Yv, Zv);

  /* put the values in the destination matrix */
  dest[0][0] = Xv[0]; dest[0][1] = Yv[0]; dest[0][2] = Zv[0];
  dest[1][0] = Xv[1]; dest[1][1] = Yv[1]; dest[1][2] = Zv[1];
  dest[2][0] = Xv[2]; dest[2][1] = Yv[2]; dest[2][2] = Zv[2];
  dest[3][0] = pos[0]; dest[3][1] = pos[1]; dest[3][2] = pos[2];
  dest[0][3] = dest[1][3] = dest[2][3] = (real_t)0; dest[3][3] = (real_t)1;
}

/*
 * Makes a perspective projection matrix for a viewport `wv' units wide, 
 * with the projection centre located `d' units behind the vieport. The 
 * viewport size in pixels is `ws' wide and `hs' high. 
 * The resulting matrix is stored in `dest'. If one of the viewport sizes
 * should be <= 0, or if `d' is <= 0, `dest' is set to the identity matrix. 
 */
void 
at_project(mat4_t dest, real_t d, real_t wv, int ws, int hs)
{
  /* set matrix to identity */
  at_id(dest);

  /* check parameters */
  if (d <= (real_t)0 || wv <= (real_t)0 || ws <= 0 || hs <= 0) {
    return;
  }

  /* create matrix values */
  ws /= 2; hs /= 2;
  d = (real_t)1/d;

  /* fill matrix */
  dest[0][0] = dest[1][1] = ws/wv;
  dest[2][0] = ws*d; dest[2][1] = hs*d; dest[2][3] = d;
  dest[3][0] = ws; dest[3][1] = hs;
}







