/* $Id: at.h,v 1.3 1997/10/28 19:36:07 rsmit06 Exp $
 *
 * Header file for affine transforms library (libat).
 * Copyright (C) 1997  R.F. Smith
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Log: at.h,v $
 * Revision 1.3  1997/10/28 19:36:07  rsmit06
 * Added at_project function, enhanced comments.
 *
 * Revision 1.2  1997/09/28 11:08:57  rsmit06
 * Changed library name to `at' (for Affine Transforms).
 *
 * Revision 1.1  1997/09/28 10:59:53  rsmit06
 * Initial revision
 *
 */

#ifndef LIBAT_H
#define LIBAT_H

typedef float real_t;           /* floating point type */
typedef real_t mat4_t[4][4];    /* 3D homogeneous matrix */
typedef real_t vec4_t[4];       /* 3D homogeneous vector */
typedef real_t vec3_t[3];       /* 3D vector */

/* 
 * Takes an angle in degrees, and returns this angle in radians.
 */
extern real_t at_rad(real_t deg);

/* 
 * Transforms the matrix `m' into the identity:
 * 
 *     |1 0 0 0|
 * m = |0 1 0 0|
 *     |0 0 1 0|
 *     |0 0 0 1|
 */
extern void at_id(mat4_t m);

/* 
 * Copies a matrix from `src' to `dest'.
 */
extern void at_cpy(mat4_t dest, mat4_t src);

/* 
 * Multiplies matrices `m1' and `m2', and stores the result in `res'.
 * This has the effect of applying the transformation embodied in `m2' 
 * after `m1'.
 */
extern void at_mul(mat4_t res, mat4_t m1, mat4_t m2);

/* 
 * Concatates matrices: multiplies `res' with `m1' 
 * and puts the result in `res'.
 */
extern void at_cat(mat4_t res, mat4_t m1);

/* 
 * Calculates and returns the determinant of a matrix.
 */
extern real_t at_det(mat4_t m);

/* 
 * If `src' is invertable, it puts the inverse in `dest', and returns 0.
 * Returns 1 if `src' cannot be inverted.
 */
extern int at_inv(mat4_t dest, mat4_t src);

/* 
 * Calculates rotation matrices for rotations around x, y and z axis.
 */
extern void at_rotx(mat4_t dest, real_t rad);
extern void at_roty(mat4_t dest, real_t rad);
extern void at_rotz(mat4_t dest, real_t rad);

/* 
 * Changes `dest' into a matrix that implements a translation along the
 * vector `trans'.
 */
extern void at_tr(mat4_t dest, vec3_t trans);

/* 
 * Calculates a matrix that implements a scaling transform and stores it in
 * `dest'. The vector `scale' contains the scaling values for the x-, y- and
 * z-directions. 
 */
extern void at_sc(mat4_t dest, vec3_t scale);

/* Makes a viewing matrix, that looks at the origin from a distance d. The
 * projection on the x-y plane of the line from the origin to d is at an
 * angle phi radians from the x-axis. The line is at an angle of theta
 * radians form the x-y plane. The calculated matrix is returned in dest.
 *
 *     | Z+         These functions assume a coordinate system
 *     |            with the axis arranged as shown left.
 *     |
 *     |______ Y+          \ Z'+         When transformed, the coordinate
 *    /                     \            system looks like this: X' and Y'
 *   /                       \_____ X'+  define the screen plane, while Z'
 *  / X+                     |           (the looking direction) points
 *                           |           to the origin of the global
 *                           | Y'+       coordinate system.
 *
 */
extern void at_view1(mat4_t dest, real_t phi, real_t theta, real_t d);

/*
 * Builds a viewing transform matrix and stores it in `dest'. The
 * parameters for the viewing system are as follows:
 * `pos' is the origin of the view coordinate system.
 * `look' is the direction in which the camera looks. 
 * `up' establishes the direction to the top of the screen. 
 *      It is projected onto the plane that has `look' as it's normal.
 *      It maps to *negative* Yview coordinates!
 * If either `look' or `up' has length zero, `dest' is set to the 
 * identity matrix!
 */
extern void at_view3(mat4_t dest, vec3_t pos, vec3_t look, vec3_t up);

/*
 * Makes a perspective projection matrix for a viewport `wv' units wide, 
 * with the projection centre located `d' units behind the vieport. The 
 * viewport size in pixels is `ws' wide and `hs' high. 
 * The resulting matrix is stored in `dest'. If one of the viewport sizes
 * should be <= 0, or if `d' is <= 0, `dest' is set to the identity matrix. 
 */
extern void at_project(mat4_t dest, real_t d, real_t wv, int ws, int hs);

/*
 * Homegenizes a vector: converts a vector to homogeneous coordinates.
 * [x,y,z] => [x,y,z,1]
 */
extern void at_hom(vec4_t dest, vec3_t src);

/*
 * Transforms [x,y,z,W] into [x/W,y/W,z/W], if W != 0.
 * We call this `de-homogenizing'
 */
extern void at_dehom(vec3_t dest, vec4_t src);

/* 
 * Transforms `v' using the homogeneous coordinates of matrix `m'.
 * 
 *                           |r11 r12 r13 0|
 * [x',y',z',w'] = [x,y,z,w]*|r21 r22 r23 0|
 *                           |r31 r32 r33 0|
 *                           |tx  ty  tz  1|
 */
extern void at_xf4(vec4_t res, mat4_t m, vec4_t v);

/* 
 * Transforms `v' using `m' without applying the translation
 * component of `m'.
 *                      |r11 r12 r13|
 * [x',y',z'] = [x,y,z]*|r21 r22 r23|
 *                      |r31 r32 r33|
 */
extern void at_xf3(vec3_t res, mat4_t m, vec3_t v);

/*
 * Calculates the cross product of the vectors `v1' and `v2', and returns it 
 * in vector `res'.
 */
extern void at_cross(vec3_t res, vec3_t v1, vec3_t v2);

/* 
 * Calculates and returns the dot product of `v1' and `v2'.
 * This is a measure for the angle between vectors.
 * It has the following properties:
 *   v1 . v2 > 0 if angle < 90 deg.
 *   v1 . v2 = 0 if angle = 90 deg.
 *   v1 . v2 < 0 if angle > 90 deg.
 */
extern real_t at_dot(vec3_t v1, vec3_t v2);

/* 
 * Adds vectors: `res' = `v1' + `v2'
 */
extern void at_add(vec3_t res, vec3_t v1, vec3_t v2);

/*
 * Returns the length of `src'.
 */
extern real_t at_len(vec3_t src);

/*
 * Normalizes the vector `src'. 
 * Exits the program if the length of `src' is 0.
 */
extern void at_norm(vec3_t src);


#endif /* LIBAT_H */
