/*
 * This software is copyrighted as noted below.  It may be freely copied,
 * modified, and redistributed, provided that the copyright notice is
 * preserved on all copies.
 *
 * There is no warranty or other guarantee of fitness for this software,
 * it is provided solely "as is".  Bug reports or fixes may be sent
 * to the author, who may or may not act on them as he desires.
 *
 * You may not include this software in a program or other software product
 * without supplying the source, or without informing the end-user that the
 * source is available for no extra charge.
 *
 * If you modify this software, you should include a notice giving the
 * name of the person performing the modification, the date of modification,
 * and the reason for such modification.
 *
 * Author:      Bruno Levy
 *
 * Copyright (c) 1996, Bruno Levy.
 *
 */
/*
 *
 * DOSgport.C
 *
 */


#include "dosgport.h"
#include "stdmok.h"
#include <stdlib.h>
#include <dos.h>
#include <graphics.h>

// I don't know how to specify a pointer in real mode
// from djgpp for interrupt information buffer (not sure if it's possible ...)
// that's why I don't use VESA info functions.

VESA_mode DOSGraphicPort::_vesa_table[] =
{
   { VESA_256,  320,  200, 0x13  },
   { VESA_256,  640,  400, 0x100 },
   { VESA_256,  640,  480, 0x101 },
   { VESA_256,  800,  600, 0x103 },
   { VESA_256, 1024,  768, 0x105 },
   { VESA_256, 1280, 1024, 0x107 },

   { VESA_32K,  320,  200, 0x10D },
   { VESA_64K,  320,  200, 0x10E },
   { VESA_16M,  320,  200, 0x10F },

   { VESA_32K,  640,  480, 0x110 },
   { VESA_64K,  640,  480, 0x111 },
   { VESA_16M,  640,  480, 0x112 },

   { VESA_32K,  800,  600, 0x113 },
   { VESA_64K,  800,  600, 0x114 },
   { VESA_16M,  800,  600, 0x115 },

   { VESA_32K, 1024,  768, 0x116 },
   { VESA_64K, 1024,  768, 0x117 },
   { VESA_16M, 1024,  768, 0x118 },

   { VESA_32K, 1280, 1024, 0x119 },
   { VESA_64K, 1280, 1024, 0x11A },
   { VESA_16M, 1280, 1024, 0x11B },

   { 0       ,  0  ,  0  , 0     }

};


DOSGraphicPort::DOSGraphicPort(char *name, 
				 ScrCoord width, ScrCoord height, int verbose_level): 
       GraphicPort(name, width, height, verbose_level)
{

  int i;

  _proc_name = "DOSGraphicPort";

  _error_code = DGPE_OK;

  _width  = width;
  _height = height;

  _bits_per_pixel  = 8;
  _bytes_per_pixel = 1;


  char *quiet;

  if(!((quiet = getenv("GQUIET")) && !strcmp(quiet,"yes")))
    {
    ScreenClear();
    ScreenSetCursor(0,0);
    cout << endl;
    cout << " ----==== TAGL port for MS-DOS ====----" << endl;
    cout << endl;
    cout << "This port uses VESA interrupts, it\'s not very fast." << endl;
    cout << "Try the Linux/Unix version, it runs much faster." << endl;
    cout << endl;
    cout << endl;
    cout << "For this dos version:" << endl;
    cout << endl;
    cout << "  o Make sure your VESA driver is loaded" << endl;
    cout << endl;
    cout << "  o To change the resolution:" << endl;
    cout << "type \'set GSVGAMODE=GWidthxHeightxDepth\' before running" << endl;
    cout << "where Depth can be either \'256\' or \'32K\' or \'64K\' or \'16M\'" << endl;
    cout << endl;
    cout << "  o To disable this message:" << endl;
    cout << "type \'set GQUIET=yes\' before running" << endl;
    cout << endl;
    cout << "  o To enable .grd/.grn/.drv bank switching" << endl;
    cout << "type \'set GLINEAR=yes\' before running" << endl; 
    cout << "GO32 env. variable must be set properly" << endl;
    cout << "for instance:" << endl;
    cout << "set GO32=ansi driver c:/djgpp/drivers/atipro.grd gw 1024 gh 768 nc 256" << endl;
    cout << endl;
    cout << "  o To disable VESA driver (use .grd/.grn/.vdr only):" << endl;
    cout << "type \'set GVESA=no\' before running" << endl;    
    cout << "GLINEAR must be set to \'yes\' when using this option" << endl;
    cout << endl;
    cout << "press any key to continue ..." << endl;
    while(!kbhit());
    }


  if(!SetupScreen())
    return;

  if(!AllocateImgBuffer())
    return;

  Attributes().Set(GPA_DBUFF);

  for(i=0; i<256; i++)
    _colormap[i] = i;

  ColormapMode();

  _clip.Set(0,0,_width-1,_height-1);

  _graph_mem = _v_graph_mem;

  VESA_InitMouse();

}


////
////
//
//   Attributes handling & destructor
//
////
////

DOSGraphicPort::~DOSGraphicPort(void)
{
  if(_resources.Get(DGPR_IMGBUFFER))
    FreeImgBuffer();

  if(_resources.Get(DGPR_ZBUFFER))
    FreeZBuffer();

  VESA_StdMode(0x03);
}

void DOSGraphicPort::CommitAttributes(void)
{
  Flags changed;

  changed.SetAll(_last_attributes.GetAll() ^ Attributes().GetAll());

  if(changed.Get(GPA_CMAP) && Attributes().Get(GPA_CMAP))
    ColormapMode();

  if(changed.Get(GPA_RGB)  && Attributes().Get(GPA_RGB))
    RGBMode();

  if(changed.Get(GPA_ZBUFF))
    ZBuffer(Attributes().Get(GPA_ZBUFF));

  if(changed.Get(GPA_DBUFF))
    {
      if(Attributes().Get(GPA_DBUFF))
	DoubleBuffer();
      else
	SingleBuffer();
    }
}

////
////
//
//   Set video mode 
//
////
////


int DOSGraphicPort::SetupScreen() 
{
  char *envmode;
  char  smode[600];

  _r_graph_mem = (unsigned char *)0xD0000000; // djgpp maps graph mem here !
                                              // may be different with other
                                              // DOS C++ environment



  // if GLINEAR environment variable defined,
  // try to enable linear graphics.

  if(getenv("GLINEAR"))
     {
       char *gvesa;
       (*this)[MSG_INFO] << "Trying go32's video driver \n";
       
       GrSetMode(GR_default_graphics); 
         // This is meant to enable linear graph mem
         // emulation using the MMU.
         // Video mode will be set using the VESA interrupt.
                     
       if((gvesa = getenv("GVESA")) && strcmp(gvesa,"yes"))
          {
          (*this)[MSG_INFO]    << "Using go32 graphic driver mode\n";
          (*this)[MSG_WARNING] << "Assuming 256 colors\n";
          _width  = GrSizeX();
          _height = GrSizeY(); 
          _bytes_per_pixel = 1;
          _bits_per_pixel  = 8;
          _R_mask = 3 << 4;
          _G_mask = 3 << 2;
          _B_mask = 3;
          _bytes_per_line = _width;
          _linear = 1;
          _size = _width * _height * _bytes_per_pixel;
          return 1;   
          }
     }



  // if GSVGAMODE environment variable not defined,
  // use 256 color driver.

  if(!(envmode = getenv("GSVGAMODE")))
       {
       (*this)[MSG_WARNING] << "GSVGAMODE not specified, using 320x200x256\n";
       envmode = "G320x200x256";
       }
  
  _vesa_mode = NULL;

  int i;
  for(i=0; !_vesa_mode && _vesa_table[i].type; i++)
     {
        switch(_vesa_table[i].type)
	   {
           case VESA_256:
             sprintf(smode,"G%dx%dx256",
                     _vesa_table[i].width,
                     _vesa_table[i].height);
           break;
           case VESA_32K:
             sprintf(smode,"G%dx%dx32K",
                     _vesa_table[i].width,
                     _vesa_table[i].height);
           break;
           case VESA_64K:
             sprintf(smode,"G%dx%dx64K",
                     _vesa_table[i].width,
                     _vesa_table[i].height);
           break;
           case VESA_16M:
             sprintf(smode,"G%dx%dx16M",
                     _vesa_table[i].width,
                     _vesa_table[i].height);
           break;
           default:
           (*this)[MSG_ERROR] << "Internal mode table corrupted, exiting\n";
           exit(-1);
           break;
           }
     if(!strcmp(smode, envmode))
        _vesa_mode = &(_vesa_table[i]);        
     }

  if(!_vesa_mode)
      {
      (*this)[MSG_ERROR] << "Invalid mode in GSVGAMODE\n";
      return 0;
      }

  if(VESA_SetMode(_vesa_mode->mode))
     {
     (*this)[MSG_ERROR] << "Could not set VESA mode " << smode << "\n";
     return 0;
     }

    
  _linear = (_vesa_mode->type   == VESA_256) && 
            (_vesa_mode->width  == 320)      &&
            (_vesa_mode->height == 200);

  _linear = _linear || getenv("GLINEAR");


  if(_linear)
    (*this)[MSG_INFO] << "Linear adressing mode is available. Good.\n";

  // Set graphic fields

  _width           = _vesa_mode->width;
  _height          = _vesa_mode->height;

  switch(_vesa_mode->type)
    {
    case VESA_256:
    _bytes_per_pixel = 1;
    _bits_per_pixel  = 8;
    break;
    case VESA_32K:
    _bytes_per_pixel = 2;
    _bits_per_pixel  = 15;
    break;
    case VESA_64K:
    _bytes_per_pixel = 2;
    _bits_per_pixel  = 16;
    break;
    case VESA_16M:
    _bytes_per_pixel = 3;
    _bits_per_pixel  = 24;
    break;
    }


  _bytes_per_line  = _width * _bytes_per_pixel;

  _size  = _width * _height * _bytes_per_pixel;
  _pages = _size >> 16;
  _pad   = _size & 65535;


  (*this)[MSG_INFO] << "width=" << _width << " height=" << _height << "\n";

  // I'll get this from VESA interrupt when I know how
  // specifying a segment and offset for the info buffer ...

  switch(_bits_per_pixel)
    {
    case 8:
      (*this)[MSG_INFO] << "8bpp mode\n";
      _R_mask = ((UColorCode)3) << 4;
      _G_mask = ((UColorCode)3) << 2;
      _B_mask = ((UColorCode)3);
      break;
    case 15:
      (*this)[MSG_INFO] << "15bpp HiColor mode, waoow\n";
      _R_mask = ((UColorCode)31) << 10;
      _G_mask = ((UColorCode)31) << 5;
      _B_mask = ((UColorCode)31);
      break;
    case 16:
      (*this)[MSG_INFO] << "16bpp HiColor mode, waoow\n";
      (*this)[MSG_WARNING] << "Assuming weight is 565\n";
      _R_mask = ((UColorCode)31) << 11;
      _G_mask = ((UColorCode)63) << 5;
      _B_mask = ((UColorCode)31);
      break;
    case 24:
      (*this)[MSG_INFO] << "24bpp TrueColor mode. You've got the high score\n";

      switch(_bytes_per_pixel)
	{
	case 3:
	case 4:
	  _R_mask = ((UColorCode)255) << 16;
	  _G_mask = ((UColorCode)255) << 8;
	  _B_mask = ((UColorCode)255);
	  break;
	default:
	  (*this)[MSG_WARNING] << "Unknown mode - bytes per pixel:" << _bytes_per_pixel
	                       << " bits per pixel:" << _bits_per_pixel << "\n";
	  (*this)[MSG_WARNING] << "might not work for this mode\n";
	  _R_mask = 0;
	  _G_mask = 0;
	  _B_mask = 0;
	  break;
	}

      if(getenv("RGB_MISORDERED")) // I have got a Mach32 :-)
	{                          // Set this env variable
	  _R_mask <<= 8;           // for Mach32-like boards
	  _G_mask <<= 8;           // when in 32bpp mode
	  _B_mask <<= 8;
	}
      break;

    default:
      (*this)[MSG_WARNING] << "Unknown mode - bytes per pixel:" << _bytes_per_pixel
                           << " bits per pixel:" << _bits_per_pixel << "\n";
      (*this)[MSG_WARNING] << "might not work for this mode\n";
      _R_mask = 0;
      _G_mask = 0;
      _B_mask = 0;
      break;
    }

  // show graphic endianess

  if(_verbose_level >= MSG_INFO)
    {
      fprintf(stderr,"[TAGL] %s:: bytes per pixel = %d\n", 
	      _proc_name, _bytes_per_pixel);
      fprintf(stderr,"[TAGL] %s:: R mask = ",
	      _proc_name); printb(_R_mask); fprintf(stderr,"\n");
      fprintf(stderr,"[TAGL] %s:: G mask = ",
	      _proc_name); printb(_G_mask); fprintf(stderr,"\n");
      fprintf(stderr,"[TAGL] %s:: B mask = ",
	      _proc_name); printb(_B_mask); fprintf(stderr,"\n");
    }

  return 1;
}

////
////
//
// VESA functions
//
////
////


int DOSGraphicPort::VESA_SetMode(int vmode)
{
  union REGS reg;

  reg.h.ah=0x4f;
  reg.h.al=0x02;
  reg.x.bx=vmode; 

  int86(0x10, &reg, &reg);

  if(reg.h.al != 0x4f)
     return -1;

  return reg.h.ah;
}


int DOSGraphicPort::VESA_StdMode(int vmode)
{
  union REGS reg;

  reg.h.ah=0x00;
  reg.h.al=vmode;

  int86(0x10, &reg, &reg);

  return 1;
}


int DOSGraphicPort::VESA_BankSwitch(int bank)
{
  union REGS reg;

  reg.h.ah=0x4f;
  reg.h.al=0x05;
  reg.h.bh=0x00;
  reg.h.bl=0x00; 
  reg.x.dx=bank;

  int86(0x10, &reg, &reg);

  if(reg.h.al != 0x4f)
     return -1;

  return reg.h.ah;
}

int DOSGraphicPort::VESA_SetRGB(int idx, int r, int g, int b)
{
  outportb(0x3C8,idx);
  outportb(0x3C9,r);
  outportb(0x3C9,g);
  outportb(0x3C9,b);
}

void DOSGraphicPort::VESA_InitMouse(void)
{
union REGS reg;
reg.x.ax = 0x00;
int86(0x33, &reg, &reg); // init mouse driver

// reg.x.ax = 0x07;
// reg.x.cx = 0;
// reg.x.dx = _width - 1;
// int86(0x33, &reg, &reg); // set x range

// reg.x.ax = 0x08;
// reg.x.cx = 0;
// reg.x.dx = _height - 1;
// int86(0x33, &reg, &reg); // set y range


// reg.x.ax = 0x01;
// int86(0x33, &reg, &reg);  // show cursor


}

void DOSGraphicPort::VESA_GetDeltaMouse(int *dx, int *dy)
{
   union REGS reg;
   short dxx, dyy;

   reg.x.ax = 0x0b;

   int86(0x33, &reg, &reg);  

   dxx = *(short *)&reg.x.cx;
   dyy = *(short *)&reg.x.dx;

   if(dx)
     *dx = dxx;

   if(dy)
     *dy = dyy;  
}

int DOSGraphicPort::VESA_GetMouse(int *dx, int *dy)
{
 union REGS reg;

  reg.x.ax = 0x03;

  int86(0x33, &reg, &reg);

  if(dx)
    *dx = reg.x.cx;

  if(dy)
    *dy = reg.x.dx;

 return reg.x.bx; 
}


////
////
//
// ImgBuffer utilities
//
////
////

int DOSGraphicPort::AllocateImgBuffer(void)
{
  if(_resources.Get(DGPR_IMGBUFFER))
    {
      (*this)[MSG_WARNING] << "Img buffer already allocated\n";
      return 1;
    }

  (*this)[MSG_RESOURCE] << "allocating image buffer\n";
  _v_graph_mem =  new ColorIndex[_bytes_per_line * _height];
  if(!_v_graph_mem)
    {
      (*this)[MSG_WARNING] << "could not alloc image buffer\n";
      _error_code = DGPE_MALLOC;
      return 0;
    }
  _resources.Set(DGPR_IMGBUFFER);
  return 1;
}

void DOSGraphicPort::FreeImgBuffer(void)
{
  if(!_resources.Get(DGPR_IMGBUFFER))
    {
      (*this)[MSG_WARNING] << "Img buffer already freed or not allocated\n";
      return;
    }
  (*this)[MSG_RESOURCE] << "freeing image buffer\n";
  delete[] _v_graph_mem;
  _resources.Reset(DGPR_IMGBUFFER);
}


////
////
//
// ZBuffer 
//
////
////

int DOSGraphicPort::AllocateZBuffer(void)
{
  if(_resources.Get(DGPR_ZBUFFER))
    {
      (*this)[MSG_WARNING] << "ZBuffer already allocated\n";
      return 1;
    }

  if(!(_z_mem = new ZCoord[_bytes_per_line*_height]))
    {
      (*this)[MSG_ERROR] << "could not alloc ZBuffer\n";
      _error_code = DGPE_MALLOC;
      return 0;
    }

  (*this)[MSG_RESOURCE] << "ZBuffer allocated\n";
  _resources.Set(DGPR_ZBUFFER);
  return 1;
}

void DOSGraphicPort::FreeZBuffer(void)
{
  if(!_resources.Get(DGPR_ZBUFFER))
    {
      (*this)[MSG_WARNING] << "ZBuffer already freed or not allocated\n";
      return;
    }
  (*this)[MSG_RESOURCE] << "Freeing ZBuffer\n";
  delete[] _z_mem;
  _resources.Reset(DGPR_ZBUFFER);
}


////
////
//
// Resize frame buffer & Z buffer
//
////
////


int DOSGraphicPort::Resize(const ScrCoord width, const ScrCoord height)
{
  int retval = 1;

  (*this)[MSG_INFO] << "Resizing to " << width << "x" << height << "\n";

  _width  = width;
  _height = height;
  _bytes_per_line = width;

  SetupScreen();

  if(Attributes().Get(GPA_DBUFF))
    {
      FreeImgBuffer();
      retval = AllocateImgBuffer();
    }

  if(Attributes().Get(GPA_ZBUFF))
    {
      FreeZBuffer();
      retval &= AllocateZBuffer();
    }

  _clip.Set(0,0,_width,_height);

  RestoreColortable();

  return retval;
}


////
////
//
// Colormap utilities
//
////
////

void DOSGraphicPort::RestoreColortable(void)
{
  int c;

  for(c=0; c<GP_COLORMAP_SZ; c++)
    if(_colortable[c].Stat().Get(CC_USED))
      {
	ColorComponent r,g,b;
	_colortable[c].Get(&r,&g,&b);
	MapColor(c,r,g,b);
      }
}

void DOSGraphicPort::ResetColortable(void)
{
  int c;

  for(c=0; c<GP_COLORMAP_SZ; c++)
    _colortable[c].Stat().Reset(CC_USED);
}

////
////
//
// High level functions
//
////
////

void DOSGraphicPort::MapColor(const ColorIndex idx, 
			    const ColorComponent r, 
			    const ColorComponent g, 
			    const ColorComponent b )
{
  StoreColor(idx,r,g,b);

  _colortable[idx].Stat().Set(CC_USED);
  _colortable[idx].Set(r,g,b);

  if(_bits_per_pixel == 8)
    {
      VESA_SetRGB(idx,r >> 2,g >> 2,b >> 2);
      _truecolormap[idx] = _colormap[idx];
    }
  else
    StoreColor(idx,r,g,b);
}  

void DOSGraphicPort::RGBMode(void)
{
  int r,g,b;

  (*this)[MSG_INFO] << "Switching to RGB mode\n";

  ResetColortable();

  for(r=0; r<4; r++)
     for(g=0; g<4; g++)
        for(b=0; b<4; b++)
	  MapColor(b + (g << 2) + (r << 4), r << 6, g << 6, b << 6);

  Attributes().Set(GPA_RGB);
  Attributes().Reset(GPA_CMAP);
}

void DOSGraphicPort::ColormapMode(void)
{

  (*this)[MSG_INFO] << "Switching to colormap mode\n";

  ResetColortable();

// default colors

  MapColor(BLACK,   0,   0,   0   );
  MapColor(RED,     255, 0,   0   );
  MapColor(GREEN,   0,   255, 0   );
  MapColor(YELLOW,  255, 255, 0   );
  MapColor(BLUE,    0,   0,   255 );
  MapColor(MAGENTA, 255, 0,   255 );
  MapColor(CYAN,    0,   255, 255 );
  MapColor(WHITE,   255, 255, 255 );

  Attributes().Set(GPA_CMAP);
  Attributes().Reset(GPA_RGB);
}


int DOSGraphicPort::SingleBuffer(void)
{
  if(!_linear)
    {
      (*this)[MSG_WARNING] << "SingleBuffer mode not available for non-linear framebuffers\n";
      return 0;
    }
  _graph_mem = _r_graph_mem;
  if(Attributes().Get(GPA_DBUFF))
    FreeImgBuffer();
  Attributes().Reset(GPA_DBUFF);
  (*this)[MSG_INFO] << "switching to SingleBuffer mode\n";
  return 1;
}

int DOSGraphicPort::DoubleBuffer(void)
{
  if(!Attributes().Get(GPA_DBUFF))
    {
      if(!AllocateImgBuffer())
	return 0;
    }
  Attributes().Set(GPA_DBUFF);
  _graph_mem = _v_graph_mem;
  return 1;
}

int DOSGraphicPort::SwapBuffers(void)
{
  if(Attributes().Get(GPA_DBUFF))
    {
//      vga_waitretrace();

      if(_linear)
	memcpy(_r_graph_mem, _v_graph_mem, _size);
      else
	{
	  int i;
	  char *graph_ptr = _v_graph_mem;
	  
	  for(i=0; i<_pages; i++)
	    {
	      VESA_BankSwitch(i);
	      memcpy(_r_graph_mem, graph_ptr, 65536);
	      graph_ptr += 65536;
	    }
	  if(_pad)
	    {
	      VESA_BankSwitch(_pages);
	      memcpy(_r_graph_mem, graph_ptr, _pad);
	    }
	}
    }
  return 1;
}


int DOSGraphicPort::WaitEvent(void)
{
  return 1; // not implemented yet ! 
            // Will use interrupts or something like that ...
}

char DOSGraphicPort::GetKey(void)
{
  if(kbhit())
    return getkey();
}

int  DOSGraphicPort::GetMouse(ScrCoord *xx, ScrCoord *yy)
{
  int buttons, retval = 0;
  static int x = 0;
  static int y = 0;
  int dx, dy;

  VESA_GetDeltaMouse(&dx, &dy);
//  dx >>= 3;  // Mickey to screen conversion
//  dy >>= 3;

  x += dx;
  y += dy;

  if(x < 0)
    x = 0;

  if(y < 0)
    y = 0;

  if(x > _width - 1)
    x = _width - 1;

  if(y > _height - 1)
    y = _height - 1;

  buttons = VESA_GetMouse();

// buttons are not in the same order as X

  if(buttons & 1)
    retval |= 1;
  if(buttons & 2)
    retval |= 4;
  if(buttons & 4)
    retval |= 2;

  *xx = x;
  *yy = y;

  return retval;
}

int DOSGraphicPort::ZBuffer(const int yes)
{
  if(yes && !(Attributes().Get(GPA_ZBUFF)))
    {
      AllocateZBuffer();
      Attributes().Set(GPA_ZBUFF);
    }
  else
    if(Attributes().Get(GPA_ZBUFF))
      {
	FreeZBuffer();
	Attributes().Reset(GPA_ZBUFF);
      }
}
  

int DOSGraphicPort::SetGeometry(const ScrCoord width, const ScrCoord height)
{
  return Resize(width, height);
}


GraphicPort* DOSGraphicPort::Make(char *name, 
				   ScrCoord width, ScrCoord height, int verb)
{

// There can be here a test that checks if the VESA extension is present.

  DOSGraphicPort *GP = new DOSGraphicPort(name, width, height, verb);
  if(GP->ErrorCode())
    {
      delete GP;
      return NULL;
    }
  return GP;
}

static class Stub
{
public:
  Stub(void) 
    { 
      GraphicPort::Register(DOSGraphicPort::Make,GP_VC_HANGS); 
    } 
} Dummy;

