/*
 * This software is copyrighted as noted below.  It may be freely copied,
 * modified, and redistributed, provided that the copyright notice is
 * preserved on all copies.
 *
 * There is no warranty or other guarantee of fitness for this software,
 * it is provided solely "as is".  Bug reports or fixes may be sent
 * to the author, who may or may not act on them as he desires.
 *
 * You may not include this software in a program or other software product
 * without supplying the source, or without informing the end-user that the
 * source is available for no extra charge.
 *
 * If you modify this software, you should include a notice giving the
 * name of the person performing the modification, the date of modification,
 * and the reason for such modification.
 *
 * Author:      Bruno Levy
 *
 * Copyright (c) 1996, Bruno Levy.
 *
 */
/*
 *
 * SVGAgport.C
 *
 */


#include "SVGAgport.h"
#include <stdlib.h>
#include <vgamouse.h>

/*
OpCode SVGAGraphicPort::WAITVBL = GraphicComponent::UniqueOpCode();
OpCode SVGAGraphicPort::HARDSWP = GraphicComponent::UniqueOpCode();
*/

OpCode SVGAGraphicPort::WAITVBL = 200;
OpCode SVGAGraphicPort::HARDSWP = 201;

long SVGAGraphicPort::Cntl(OpCode op, int arg)
{
  if(op == WAITVBL)
    {
      vga_waitretrace();
      return 1;
    }
  else
  if(op == HARDSWP)
    {
      (*this)[MSG_INFO] << "Setting hardware double buffer\n";

      if(!(_modeinfo->flags & CAPABLE_LINEAR))
	{
	  (*this)[MSG_WARNING] << "Failed - graph mem cannot be linear\n";
	  return 0;
	}
      
      if(_modeinfo->memory*1024 < 2*_size)
	{
	  (*this)[MSG_WARNING] << "Failed - not enough graph mem\n";
	  (*this)[MSG_WARNING] << _modeinfo->memory << "KB VRAM\n";
	  return 0;
	}

      if(_resources.Get(SGPR_IMGBUFFER))
	FreeImgBuffer();

      _resources.Set(SGPR_HARDSWP);

      _graph_mem = _r_graph_mem;

      if(Attributes().Get(GPA_DBUFF))
	vga_setdisplaystart(_size);

      return 1;
    }

  return 0;
}


SVGAGraphicPort::SVGAGraphicPort(char *name, 
				 ScrCoord width, ScrCoord height, int verbose_level): 
       GraphicPort(name, width, height, verbose_level)
{

  int i;

  _proc_name = "SVGAGraphicPort";

  _error_code = SGPE_OK;

  _width  = width;
  _height = height;

  _bits_per_pixel  = 8;
  _bytes_per_pixel = 1;

  vga_init();
  vga_setmousesupport(1);

  if(!SetupScreen())
    return;

  if(!AllocateImgBuffer())
    return;

  Attributes().Set(GPA_DBUFF);

  for(i=0; i<256; i++)
    _colormap[i] = i;

  ColormapMode();

  _clip.Set(0,0,_width-1,_height-1);

  _graph_mem = _v_graph_mem;

}


////
////
//
//   Attributes handling & destructor
//
////
////

SVGAGraphicPort::~SVGAGraphicPort(void)
{
  if(_resources.Get(SGPR_IMGBUFFER))
    FreeImgBuffer();

  if(_resources.Get(SGPR_ZBUFFER))
    FreeZBuffer();

  vga_setmode(TEXT);
}

void SVGAGraphicPort::CommitAttributes(void)
{
  Flags changed;

  changed.SetAll(_last_attributes.GetAll() ^ Attributes().GetAll());

  if(changed.Get(GPA_CMAP) && Attributes().Get(GPA_CMAP))
    ColormapMode();

  if(changed.Get(GPA_RGB)  && Attributes().Get(GPA_RGB))
    RGBMode();

  if(changed.Get(GPA_ZBUFF))
    ZBuffer(Attributes().Get(GPA_ZBUFF));

  if(changed.Get(GPA_DBUFF))
    {
      if(Attributes().Get(GPA_DBUFF))
	DoubleBuffer();
      else
	SingleBuffer();
    }
}

////
////
//
//   Set video mode 
//
////
////


int SVGAGraphicPort::SetupScreen() 
{

  // if GSVGAMODE environment variable not defined,
  // use 256 color driver.
  
  if((_mode = vga_getdefaultmode()) < 0)
    if(_width <= 320)
      _mode = G320x200x256;
    else
      if(_width <= 640)
	_mode = G640x480x256;
      else
	if(_width <= 800)
	  _mode = G800x600x256;
	else
	  _mode = G1024x768x256;

  // if required mode not supported,
  // use 320x200x256

  if(!vga_hasmode(_mode))
    {
      (*this)[MSG_WARNING] << "could not open " << _width << "*" << _height << " screen\n";
      (*this)[MSG_WARNING] << "trying 320x200x256 ...\n";

      _mode = G320x200x256;

      if(!vga_hasmode(_mode))
	{
	  (*this)[MSG_ERROR] << "seems you've got an antique graphic board...\n";
	  _error_code = SGPE_MODE;
	  return 0;
	}
    }

  vga_setmode(_mode);
  _modeinfo = vga_getmodeinfo(_mode);


  // Disable bank switching if possible

  _linear = _modeinfo->flags & CAPABLE_LINEAR;
  if(_linear)
    vga_setlinearaddressing();
    
  _linear = _linear || (_mode == G320x200x256);

  if(_linear)
    (*this)[MSG_INFO] << "Linear adressing mode is available. Good.\n";

  // Get info from SVGAlib

  _width           = _modeinfo->width;
  _height          = _modeinfo->height;
  _bytes_per_line  = _modeinfo->linewidth;
  _bytes_per_pixel = _modeinfo->bytesperpixel;
  _bits_per_pixel  = nbrbits((UColorCode)_modeinfo->colors - 1);

  _size  = _width * _height * _bytes_per_pixel;
  _pages = _size >> 16;
  _pad   = _size & 65535;

  _r_graph_mem = vga_getgraphmem();

  (*this)[MSG_INFO] << "width=" << _width << " height=" << _height << "\n";

  // Damned, SVGAlib does not provide Rmask, Gmask and Bmask.
  // Following code is inspired by grlib.

  switch(_bits_per_pixel)
    {
    case 8:
      (*this)[MSG_INFO] << "8bpp mode\n";
      _R_mask = ((UColorCode)3) << 4;
      _G_mask = ((UColorCode)3) << 2;
      _B_mask = ((UColorCode)3);
      break;
    case 15:
      (*this)[MSG_INFO] << "15bpp HiColor mode, waoow\n";
      _R_mask = ((UColorCode)31) << 10;
      _G_mask = ((UColorCode)31) << 5;
      _B_mask = ((UColorCode)31);
      break;
    case 16:
      (*this)[MSG_INFO] << "16bpp HiColor mode, waoow\n";
      (*this)[MSG_WARNING] << "Assuming weight is 565\n";
      _R_mask = ((UColorCode)31) << 11;
      _G_mask = ((UColorCode)63) << 5;
      _B_mask = ((UColorCode)31);
      break;
    case 24:
      (*this)[MSG_INFO] << "24bpp TrueColor mode. You've got the high score\n";

      switch(_bytes_per_pixel)
	{
	case 3:
	case 4:
	  _R_mask = ((UColorCode)255) << 16;
	  _G_mask = ((UColorCode)255) << 8;
	  _B_mask = ((UColorCode)255);
	  break;
	default:
	  (*this)[MSG_WARNING] << "Unknown mode - bytes per pixel:" << _bytes_per_pixel
	                       << " bits per pixel:" << _bits_per_pixel << "\n";
	  (*this)[MSG_WARNING] << "might not work for this mode\n";
	  _R_mask = 0;
	  _G_mask = 0;
	  _B_mask = 0;
	  break;
	}

      if(_modeinfo->flags & RGB_MISORDERED) // I have got a Mach32 :-)
	{
	  _R_mask <<= 8;
	  _G_mask <<= 8;
	  _B_mask <<= 8;
	}
      break;

    default:
      (*this)[MSG_WARNING] << "Unknown mode - bytes per pixel:" << _bytes_per_pixel
                           << " bits per pixel:" << _bits_per_pixel << "\n";
      (*this)[MSG_WARNING] << "might not work for this mode\n";
      _R_mask = 0;
      _G_mask = 0;
      _B_mask = 0;
      break;
    }

  // show graphic endianess

  if(_verbose_level >= MSG_INFO)
    {
      fprintf(stderr,"[TAGL] %s:: bytes per pixel = %d\n", 
	      _proc_name, _bytes_per_pixel);
      fprintf(stderr,"[TAGL] %s:: R mask = ",
	      _proc_name); printb(_R_mask); fprintf(stderr,"\n");
      fprintf(stderr,"[TAGL] %s:: G mask = ",
	      _proc_name); printb(_G_mask); fprintf(stderr,"\n");
      fprintf(stderr,"[TAGL] %s:: B mask = ",
	      _proc_name); printb(_B_mask); fprintf(stderr,"\n");
    }

  return 1;
}

////
////
//
// ImgBuffer utilities
//
////
////

int SVGAGraphicPort::AllocateImgBuffer(void)
{
  if(_resources.Get(SGPR_IMGBUFFER))
    {
      (*this)[MSG_WARNING] << "Img buffer already allocated\n";
      return 1;
    }

  (*this)[MSG_RESOURCE] << "allocating image buffer\n";
  _v_graph_mem =  new ColorIndex[_bytes_per_line * _height];
  if(!_v_graph_mem)
    {
      (*this)[MSG_WARNING] << "could not alloc image buffer\n";
      _error_code = SGPE_MALLOC;
      return 0;
    }
  _resources.Set(SGPR_IMGBUFFER);
  return 1;
}

void SVGAGraphicPort::FreeImgBuffer(void)
{
  if(!_resources.Get(SGPR_IMGBUFFER))
    {
      (*this)[MSG_WARNING] << "Img buffer already freed or not allocated\n";
      return;
    }
  (*this)[MSG_RESOURCE] << "freeing image buffer\n";
  delete[] _v_graph_mem;
  _resources.Reset(SGPR_IMGBUFFER);
}


////
////
//
// ZBuffer 
//
////
////

int SVGAGraphicPort::AllocateZBuffer(void)
{
  if(_resources.Get(SGPR_ZBUFFER))
    {
      (*this)[MSG_WARNING] << "ZBuffer already allocated\n";
      return 1;
    }

  if(!(_z_mem = new ZCoord[_bytes_per_line*_height]))
    {
      (*this)[MSG_ERROR] << "could not alloc ZBuffer\n";
      _error_code = SGPE_MALLOC;
      return 0;
    }

  (*this)[MSG_RESOURCE] << "ZBuffer allocated\n";
  _resources.Set(SGPR_ZBUFFER);
  return 1;
}

void SVGAGraphicPort::FreeZBuffer(void)
{
  if(!_resources.Get(SGPR_ZBUFFER))
    {
      (*this)[MSG_WARNING] << "ZBuffer already freed or not allocated\n";
      return;
    }
  (*this)[MSG_RESOURCE] << "Freeing ZBuffer\n";
  delete[] _z_mem;
  _resources.Reset(SGPR_ZBUFFER);
}


////
////
//
// Resize frame buffer & Z buffer
//
////
////


int SVGAGraphicPort::Resize(const ScrCoord width, const ScrCoord height)
{
  int retval = 1;

  (*this)[MSG_INFO] << "Resizing to " << width << "x" << height << "\n";

  _width  = width;
  _height = height;
  _bytes_per_line = width;

  SetupScreen();

  if(Attributes().Get(GPA_DBUFF))
    {
      FreeImgBuffer();
      retval = AllocateImgBuffer();
    }

  if(Attributes().Get(GPA_ZBUFF))
    {
      FreeZBuffer();
      retval &= AllocateZBuffer();
    }

  _clip.Set(0,0,_width,_height);

  RestoreColortable();

  return retval;
}


////
////
//
// Colormap utilities
//
////
////

void SVGAGraphicPort::RestoreColortable(void)
{
  int c;

  for(c=0; c<GP_COLORMAP_SZ; c++)
    if(_colortable[c].Stat().Get(CC_USED))
      {
	ColorComponent r,g,b;
	_colortable[c].Get(&r,&g,&b);
	MapColor(c,r,g,b);
      }
}

void SVGAGraphicPort::ResetColortable(void)
{
  int c;

  for(c=0; c<GP_COLORMAP_SZ; c++)
    _colortable[c].Stat().Reset(CC_USED);
}

////
////
//
// High level functions
//
////
////

void SVGAGraphicPort::MapColor(const ColorIndex idx, 
			    const ColorComponent r, 
			    const ColorComponent g, 
			    const ColorComponent b )
{
  StoreColor(idx,r,g,b);

  _colortable[idx].Stat().Set(CC_USED);
  _colortable[idx].Set(r,g,b);

  if(_bits_per_pixel == 8)
    {
      vga_setpalette(idx,r >> 2,g >> 2,b >> 2);
      _truecolormap[idx] = _colormap[idx];
    }
  else
    StoreColor(idx,r,g,b);
}  

void SVGAGraphicPort::RGBMode(void)
{
  int r,g,b;

  (*this)[MSG_INFO] << "Switching to RGB mode\n";

  ResetColortable();

  for(r=0; r<4; r++)
     for(g=0; g<4; g++)
        for(b=0; b<4; b++)
	  MapColor(b + (g << 2) + (r << 4), r << 6, g << 6, b << 6);

  Attributes().Set(GPA_RGB);
  Attributes().Reset(GPA_CMAP);
}

void SVGAGraphicPort::ColormapMode(void)
{

  (*this)[MSG_INFO] << "Switching to colormap mode\n";

  ResetColortable();

// default colors

  MapColor(BLACK,   0,   0,   0   );
  MapColor(RED,     255, 0,   0   );
  MapColor(GREEN,   0,   255, 0   );
  MapColor(YELLOW,  255, 255, 0   );
  MapColor(BLUE,    0,   0,   255 );
  MapColor(MAGENTA, 255, 0,   255 );
  MapColor(CYAN,    0,   255, 255 );
  MapColor(WHITE,   255, 255, 255 );

  Attributes().Set(GPA_CMAP);
  Attributes().Reset(GPA_RGB);
}


int SVGAGraphicPort::SingleBuffer(void)
{
  if(!_linear)
    {
      (*this)[MSG_WARNING] << "SingleBuffer mode not available for non-linear framebuffers\n";
      return 0;
    }
  _graph_mem = _r_graph_mem;
  if(_resources.Get(SGPR_HARDSWP))
    vga_setdisplaystart(0);
  if(Attributes().Get(GPA_DBUFF))
    FreeImgBuffer();
  Attributes().Reset(GPA_DBUFF);
  (*this)[MSG_INFO] << "switching to SingleBuffer mode\n";
  return 1;
}

int SVGAGraphicPort::DoubleBuffer(void)
{
  if(_resources.Get(SGPR_HARDSWP))
    {
      Attributes().Set(GPA_DBUFF);
      _v_graph_mem = _r_graph_mem;
      vga_setdisplaystart(_size);
      return 1;
    }

  if(!Attributes().Get(GPA_DBUFF))
    {
      if(!AllocateImgBuffer())
	return 0;
    }
  Attributes().Set(GPA_DBUFF);
  _graph_mem = _v_graph_mem;
  return 1;
}

int SVGAGraphicPort::SwapBuffers(void)
{
  if(Attributes().Get(GPA_DBUFF))
    {

      if(_resources.Get(SGPR_HARDSWP))
	{
	  if(_graph_mem == _r_graph_mem)
	    {
	      _graph_mem = _r_graph_mem + _size;
	      vga_setdisplaystart(0);
	    }
	  else
	    {
	      _graph_mem = _r_graph_mem;
	      vga_setdisplaystart(_size);
	    }
	  return 1;
	}

      if(_linear)
	memcpy(_r_graph_mem, _v_graph_mem, _size);
      else
	{
	  int i;
	  char *graph_ptr = _v_graph_mem;
	  
	  for(i=0; i<_pages; i++)
	    {
	      vga_setpage(i);
	      memcpy(_r_graph_mem, graph_ptr, 65536);
	      graph_ptr += 65536;
	    }
	  if(_pad)
	    {
	      vga_setpage(_pages);
	      memcpy(_r_graph_mem, graph_ptr, _pad);
	    }
	}
    }
  return 1;
}


int SVGAGraphicPort::WaitEvent(void)
{
  return 1; // not implemented yet ! 
            // will try with a select() call on stdin and mouse fildes ... 
}

char SVGAGraphicPort::GetKey(void)
{
  char q = vga_getkey();

  switch(q)
    {

    case 10:
      return GK_Return;
      break;

    case 27:
      q = vga_getkey();
      q = vga_getkey();
      switch(q)
	{
	case 68:
	  return GK_Left;
	  break;
	case 67:
	  return GK_Right;
	  break;
	case 65:
	  return GK_Up;
	  break;
	case 66:
	  return GK_Down;
	  break;
	default:
	  return 0;
	  break;
	}
      break;
    default:
      return q;
    }
  return q;
}

int  SVGAGraphicPort::GetMouse(ScrCoord *x, ScrCoord *y)
{
  int buttons, retval = 0;
  mouse_update();
  *x = mouse_getx();
  *y = mouse_gety();


// buttons are in reverse order than X
  buttons = mouse_getbutton();
  if(buttons & 1)
    retval |= 4;
  if(buttons & 2)
    retval |= 2;
  if(buttons & 4)
    retval |= 1;

  return retval;
}

int SVGAGraphicPort::ZBuffer(const int yes)
{
  if(yes && !(Attributes().Get(GPA_ZBUFF)))
    {
      AllocateZBuffer();
      Attributes().Set(GPA_ZBUFF);
    }
  else
    if(Attributes().Get(GPA_ZBUFF))
      {
	FreeZBuffer();
	Attributes().Reset(GPA_ZBUFF);
      }
}
  

int SVGAGraphicPort::SetGeometry(const ScrCoord width, const ScrCoord height)
{
  return Resize(width, height);
}


GraphicPort* SVGAGraphicPort::Make(char *name, 
				   ScrCoord width, ScrCoord height, int verb)
{
  SVGAGraphicPort *GP = new SVGAGraphicPort(name, width, height, verb);
  if(GP->ErrorCode())
    {
      delete GP;
      return NULL;
    }
  return GP;
}

static class Stub
{
public:
  Stub(void) 
    { 
      GraphicPort::Register(SVGAGraphicPort::Make,GP_VC_HANGS); 
    } 
} Dummy;

