/*
 * This software is copyrighted as noted below.  It may be freely copied,
 * modified, and redistributed, provided that the copyright notice is
 * preserved on all copies.
 *
 * There is no warranty or other guarantee of fitness for this software,
 * it is provided solely "as is".  Bug reports or fixes may be sent
 * to the author, who may or may not act on them as he desires.
 *
 * You may not include this software in a program or other software product
 * without supplying the source, or without informing the end-user that the
 * source is available for no extra charge.
 *
 * If you modify this software, you should include a notice giving the
 * name of the person performing the modification, the date of modification,
 * and the reason for such modification.
 *
 * Author:      Bruno Levy
 *
 * Copyright (c) 1996, Bruno Levy.
 *
 */
//
// xcontexts.cc
//
// TAGL sample program
// o Demonstrate how to bind a given window
//     to a GraphicPort.
// o Demonstrate multiple rendering 
//     contexts in the same process.
//

#include "gport.h"
#include "polyeng.h"
#include <math.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <iostream.h>
#include <stdlib.h>
#include "processors/gport/Xgport.h"

int width  = 320;
int height = 200;

Display     *display;
Window      window;
Window      subw1;
Window      subw2;
int         screen;
XVisualInfo vinfo;

void XInit(void)
{

// Step 1 : Open display

  if(!(display = XOpenDisplay("")))
   {
     cerr << "cannot open display" << endl;
     exit(-1);
   }

  screen = DefaultScreen(display);

// Step 2 : find a visual

  if(getenv("GX8BIT"))
    goto mygx8bit;

  if(XMatchVisualInfo(display, screen, 24, TrueColor, &vinfo))
        cout << "24bpp TrueColor display, you've got the high-score !!\n";
  else if(XMatchVisualInfo(display, screen, 16, TrueColor, &vinfo))
        cout << "16bpp TrueColor display, waooow !!\n";
  else if(XMatchVisualInfo(display, screen, 15, TrueColor, &vinfo))
        cout << "15bpp TrueColor display, waooow !!\n";
  else
  mygx8bit:
       if(XMatchVisualInfo(display, screen, 8, PseudoColor, &vinfo))
        cout << "8bpp PseudoColor display\n";
  else if(XMatchVisualInfo(display, screen, 8, GrayScale, &vinfo))
        cout << "8bpp GrayScale display\n";
  else
    {
      cerr << "Invalid visual type" << endl;
      exit(-1);
    }

// Step 3 : Open a window

  XSizeHints  hint;
  unsigned    long fg, bg;
 
  hint.width      = 2 * width + 8;
  hint.height     = height + 4;
  hint.min_width  = 100;
  hint.min_height = 100;
  hint.max_width  = 1024;
  hint.max_height = 1024;

  hint.flags = PSize | PMinSize | PMaxSize;
  
  // foreground and background
  
  fg = WhitePixel (display, screen);
  bg = BlackPixel (display, screen);
  
  // build window

  // colormap displays
  if(vinfo.depth <= 8)
      window = XCreateSimpleWindow (display,
				     DefaultRootWindow (display),
				     hint.x, hint.y,
				     hint.width, hint.height,
				     4, fg, bg);
  
  else
  // truecolor displays
      {
	  XSetWindowAttributes wattr;
	  int                  cmap;

	  cmap = XCreateColormap(display, DefaultRootWindow (display),
				 vinfo.visual, AllocNone);

	  wattr.colormap = cmap;
	  wattr.border_pixel = 0;

	  window = XCreateWindow       (display,
					 DefaultRootWindow (display),
					 hint.x, hint.y,
					 hint.width, hint.height,
					 0, 
					 vinfo.depth,
					 InputOutput,
					 vinfo.visual,
					 CWColormap | CWBorderPixel, &wattr);
      }

  XSelectInput(display, window, StructureNotifyMask);
  XSetStandardProperties (display, 
			  window, 
			  "Xtagl", 
			  "Xtagl", None, NULL, 0, &hint);
  

  subw1 = XCreateSimpleWindow(display, window, 0, 0, width, height, 2, fg, bg);
  XMapWindow(display, subw1);

  subw2 = XCreateSimpleWindow(display, window, width+4, 0, width, height, 2, fg, bg);
  XMapWindow(display, subw2);


  XMapWindow(display, window);

  // Wait for window map.
  while(1) 
    {
      XEvent xev;
      
      XNextEvent(display, &xev);
      if(xev.type == MapNotify && xev.xmap.event == window)
	break;
    }

  cerr << "Main window created and mapped" << endl;


  XSelectInput(display, window, KeyPressMask | StructureNotifyMask | ExposureMask 
	       | PointerMotionMask);

  XSelectInput(display, subw1, KeyPressMask | StructureNotifyMask | ExposureMask 
	       | PointerMotionMask);


  XSelectInput(display, subw2, KeyPressMask | StructureNotifyMask | ExposureMask 
	       | PointerMotionMask);
}


//
// Cube vertices
//

float vertex[8][3] =
{
   {-1000, -1000, -1000},
   {-1000, -1000,  1000},
   {-1000,  1000, -1000},
   {-1000,  1000,  1000},
   { 1000, -1000, -1000},
   { 1000, -1000,  1000},
   { 1000,  1000, -1000},
   { 1000,  1000,  1000},
};

//
// TAGL vertices (one per cube vertex)
//

GVertex Vertex[8];

//
// Cube faces
// (indexes of the vertices)
//

int face[6][4] =
{ 
  {1, 5, 7, 3},
  {2, 6, 4, 0},
  {0, 1, 3, 2},
  {4, 6, 7, 5},
  {2, 3, 7, 6},
  {0, 4, 5, 1}
};

//
// Install a color ramp between index idx1 and index idx2.
// A linear interpolation is performed between (r1,g1,b1)
// and (r2,g2,b2).

void ColorRamp(GraphicPort *GP, 
               ColorIndex idx1, 
               ColorComponent r1, ColorComponent g1, ColorComponent b1,
               ColorIndex idx2,
               ColorComponent r2, ColorComponent g2, ColorComponent b2)
	       
{
int i;
for(i=idx1; i<=idx2; i++)
  GP->MapColor(i,
               r1 + (i - (int)idx1)*(r2 - r1)/((int)idx2 - (int)idx1),
               g1 + (i - (int)idx1)*(g2 - g1)/((int)idx2 - (int)idx1),
               b1 + (i - (int)idx1)*(b2 - b1)/((int)idx2 - (int)idx1)
	      );
}


//
// Some colors
// indexes 1..6  are used for the cube (red)
//

void InitColors(PolygonEngine *PE)
{
  ColorRamp(PE->Port(), 1, 100,0,0,  6, 255,0,0);
}


//
// Draw a RGB triangle with angle r.
//

void drawtri(PolygonEngine *PE, float r)
{
double pts[3][2];

pts[0][0] = width / 2;
pts[0][1] = 30;
pts[1][0] = 30;
pts[1][1] = height - 30;
pts[2][0] = width  - 30;
pts[2][1] = height - 30;

GVertex vtx[3];

int i;   
   
for(i=0; i<3; i++)
   {
      double x1 = pts[i][0] - width/2;
      double y1 = pts[i][1] - height/2;
      double x2 =  cos(r) * x1 + sin(r) * y1;
      double y2 = -sin(r) * x1 + cos(r) * y1;
      
      vtx[i].x = (int)x2 + width  / 2;
      vtx[i].y = (int)y2 + height / 2;
   }
   
vtx[0].r = 255;
vtx[0].g = 0;
vtx[0].b = 0;

vtx[1].r = 0;
vtx[1].g = 255;
vtx[1].b = 0;

vtx[2].r = 0;
vtx[2].g = 0;
vtx[2].b = 255;

PE->Reset();
for(i=0; i<3; i++)
  PE->Push(&(vtx[i]));
PE->FillPoly();  
}


//
// Draw a face of the cube 
// (+ backface culling)
//

void drawface(PolygonEngine *PE, int face_idx, ColorCode color)
{

// compute Z coordinate of the cross product
// of two vectors of the face. If < 0, it's
// a backface -> cull.

int x0 = Vertex[face[face_idx][0]].x;
int x1 = Vertex[face[face_idx][1]].x;
int x2 = Vertex[face[face_idx][2]].x;

int y0 = Vertex[face[face_idx][0]].y;
int y1 = Vertex[face[face_idx][1]].y;
int y2 = Vertex[face[face_idx][2]].y;

int z = (x0 - x1) * (y2 - y1) - 
        (x2 - x1) * (y0 - y1) ;

if(z > 0)
   {
   PE->VAttributes().c = color << D_SHIFT;
   PE->Reset();
   for(int i=0; i<4; i++)
   PE->Push(&(Vertex[face[face_idx][i]]));
   PE->FillPoly();
   }
}


//
// Draw a cube, with rotations rx,ry,rz. 
//

void drawcube(PolygonEngine *PE, float rx, float ry, float rz, ColorCode color)
{
   float x1,y1,z1;
   float x2,y2,z2;  
   int i;

   for(i=0; i<8; i++)
      {
      x1 = vertex[i][0]; 
      y1 = vertex[i][1];
      z1 = vertex[i][2];
      
      x2 = x1;
      y2 =  y1 * cos(rx) + z1 * sin(rx);
      z2 = -y1 * sin(rx) + z1 * cos(rx);
      
      x1 =  x2 * cos(ry) + z2 * sin(ry);
      y1 = y2;
      z1 = -x2 * sin(ry) + z2 * cos(ry);

      x2 =  x1 * cos(rz) + y1 * sin(rz);
      y2 = -x1 * sin(rz) + y1 * cos(rz);
      z2 = z1;

      Vertex[i].x = ((int)x2 * width  / (320 * 16)) + width  / 2;
      Vertex[i].y = ((int)y2 * height / (200 * 16)) + height / 2;
      Vertex[i].z = ((int)z2 + 2000) / 4;
      }
      
   
   for(i=0; i<6; i++)
      drawface(PE, i, i + color); 
}


int main(int argc, char **argv)
{
   double r = 0.0;
   
   XInit();

   cerr << "X11 windows created" << endl;

   XGraphicPort  *GP1 = new XGraphicPort("Context1: cube (Colormap mode)",
                                          display, &vinfo);
		  GP1->Bind(subw1);			  
   PolygonEngine *PE1 = PolygonEngine::Make(GP1);

   InitColors(PE1);
   
   GP1->SaveContext();
   
   XGraphicPort   *GP2 = new XGraphicPort("Context2: triangle (RGB mode)", 
                                          display, &vinfo);
		  GP2->Bind(subw2);			  
   PolygonEngine *PE2 = PolygonEngine::Make(GP2);
   GP2->RGBMode();
   PE2->RGBMode();
   PE2->Gouraud();

   GP2->SaveContext();

   int btn=0;
   int k=0;

   while(!k)
     {
       int x,y;

       GP2->SaveContext();
       GP1->SetContext();

       btn = GP1->GetMouse(&x,&y);
       k   = GP1->GetKey();

       GP1->GetGeometry(&width, &height);
       PE1->Port()->Clear((ColorIndex)0);
       drawcube(PE1,  (float)y/100.0,  -(float)x/100.0,  0, 1);
       PE1->Port()->SwapBuffers();

       GP1->SaveContext();
       GP2->SetContext();

       k |= GP2->GetKey();

       GP2->GetGeometry(&width, &height);
       PE2->Port()->Clear(30,30,30);
       drawtri(PE2,  r);
       r += 0.1;
       PE2->Port()->SwapBuffers();
     }
   
   delete PE1;
//   GP1->SetContext();
   delete GP1;
   delete PE2;
//   GP2->SetContext();
   delete GP2;

   return 0;
}

