/*
 * This software is copyrighted as noted below.  It may be freely copied,
 * modified, and redistributed, provided that the copyright notice is
 * preserved on all copies.
 *
 * There is no warranty or other guarantee of fitness for this software,
 * it is provided solely "as is".  Bug reports or fixes may be sent
 * to the author, who may or may not act on them as he desires.
 *
 * You may not include this software in a program or other software product
 * without supplying the source, or without informing the end-user that the
 * source is available for no extra charge.
 *
 * If you modify this software, you should include a notice giving the
 * name of the person performing the modification, the date of modification,
 * and the reason for such modification.
 *
 * Author:      Bruno Levy
 *
 * Copyright (c) 1996, Bruno Levy.
 *
 */
//
// cube.cc
//
// TAGL sample program
// Rotate a cube with the mouse
// 

#include "gport.h"
#include "polyeng.h"
#include <math.h>
#include <time.h>

int width  = 320;
int height = 200;

int size;

//
// TGA reading
//


char TGA_Signature[] = { 0x00, 0x00, 0x02, 0x00, 0x00, 
                         0x00, 0x00, 0x00, 0x00, 0x00 };
                         

typedef struct
{
unsigned char Signature[10];
unsigned char FirstLine_L;
unsigned char FirstLine_H;
unsigned char Width_L;
unsigned char Width_H;
unsigned char Height_L;
unsigned char Height_H;
unsigned char bpp;
unsigned char mode;
} TGA_Header;

typedef struct
{
unsigned char B;
unsigned char G;
unsigned char R;
} TGA_Pixel;


UColorCode*
TexRead(char* filename, GraphicPort* GP)
{
   FILE* f = fopen(filename,"r");
   TGA_Header H;
   TGA_Pixel *Line;
   int x,y;
   
   
   if(!f)
     {
	cerr << "Cannot open " << filename << endl;
	return 0;
     }
   
   fread((char *)&H, sizeof(TGA_Header), 1, f);
   
   if(memcmp(TGA_Signature, H.Signature, (unsigned int)sizeof(TGA_Signature)))
     {
	cerr << "invalid TGA signature" << endl;
	return 0;
     }
   
   if(H.mode != 32)
     {
	cerr << "Sorry, vtga cannot handle this kind of TGA file" << endl;
	return 0;
     }

   if(H.bpp != 24)
     {
	cerr << "Sorry, vtga can handle 24bpp TGA files only" << endl;
	return 0;
     }

   int Width     = (int)H.Width_L     + (((int)H.Width_H)     << 8);
   int Height    = (int)H.Height_L    + (((int)H.Height_H)    << 8);
   int FirstLine = (int)H.FirstLine_L + (((int)H.FirstLine_H) << 8);
   
   size = Width;
   
   GTexel* texture = new GTexel[size * size];
   memset(texture, 0, size*size*sizeof(GTexel));
   Line = new TGA_Pixel[Width];
   
   for(y = 0; (y < Height) && (y < size); y++)
     {
	fread((char *)Line, Width * sizeof(TGA_Pixel), 1, f);
	
	for(x=0; (x < Width) && (x < size); x++)
	  {
	     texture[y*size+x].r = Line[x].R;
	     texture[y*size+x].g = Line[x].G;
	     texture[y*size+x].b = Line[x].B;	     
          }
   
   }
   
   delete[] Line;
   
   return (UColorCode*)texture;
   
}


//
// Texture mapping.
//

int tex_coords[4][2] =
{
     {0, 0},
     {0, 1},
     {1, 1},
     {1, 0}
};

//
// Cube vertices.
//

float vertex[8][3] =
{
   {-1000, -1000, -1000},
   {-1000, -1000,  1000},
   {-1000,  1000, -1000},
   {-1000,  1000,  1000},
   { 1000, -1000, -1000},
   { 1000, -1000,  1000},
   { 1000,  1000, -1000},
   { 1000,  1000,  1000},
};


//
// face colors
//


float colors[6][3] =
{
     {1.0, 0.0, 0.0},
     {0.0, 1.0, 0.0},
     {0.0, 0.0, 1.0},
     {1.0, 1.0, 0.0},
     {1.0, 0.0, 1.0},
     {0.0, 1.0, 1.0}
};

//
// TAGL vertices (one per cube vertex)
//

GVertex Vertex[8];

//
// Cube faces
// (indexes of the vertices)
//

int face[6][4] =
{ 
  {1, 5, 7, 3},
  {2, 6, 4, 0},
  {0, 1, 3, 2},
  {4, 6, 7, 5},
  {2, 3, 7, 6},
  {0, 4, 5, 1}
};

//
// Draw a face of the cube 
// (+ backface culling)
//

void drawface(PolygonEngine *PE, int face_idx, ColorCode color)
{

// compute Z coordinate of the cross product
// of two vectors of the face. If < 0, it's
// a backface -> cull.

int x0 = Vertex[face[face_idx][0]].x;
int x1 = Vertex[face[face_idx][1]].x;
int x2 = Vertex[face[face_idx][2]].x;

int y0 = Vertex[face[face_idx][0]].y;
int y1 = Vertex[face[face_idx][1]].y;
int y2 = Vertex[face[face_idx][2]].y;

int z = (x0 - x1) * (y2 - y1) - 
        (x2 - x1) * (y0 - y1) ;

if(z > 0)
   {
   PE->VAttributes().c = color << D_SHIFT;
   PE->Reset();
   for(int i=0; i<4; i++)
	{
	   Vertex[face[face_idx][i]].X = tex_coords[i][0];
	   Vertex[face[face_idx][i]].Y = tex_coords[i][1];	   
	   PE->Push(&(Vertex[face[face_idx][i]]));
	}
   PE->FillPoly();
   }
}

void draw(PolygonEngine *PE, float rx, float ry, float rz, ColorCode color)
{
   float x1,y1,z1;
   float x2,y2,z2;  
   int i;

   for(i=0; i<8; i++)
      {
      x1 = vertex[i][0]; 
      y1 = vertex[i][1];
      z1 = vertex[i][2];
      
      x2 = x1;
      y2 =  y1 * cos(rx) + z1 * sin(rx);
      z2 = -y1 * sin(rx) + z1 * cos(rx);
      
      x1 =  x2 * cos(ry) + z2 * sin(ry);
      y1 = y2;
      z1 = -x2 * sin(ry) + z2 * cos(ry);

      x2 =  x1 * cos(rz) + y1 * sin(rz);
      y2 = -x1 * sin(rz) + y1 * cos(rz);
      z2 = z1;

      Vertex[i].x = ((int)x2 * width  / (320 * 16)) + width  / 2;
      Vertex[i].y = ((int)y2 * height / (200 * 16)) + height / 2;
      Vertex[i].z = ((int)z2 + 2000) / 4;
      }
      
   
   for(i=0; i<6; i++)
     {
	PE->VAttributes().r = (int)(colors[i][0] * 255.0);
	PE->VAttributes().g = (int)(colors[i][1] * 255.0);
	PE->VAttributes().b = (int)(colors[i][2] * 255.0);	
	drawface(PE, i, i + color);
     }
}


int main(int argc, char **argv)
{
   
   if(argc != 2)
     {
	cerr << argv[0] << ":invalid # of args" << endl;
	cerr << "Usage: " << argv[0] << " <texture_file.tga>" << endl;
	exit(-1);
     }
   
   
   GraphicPort   *GP = GraphicPort::Make(argv[0], width, height);
   PolygonEngine *PE = PolygonEngine::Make(GP);
   
   GP->RGBMode();
   PE->RGBMode();

   UColorCode* texture = TexRead(argv[1], GP);
   GP->TextureBind(texture, size);
   PE->Texture(1);
   
   int btn=0;
   int k=0;

   int now = time(NULL);
   int frames = 0;

   for(int i=0; i<4; i++)
     {
	tex_coords[i][0] *= size;
	tex_coords[i][1] *= size;
     }
   
   while(!k)
     {
	int x,y;
	btn = GP->GetMouse(&x,&y);
	k   = GP->GetKey();
	
	GP->GetGeometry(&width, &height);
	PE->Port()->Clear(10,10,10);
	draw(PE,  (float)y/100.0,  -(float)x/100.0,  0, 1);
	PE->Port()->SwapBuffers();

	frames++;
     }
   
   now = time(NULL) - now;
   
   cerr << "fps: " << (float)frames / (float)now << endl;
   
   delete PE;
   delete GP;
   return 0;
}

