/*
 * This software is copyrighted as noted below.  It may be freely copied,
 * modified, and redistributed, provided that the copyright notice is
 * preserved on all copies.
 *
 * There is no warranty or other guarantee of fitness for this software,
 * it is provided solely "as is".  Bug reports or fixes may be sent
 * to the author, who may or may not act on them as he desires.
 *
 * You may not include this software in a program or other software product
 * without supplying the source, or without informing the end-user that the
 * source is available for no extra charge.
 *
 * If you modify this software, you should include a notice giving the
 * name of the person performing the modification, the date of modification,
 * and the reason for such modification.
 *
 * Author:      Bruno Levy
 *
 * Copyright (c) 1996, Bruno Levy.
 *
 */
//
// cube.cc
//
// TAGL sample program
// Rotate a cube with the mouse
// 

#include "gport.h"
#include "polyeng.h"
#include <math.h>

int width  = 320;
int height = 200;

//
// Cube vertices
//

float vertex[8][3] =
{
   {-1000, -1000, -1000},
   {-1000, -1000,  1000},
   {-1000,  1000, -1000},
   {-1000,  1000,  1000},
   { 1000, -1000, -1000},
   { 1000, -1000,  1000},
   { 1000,  1000, -1000},
   { 1000,  1000,  1000},
};

//
// TAGL vertices (one per cube vertex)
//

GVertex Vertex[8];

//
// Cube faces
// (indexes of the vertices)
//

int face[6][4] =
{ 
  {1, 5, 7, 3},
  {2, 6, 4, 0},
  {0, 1, 3, 2},
  {4, 6, 7, 5},
  {2, 3, 7, 6},
  {0, 4, 5, 1}
};

//
// Install a color ramp between index idx1 and index idx2.
// A linear interpolation is performed between (r1,g1,b1)
// and (r2,g2,b2).

void ColorRamp(GraphicPort *GP, 
               ColorIndex idx1, 
               ColorComponent r1, ColorComponent g1, ColorComponent b1,
               ColorIndex idx2,
               ColorComponent r2, ColorComponent g2, ColorComponent b2)
	       
{
int i;
for(i=idx1; i<=idx2; i++)
  GP->MapColor(i,
               r1 + (i - (int)idx1)*(r2 - r1)/((int)idx2 - (int)idx1),
               g1 + (i - (int)idx1)*(g2 - g1)/((int)idx2 - (int)idx1),
               b1 + (i - (int)idx1)*(b2 - b1)/((int)idx2 - (int)idx1)
	      );
}


//
// Some colors
// indexes 1..6  are used for the cube (red)
//

void InitColors(PolygonEngine *PE)
{
  ColorRamp(PE->Port(), 1, 100,0,0,  6, 255,0,0);
}

//
// Draw a face of the cube 
// (+ backface culling)
//

void drawface(PolygonEngine *PE, int face_idx, ColorCode color)
{

// compute Z coordinate of the cross product
// of two vectors of the face. If < 0, it's
// a backface -> cull.

int x0 = Vertex[face[face_idx][0]].x;
int x1 = Vertex[face[face_idx][1]].x;
int x2 = Vertex[face[face_idx][2]].x;

int y0 = Vertex[face[face_idx][0]].y;
int y1 = Vertex[face[face_idx][1]].y;
int y2 = Vertex[face[face_idx][2]].y;

int z = (x0 - x1) * (y2 - y1) - 
        (x2 - x1) * (y0 - y1) ;

if(z > 0)
   {
   PE->VAttributes().c = color << D_SHIFT;
   PE->Reset();
   for(int i=0; i<4; i++)
   PE->Push(&(Vertex[face[face_idx][i]]));
   PE->FillPoly();
   }
}

void draw(PolygonEngine *PE, float rx, float ry, float rz, ColorCode color)
{
   float x1,y1,z1;
   float x2,y2,z2;  
   int i;

   for(i=0; i<8; i++)
      {
      x1 = vertex[i][0]; 
      y1 = vertex[i][1];
      z1 = vertex[i][2];
      
      x2 = x1;
      y2 =  y1 * cos(rx) + z1 * sin(rx);
      z2 = -y1 * sin(rx) + z1 * cos(rx);
      
      x1 =  x2 * cos(ry) + z2 * sin(ry);
      y1 = y2;
      z1 = -x2 * sin(ry) + z2 * cos(ry);

      x2 =  x1 * cos(rz) + y1 * sin(rz);
      y2 = -x1 * sin(rz) + y1 * cos(rz);
      z2 = z1;

      Vertex[i].x = ((int)x2 * width  / (320 * 16)) + width  / 2;
      Vertex[i].y = ((int)y2 * height / (200 * 16)) + height / 2;
      Vertex[i].z = ((int)z2 + 2000) / 4;
      }
      
   
   for(i=0; i<6; i++)
      drawface(PE, i, i + color); 
}


int main(int argc, char **argv)
{
   GraphicPort   *GP = GraphicPort::Make(argv[0], width, height);
   PolygonEngine *PE = PolygonEngine::Make(GP);
   
   InitColors(PE);

   int btn=0;
   int k=0;

   while(!k)
   {
   int x,y;
   btn = GP->GetMouse(&x,&y);
   k   = GP->GetKey();

   GP->GetGeometry(&width, &height);
   PE->Port()->Clear((ColorIndex)0);
   draw(PE,  (float)y/100.0,  -(float)x/100.0,  0, 1);
   PE->Port()->SwapBuffers();
   }
   
   delete PE;
   delete GP;
   return 0;
}

