#ifndef PENGUINPLAY_MESSAGE_COMMON
#define PENGUINPLAY_MESSAGE_COMMON

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef HAVE_VOID_RETURN_TYPE
#define VR_RETURN return
#else
#define VR_RETURN
#endif

#ifdef HAVE_PARTIAL_SPECIALIZATION
#define SP_RETURN return
#else
#define SP_RETURN
#endif


//
// Connections are stored to two lists - one at the message and one at
// the receiver.
//
// This object must know how to disconnect it from both lists!
// That's why we have this alittle odd looking list implementations here.
//

/*
 * This class basically forms the connection between message and receiver.
 *
 * It's a member of two circular linked lists - one maintained by the
 * signal and the other belonging to the receiver.
 *
 */

class ppMsgConnection
{
public:
    // first list is stored at messages
    ppMsgConnection *next;
    ppMsgConnection *prev;
    // 2nd list is stored at ppMessageBase
    ppMsgConnection *snext;
    ppMsgConnection *sprev;



    ppMsgConnection ()
	    : next(0),prev(0),snext(0),sprev(0)
    {   }


    void disconnect ()
    {
	if (next) {
	    next->prev = prev;
	    prev->next = next;
	}
	if (snext) {
	    snext->sprev = sprev;
	    sprev->snext = snext;
	}
    }
};




#ifdef HAVE_MUTABLE
#define PP_CONST const
#define PP_MUTABLE mutable
#else
#define PP_CONST
#define PP_MUTABLE
#endif


/*
 * An handle to a given connection allowing disconnection by the user
 * at any time.
 * This is an object given to user -- user can store it, mess with it
 * and call disconnect() for it. -- or user can just dump this object
 * and the disconnect is made automatically either at ppMessageBase's
 * destructor or Message's destructor. (actually ppMessageList/ppMessageSList)
 */

class ppMsgCHandle
{
private:
	PP_MUTABLE ppMsgConnection *c;

public:
  // can assign this to another object, but then the original loses
  // its ability to disconnect the message -- This is for making it
  // possible to pass ppMsgCHandle to the user.

	void operator=(PP_CONST ppMsgCHandle &obj)
	{
		c     = obj.c;
		obj.c = 0;
	}

	ppMsgCHandle (const ppMsgCHandle &conn)
	{
		ppMsgCHandle &co = const_cast<ppMsgCHandle&> (conn);
		c    = co.c;
		co.c = 0;
	}

	ppMsgCHandle (ppMsgConnection *c_) : c(c_)
	{ }

	ppMsgCHandle () : c(0)
	{ }

	void disconnect () PP_CONST
	{
		if (c)
		{
			c->disconnect();
			delete c;
			c = 0;
		}
	}
};






// This is a circular list for ppMsgConnection-items

class ppMessageList
{
private:
	ppMsgConnection enditem; // one extra element for marking start and end

public:
	ppMsgConnection * begin ()
	{ return enditem.next; }

	ppMsgConnection * end ()
	{ return &enditem; }

	void insert(ppMsgConnection *item)
	{
		item->next         = enditem.next;
		item->prev         = &enditem;
		enditem.next->prev = item;
		enditem.next       = item;
	}
	// ppMsgConnection -object handles disconnection from the list.

	ppMessageList()
	{
		enditem.next = &enditem;
		enditem.prev = &enditem;
	}

	~ppMessageList()
	{
		while (begin () != end ())
		{
			ppMsgConnection *c = begin ();
			c->disconnect (); // remove from the lists.
			delete c; // delete the object.
		}
	}
};






// This is a circular list for ppMsgConnection-items -- uses snext/sprev

class ppMessageSList
{
private:
	ppMsgConnection enditem; // one extra element for marking start and end

public:
	ppMsgConnection * begin ()
	{ return enditem.snext; }

	ppMsgConnection * end ()
	{ return &enditem; }

	void insert (ppMsgConnection *item)
	{
		item->snext          = enditem.snext;
		item->sprev          = &enditem;
		enditem.snext->sprev = item;
		enditem.snext        = item;
	}

	// ppMsgConnection -object handles Connection from the list.

	ppMessageSList()
	{
		enditem.snext = &enditem;
		enditem.sprev = &enditem;
	}

	~ppMessageSList()
	{
		while (begin () != end ())
		{
			ppMsgConnection *c=begin ();
			c->disconnect (); // remove from the lists.
			delete c; // delete the object.
		}
	}
};




/*
 * The base class required for all potential signal receivers.
 * Use always virtual inheritance for it, or your code will behave
 * really strange!
 */

class ppMessageBase
{
private:
	ppMessageSList connectionlist; // support for ppMessageN<> requires this.

public:
	void insert_connection (ppMsgConnection *item)
	{ connectionlist.insert(item); }

	virtual ~ppMessageBase ()
	{ }
};



#endif
