/************************************************************************
    ppgGGIDevice.cc:  Defines ppgGGIDevice.


    ppgGGIDevice:
        Represents a GGI based display device.


    Part of the PenguinPlay 2d library.  Please see LICENSE.TXT supplied
    with this library for copyright details.

    Copyright (C)    Adrian Ratnapala 1998.


    $Author: raka $
    $Date: 1998/11/25 10:06:31 $
    $Revision: 1.1 $
************************************************************************/


#include "ppgCommon.h"
#include <PenguinPlay/ppgGGISurface.h>
#include <PenguinPlay/ppgColour.h>
#include <PenguinPlay/ppgMode.h>
#include <PenguinPlay/ppgGC.h>
#include <PenguinPlay/ppgGGICommands.h>

/*
 * FIX: this is only temporary, GGI should be checking dlopen, not us.
 */
#include <dlfcn.h>




/**********************************************************************
 * Adiministrivia
 */
 
ppgGGIDevice::ppgGGIDevice()  
{
  /*
   * FIX: Check the return code.
   */
  if ( !(ggi_visual = ggiOpen(NULL))){
	ppFatalError("dlerror: %s\n", dlerror());
  }
  ggiSetFlags(ggi_visual, GGIFLAG_ASYNC);
};       

ppgGGIDevice::~ppgGGIDevice()
{
  ggiClose(ggi_visual);
}
  


/************************************************************
 * Mode setting
 */

/*
 * various utility funcs for the mode setting methods. 
 */ 

/* 
 * Get a graphtype associated with a certain depth.
 * (Will change the depth to a supported one).
 */

inline ggi_graphtype depth_to_graphtype(int depth)
{
  //round depth to nearest graphtype depth supported by GGI.
  #define ROUND_TO_GTYPE(MIN, DEPTH)\
    if(depth > MIN){ depth=DEPTH; return GT_##DEPTH##BIT;}

  ROUND_TO_GTYPE(24, 32);
  ROUND_TO_GTYPE(16, 24);
  ROUND_TO_GTYPE(15, 16);
  ROUND_TO_GTYPE( 8, 15);
  ROUND_TO_GTYPE( 4,  8);
  ROUND_TO_GTYPE( 1,  4);
  ROUND_TO_GTYPE( 1,  1);
  
  
  //if we are still around, that means bpp is unset (non +ive for now).
  //so we use defaults.
  return GT_AUTO;
}

inline int graphtype_to_depth(ggi_graphtype gt)
{
  return GT_DEPTH_MASK & gt; //that was pathetic wasn't it.
}

void pg_mode_to_ggi_mode(ppgMode& mode, ggi_mode& gmode)
{
//get either a given mode variable, or GGI_AUTO if var is unset. 
#define MV(V)(	       			   \
    (mode.Get##V()!=mode.GetUnset##V()) ?  \
	 mode.Get##V():GGI_AUTO            \
  )
  //Should we use the GGI_DEFMODE variable instead?  Must read the
  //GGI docs.
  gmode.frames=1; //Should this be GGI_AUTO?
  gmode.visible.x = MV(Width);
  gmode.visible.y = MV(Height);
  gmode.virt.x = MV(VirtWidth);
  gmode.virt.y = MV(VirtHeight);
  gmode.dpp.x = GGI_AUTO;
  gmode.dpp.y = GGI_AUTO;
  gmode.size.x = GGI_AUTO;
  gmode.size.y = GGI_AUTO;
  gmode.graphtype=(
		   (mode.GetColourDepth()!=mode.GetUnsetColourDepth())?
		   depth_to_graphtype(mode.GetColourDepth()) :
		   GGI_AUTO
  );
#undef MV
};

void ggi_mode_to_pg_mode(ppgMode& mode, ggi_mode& gmode)
{
  int depth;
  mode.SetWidth(gmode.visible.x);
  mode.SetHeight(gmode.visible.y);
  mode.SetVirtWidth(gmode.virt.x);
  mode.SetVirtHeight(gmode.virt.y);
  mode.SetColourDepth(depth = graphtype_to_depth(gmode.graphtype));
  mode.SetBytesPP( (0x3 & depth) ? (depth / 4 + 1) : depth / 4 );
    //FIX this is worng, just enough bytes to fit the bits in? Stupid.
    //should we be going for the smallest sufficient power of two, or
    //trying to be yet more correct.
};

void undo_set_ggi_mode(ggi_visual_t visual)
{
  //FIX:  FILL THIS IN PLEASE!
}



ppgColourMode& make_colour_mode(ggi_visual_t vis) 
{
  /* 
   * FIX: We assume an RGB graphics mode.  CI modes need design work...
   *
   * FIX: Maybe we can make ppgColourMode store a ggi_pixelformat outright.
   */
 
  //FIX: Some error checking.
  const ggi_pixelformat* pix_fmt= ggiGetPixelFormat(vis);

  return *new ppgRGBAMode(
    pix_fmt->depth,
	pix_fmt->red_mask,
	pix_fmt->green_mask,
	pix_fmt->blue_mask,
	pix_fmt->alpha_mask
  );	
}


/*
 * Build an appropriate surface and surface class  to represent the whole disp.
 */
ppgOutputSurface& build_main_surface(
         ggi_visual_t ggi_visual, 
         ppgGGIDevice& dev,
	 const ppgMode& mode
     )
{
  dev.SetColourMode(make_colour_mode(ggi_visual));

  /*
   * Hmm, what do we do about the visible size?
   */
  ppgGGISurface& surf = *new ppgGGISurface(
    dev,                       //surface class
    0, 0,                      //origin of surface within visual.
    mode.GetVirtWidth(), mode.GetVirtHeight()
  );
  return surf;
}



bool ppgGGIDevice::QueryMode(ppgMode& mode)
{
  bool retval;
  ggi_mode gmode;
  
  pg_mode_to_ggi_mode(mode, gmode);
  retval=ggiCheckMode(GetGGIVisual(), &gmode);
  ggi_mode_to_pg_mode(mode, gmode);
  return retval;
}

void ppgGGIDevice::SetMode(ppgMode& mode) 
{
  {
    ggi_mode gmode;
    pg_mode_to_ggi_mode(mode, gmode);
    if( ggiSetMode( GetGGIVisual(), &gmode) ) 
      throw eNoMode("ggiSetMode Failed.");
    ggi_mode_to_pg_mode(mode, gmode);
   }

  /*
   * Make some surfaces and classes to go.
   */
   
  SetMainSurface(build_main_surface(GetGGIVisual(), *this, mode));
}




/************************************************************
 * Day to day ops.
 */



/*
 * Makes what's on screen equal what you expect.
 */
void ppgGGIDevice::Flush()
{
  ggiFlush(GetGGIVisual());
}

/*
 * Need to call this before drawing into the display.
 * ONLY the specified gc is guarnteed to work correctly.
 */
void ppgGGIDevice::BeginDraw(ppgGC& gc)
{
  //gc.SetDevice(*this);
  //The above is not needed if a GC is stuck on a single device.  Mus think.
  /*
   * FIX: Set up some GC state.
   */
}

void ppgGGIDevice::EndDraw(){}



void ppgGGIDevice::InitializeGC(ppgGC& gc)const
{
  /*
   * And Java thinks it doesn't need a preprocessor.
   */
  #define SET_HANDLER(hand)\
    gc.SetHandler(\
     ppgDrawCmd::dc##hand,\
     (ppgDrawCmd::Handler)ppgGGIDrawCmds::hand\
    )
  #define SET_EMPTY(hand)\
    gc.SetHandler(\
     ppgDrawCmd::dc##hand,\
     (ppgDrawCmd::Handler)ppgDrawCmd::EmptyHandler\
    )

  /*
   * Set the actual pointers to draw command handlers.
   */
  SET_HANDLER(PixBlit);
  SET_HANDLER(SetForeground);
  SET_HANDLER(SetBackground);
  SET_HANDLER(DrawHLine);
  
  
  #undef SET_HANDLER
  #undef SET_EMPTY
};




