/************************************************************************
    ppgGC.cc:  Defines ppgGC.


    ppgGC:
        Holds context data for drawing ops as well as providing
        the standard interface to them.


    Part of the PenguinPlay 2d library.  Please see LICENSE.TXT supplied
    with this library for copyright details.

    Copyright (C)    Adrian Ratnapala 1998.


    $Author: raka $
    $Date: 1998/11/25 10:06:28 $
    $Revision: 1.7 $
************************************************************************/


#include "ppgCommon.h"
#include <PenguinPlay/ppgColour.h>
#include <PenguinPlay/ppgGC.h>


/*************************************************************************
 *  Some nice constructors.
 */

//common behaviour of all the nice consturctors.
void init_gc(ppgGC& gc)
{
#ifdef PP_DEBUG
   for(int i=0; i < ppgDrawCmd::dcNumVals; i++)
     gc.SetHandler((ppgDrawCmd::CmdType)i, ppgDrawCmd::NullHandler);
#endif //PP_DEBUG

  #define SET_HANDLER(hand)\
    gc.SetHandler(\
     ppgDrawCmd::dc##hand,\
     (ppgDrawCmd::Handler)ppgDrawCmd::hand\
    )

  
  SET_HANDLER(SpanBlit);
  SET_HANDLER(SpanBlitRaw);

  SET_HANDLER(Blit);
  SET_HANDLER(BlitRaw);

  SET_HANDLER(DrawHLine);
  SET_HANDLER(DrawVLine);
 
  #undef SET_HANDLER

}

ppgGC::ppgGC(const ppgDevice& dev)  
{
  device=const_cast<ppgDevice*>(&dev);
  curr_surf=NULL; //FIX: get rid of this stuff.
  init_gc(*this);
}


ppgGC::~ppgGC()
{
}

 

/**************************************************************************
 * Some nice user freindly draw command wrappers.
 * (The ones which are too big to inline).
 */
 
#include "ppgCppDefs.inc"

//unpacks "in" to get the BUFF_INFO to do one of the above blits.
void ppgGC::Blit(const ppgLinearSurface& in, COORDS)const
{
  int skip = in.GetScanlineSkip();
  return Blit(
    (ppgPixel*)((char*)in.GetMemBlock() + in_y*skip) + in_x,
    skip,
    out_x, out_y,
    w, h
  );
}

//unpacks "in" to get the BUFF_INFO to do one of the above blits.
void ppgGC::BlitRaw(const ppgLinearSurface& in, COORDS)const
{
  int skip = in.GetScanlineSkip();
  return BlitRaw(
    (char*)in.GetMemBlock() + 
        in_y*skip + 
	in_x/GetDevice().GetColourMode().GetBytesPP(),
    skip,
    out_x, out_y,
    w, h
  );
}


#include "ppgCppUndefs.inc"

/************************************************
 *	Debugging stuff.
 */

#ifdef PP_DEBUG
void ppgGC::dbDump(FILE* out)const
{
  ppBase::dbDump(out);

  fprintf(out, "The Drawcommand handlers are: \n"
    "Null:             %p\n"
	"\n"
    "PixBlit :         %p\n" 
    "PixBlit Raw:      %p\n"
    "\n"
    "Blit:             %p\n" 
    "Blit Raw:         %p\n"
    "\n"
    "SpanBlit:         %p\n"
    "SpanBlit Raw:     %p\n"
    "\n"
    "TransBlit:        %p\n"
    "TransBlit Raw:    %p\n"
    "\n"
    "Clear             %p\n"
    "\n"
    "Draw Point        %p\n"
    "Draw Line         %p\n"
    "Draw HLine        %p\n"
    "Draw VLine        %p\n"
    "Draw Poly         %p\n"
    "Draw Rect         %p\n"
    "Draw Arc          %p\n"
    "\n",

    handlers[ppgDrawCmd::dcNull],
          
    handlers[ppgDrawCmd::dcPixBlit],
    handlers[ppgDrawCmd::dcPixBlitRaw],

    handlers[ppgDrawCmd::dcBlit],
    handlers[ppgDrawCmd::dcBlitRaw],
    
    handlers[ppgDrawCmd::dcSpanBlit],
    handlers[ppgDrawCmd::dcSpanBlitRaw]
          ,
    handlers[ppgDrawCmd::dcTransBlit],    
    handlers[ppgDrawCmd::dcTransBlitRaw],

    
    handlers[ppgDrawCmd::dcClear],
    
    handlers[ppgDrawCmd::dcDrawPoint],
    handlers[ppgDrawCmd::dcDrawLine],
    handlers[ppgDrawCmd::dcDrawHLine],
    handlers[ppgDrawCmd::dcDrawVLine],
    handlers[ppgDrawCmd::dcDrawPoly],
    handlers[ppgDrawCmd::dcDrawRect],
    handlers[ppgDrawCmd::dcDrawArc]
  );

  fprintf(out, "Surface Device is: %s\n", 
    dbLeakyGetObjectName(GetDevice())
  );
}


bool ppgGC::dbIntegrityOk()const
{
    char buf[128];
	if(&GetDevice())return true;
	
	ppWarning(
	     "GC %s, has no surface class defined!\n",
	     dbGetObjectName(&buf[0], sizeof(buf))
	);
	return false;
}
#endif //!PP_DEBUG
