/************************************************************************
    ppgDrawCmd.cc:  Defines ppgDrawCmd.


    ppgDrawCmd:
        Wraps up a drawing primitive invocation for buffering
        etc. Also defines types and constants for use with draw
        commands.


    Part of the PenguinPlay 2d library.  Please see LICENSE.TXT supplied
    with this library for copyright details.

    Copyright (C)    Adrian Ratnapala 1998.


    $Author: raka $
    $Date: 1998/11/25 10:06:27 $
    $Revision: 1.4 $
************************************************************************/

#include "ppgCommon.h"
#include <PenguinPlay/ppgColour.h>
#include <PenguinPlay/ppgGC.h>
#include <PenguinPlay/ppgDrawCmd.h>
#include <PenguinPlay/ppgSurface.h>
#include <PenguinPlay/ppgPixel.h>


#include "ppgCppDefs.inc"

/*********************************
  Stupid little No-Op style draw commands.  To fill the gaps
  where real ones should/might go.
****/

//This is a draw command intentionally left as a no op.
//If this gets called, all is cool
void ppgDrawCmd::EmptyHandler(ppgGC&, ppgSurface&, ...)
{}
  

#ifdef PP_DEBUG
//This is a draw command that has been left blank.  (The ultimate default).
//If this gets called, all is NOT cool (but it only gets picked up
//when debugging).
void ppgDrawCmd::NullHandler(ppgGC& gc, ppgSurface&, ...) 
{  
  ppFatalError("Performing NULL draw command %s\n", gc.dbGetLastCmd()); 
}
#endif //_PP_DEBUG





/*
 * Real draw commands for doing blits.
 *
 * The scanline blitting ones at least should be replaced, since
 * they just use repeated calls to PutPixel.  Nasty.  These ones can
 * be optimized further, but why bother.
 */


/*********************
  Single span blits.
***********/

//Blits a line of pixels, into any surface.
//THIS ROUTINE IS REALLY REALLY REALLY SLOW!
void ppgDrawCmd::SpanBlit(
                        COMMON_ARGS,
                        const ppgPixel* in, 
                        OUT_COORDS,
                        int len 
                      ) 
{
  for(int i=0; i < len; i++)
    gc.PixBlit( in[i], out_x+i, out_y);
}


//Raw blits a line of pixels, into any surface.
void ppgDrawCmd::SpanBlitRaw(COMMON_ARGS, 
                             const void* in, OUT_COORDS, int len) 
{
  //FIX: This will probably only work on little endian machines.
  ppgColourMode& c_mode = gc.GetDevice().GetColourMode();
  int Bpp = c_mode.GetDepth()/8; 
  
  for(int i=0; i < len; i++){
    gc.PixBlitRaw(*(ppgPixel*)((char*)in+i*Bpp), out_x+i, out_y);
  } 
}




/*
 * Proper rectangular blits.
 */

void ppgDrawCmd::Blit(
                     COMMON_ARGS, 
                     const ppgPixel* data, int scanline_skip,  
                     OUT_COORDS, 
                     ppgCoord w, ppgCoord h)
{
  const ppgPixel* cursor = (const ppgPixel*)data;

  for(int i=0; i < h; i++) {
    //FIX it is probably faster to call GC.GetHandler directly.
    gc.SpanBlit(cursor, out_x, out_y+i, w);
	cursor = (ppgPixel*) ((char*)cursor + scanline_skip);
  }
}


void ppgDrawCmd::BlitRaw(
                     COMMON_ARGS,
                     const void* data, int scanline_skip,  
		     OUT_COORDS,
		     ppgCoord w, ppgCoord h)
{
  const void* cursor = data;

  for(int i=0; i < h; i++) {
    //FIX it is probably faster to call GC.GetHandler directly.
    gc.SpanBlitRaw(cursor, w, out_x, out_y+i);
	cursor = (ppgRGBAPixel*) ((char*)cursor + scanline_skip);
  }
}



/*
 * Acutall "vector oriented" or "thing drawing" primitives.
 */

//One day this will support things like drawing variable sized/shaped dot etc.
//but for now it just makes a single call to ppgPutPixel.
void ppgDrawCmd::DrawPoint(COMMON_ARGS, ppgCoord x, ppgCoord y)
{
  gc.PixBlitRaw(gc.GetForeground(), x, y);
}


void ppgDrawCmd::DrawHLine(COMMON_ARGS, ppgCoord x, ppgCoord y, int len)
{
  for(int i=0; i < len; i++)
    gc.PixBlitRaw(gc.GetForeground(), x+i, y);
}

void ppgDrawCmd::DrawVLine(COMMON_ARGS, ppgCoord x, ppgCoord y, int len)
{
  for(int i=0; i < len; i++)
    gc.PixBlitRaw(gc.GetForeground(), x, y+i);
}

//I think we will leave the rest for later.  It's kind of pointless anyway,
//it being so slow and all. 


#include "ppgCppUndefs.inc"
