/************************************************************************
    ppgColour.cc:  Defines ppgColourMode.


    ppgColourMode:
        Defines mapping of pixel data to actual colours.


    Part of the PenguinPlay 2d library.  Please see LICENSE.TXT supplied
    with this library for copyright details.

    Copyright (C)    Adrian Ratnapala 1998.


    $Author: raka $
    $Date: 1998/11/25 10:06:24 $
    $Revision: 1.6 $
************************************************************************/


#include "ppgCommon.h"
#include <PenguinPlay/ppgColour.h>



/*****************************************************************************
 * ppgColourMode
 */

ppgColourMode::ppgColourMode(ColourType _type, int _depth) 
   : type(_type), depth(_depth) {};



/*****************************************************************************
 * ppgRGBAColourMode
 */

ppgRGBAMode::ppgRGBAMode(
          int _depth,
          ppu32 rmask,
          ppu32 gmask,
          ppu32 bmask,
          ppu32 amask
      ) :
      ppgColourMode(ctRGBA, _depth),
      red_mask(rmask),
      green_mask(gmask),
      blue_mask(bmask),
      alpha_mask(amask)
{
  _PP_DB_STD_OBJNAME("RGBA_COLOUR_MODE");

  unsigned int tmp;
  unsigned int red_off=0, green_off=0, blue_off=0, alpha_off=0;

  //FIX: Think about the consquences of endianess on all this.
  //I don't think we care.  I hope we don't care.

  for(tmp=GetRedMask()   ; tmp ; tmp>>=1) red_off++;
  for(tmp=GetGreenMask() ; tmp ; tmp>>=1) green_off++;
  for(tmp=GetBlueMask()  ; tmp ; tmp>>=1) blue_off++;
  for(tmp=GetAlphaMask() ; tmp ; tmp>>=1) alpha_off++;

  red_shift   = 32 - red_off;
  green_shift = 24 - green_off;
  blue_shift  = 16 - blue_off;
  alpha_shift =  8 - alpha_off;
}




ppgAbsPixel ppgRGBAMode::Canonicalize(ppgPixel pix)
   //OK, I _think_ we don't need to AND against the masks
   //OK, I _think_ we don't need to AND against the masks
   //if we do things in the right order.
   //if we do things in the right order.
{
   return 
         ( ( pix & GetRedMask()   ) << GetRedShift()   ) |
         ( ( pix & GetGreenMask() ) << GetGreenShift() ) |
         ( ( pix & GetBlueMask()  ) << GetBlueShift()  ) |
         ( ( pix & GetAlphaMask() ) << GetAlphaShift() );
}

ppgPixel ppgRGBAMode::Decanonicalize(ppgPixel pix)
{
   return 
         ( ( pix >> GetRedShift()   ) & GetRedMask()   ) |
         ( ( pix >> GetGreenShift() ) & GetGreenMask() ) |
         ( ( pix >> GetBlueShift()  ) & GetBlueMask()  ) |
         ( ( pix >> GetAlphaShift() ) & GetAlphaMask() );
}
void ppgRGBAMode::Decanonicalize(
  void* out, 
  const ppgAbsPixel* in, 
  int len)
{
  //FIX: Fill this in.
  ppDebug("Pretending to pack block of RGBA pixels");
}



/**************************************************************************
*  Debuggin stuff.
*****/

#ifdef PP_DEBUG
void ppgColourMode::dbDump(FILE* out)const
{
  ppBase::dbDump(out);
  fprintf(out, "Depth = %d\n", depth);
  fprintf(out, "Mode type = ");
  switch(type){
    case ctRGBA:     fprintf(out, "RGBA\n"); break;
    case ctIndexed:  fprintf(out, "Indexed\n"); break;
    default:         fprintf(out, "Unknown\n"); break;
  }
}

//convert a bitmask to ascii '1' and '0'.  For debugging.
char* mask_to_str(char* buf, int mask)
{
  for(unsigned int i=0; i < sizeof(mask)*8; i++)
    buf[sizeof(mask)*8-1-i] = (mask&(1<<i)) ? '1' : '0';
  return buf;
}

void ppgRGBAMode::dbDump(FILE* out)const
{
  ppgColourMode::dbDump(out);

  //so this is a bit excessive, I felt like it.
  char str_rep[33]; str_rep[32]=0;

  fprintf(out, "Red Mask   : %s\n", mask_to_str(str_rep, red_mask) );
  fprintf(out, "Green Mask : %s\n", mask_to_str(str_rep, green_mask) );
  fprintf(out, "Blue Mask  : %s\n", mask_to_str(str_rep, blue_mask) );
  fprintf(out, "Alpha Mask : %s\n", mask_to_str(str_rep, alpha_mask) );

  fprintf(out, "Red Shift   : %d\n", red_shift );
  fprintf(out, "Green Shift : %d\n", green_shift );
  fprintf(out, "Blue Shift  : %d\n", blue_shift );
  fprintf(out, "Alpha Shift : %d\n", alpha_shift );
}


bool ppgRGBAMode::dbIntegrityOk()const
{
  char buf[128];
  if(
    (red_mask   & (green_mask | blue_mask  | alpha_mask)) ||
    (green_mask & (red_mask   | blue_mask  | alpha_mask)) ||
    (blue_mask  & (red_mask   | green_mask | alpha_mask)) ||
    (alpha_mask & (red_mask   | green_mask | blue_mask ))
  ){
    ppWarning("Overlapping bitmasks in RGBA colour mode, %s", 
              dbGetObjectName(buf, sizeof(buf))
             );
    return false;
  };
  return true;
}


#endif //PP_DEBUG
