/************************************************************************
    ppgGC.h:  Defines ppgGC.


    ppgGC:
        Holds context data for drawing ops as well as providing
        the standard interface to them.


    Part of the PenguinPlay 2d library.  Please see LICENSE.TXT supplied
    with this library for copyright details.

    Copyright (C)    Adrian Ratnapala 1998.


    $Author: raka $
    $Date: 1998/11/25 10:06:03 $
    $Revision: 1.5 $
************************************************************************/

/***********************************************************************
  This is a strangely  written source file that makes heavy use of the C
  preprocessor (and screws up my syntax highlighting).  I did this for
  my own sanity (I've grown a real hatred for splitting header
  definitions over multiple lines).  It might not help your sanity, but
  beleive me, it is an attempt to _improve_ readability, I hope I have
  not done more harm than good.

  FIX: But make a lot of the #if's etc. neater.
***********************************************************************/

#ifndef _ppgGC_h
#define _ppgGC_h
#define _in_ppgGC_h

//////////////
//FIX: Don't know if this is valid C++, but it causes egcs and g++ to 
//fall down, so I'd better make a bug report.
//
//	bug-g++@prep.ai.mit.edu
//	egcs-bugs@cygnus.com
//
#include "ppgCommon.h"
#include <PenguinPlay/ppgDrawCmd.h>
#include <PenguinPlay/ppgPixel.h>
#include <PenguinPlay/ppgLinearSurface.h>
#include <PenguinPlay/ppgDevice.h>

class ppgColour;
class ppgColourMode;
class ppgSpanSprite;
class ppgLinearSurface;

#include <string.h>



//: That which draws into surfaces.
//  ppgGCs keep state info used by drawing ops.  They also
//  they export drawing primitives which are front ends to acutall
//  command handlers.
class ppgGC : public ppBase {

  /*
   * Administrative varialbes.
   */
  ppgDevice* device;


  /*
   * State variables
   */
  #include "ppgGC_types.inc"
  
  ppgDrawCmd::Handler handlers[ppgDrawCmd::dcNumVals];

  ppgSurface* curr_surf;
  ppgPixel fore_pixel;
  ppgPixel back_pixel;

  int  line_width;
  LineStyle line_style;

  FillStyle fill_style;

  TransformMode transform_mode;
  AntialiasMode antialias_mode;


  #ifdef PP_DEBUG
  /*
   * Dodgy hack to let us know what any NullCommand which be called
   * was meant to be.
   */
  char* last_cmd;
  #endif //PP_DEBUG

  
  static const ppgAbsPixel default_foreground_abs = 0xffffffff,
                           default_background_abs = 0x00000000;

		   
  public:

  /*
   * Support for dodgy debugging hacks.
   */
  #ifdef PP_DEBUG
    char* dbGetLastCmd(){return last_cmd;}
  #endif //PP_DEBUG

  /*******************
    Administrivia
  *****/

  ppgGC()
  {
    ppDebug("Default GC constructor used!\n");
  }
  ~ppgGC(); 

  ppgGC(const ppgDevice& sclass);
  //: Initialize a GC which draw according to the given colour mode.

  #ifdef PP_DEBUG
  virtual char* dbGetClassName()const{return "GC";}
  virtual void dbDump(FILE* out)const;
  virtual bool dbIntegrityOk()const; 
  #endif


  /***********
    Some nice getsets.
  ***********/
  
  ppgDrawCmd::Handler GetHandler(ppgDrawCmd::CmdType cmd)const
       { return handlers[cmd]; } 
	   //: Look up a handler func for a given DrawCmd opcode.
	   
  ppgDrawCmd::Handler SetHandler(ppgDrawCmd::CmdType cmd, 
                                 ppgDrawCmd::Handler hand)
       { return handlers[cmd]=(ppgDrawCmd::Handler)hand; }
	   //: Set up a handler func for a given DrawCmd opcode.


  _PP_GETSET_REF(Surface, curr_surf, ppgSurface); 
    //: Change our default surface.

  // _PP_GETSET_REF(Device, device, ppgDevice);
  // ????????Should GC's be able to migrate between devices?
  // If there were need, some sort of cloning mechanism might be 
  // better.
  ppgDevice& GetDevice()const{return *device;}


  static ppgAbsPixel GetDefaultForegroundAbs(){return default_foreground_abs;}
    //: Another name for "opaque white"
  static ppgAbsPixel GetDefaultBackgroundAbs(){return default_background_abs;}
    //: Another name for "transparent black"
  
  /*
   * Here be nice wrappers for draw commands.
   *
   * well, first a bit of preprocessor work.
   */

  #ifdef PP_DEBUG
    #define BC(func)                           \
      ppgGC* t = const_cast<ppgGC*>(this);     \
      t->last_cmd=#func;                       \
      GetHandler(ppgDrawCmd::func)(*t, GetSurface() 
  #else
    #define BC(func)                           \
      GetHandler(ppgDrawCmd::func)(*const_cast<ppgGC*>(this), GetSurface()
  #endif //PP_DEBUG
  

  #define CB )

  
  #include "ppgCppDefs.inc"

  /*
   * Functions for blasting pixels.
   */

  //In a normal API these would be called PutPixel, 
  //here they are blits even though that is a misnomer.
  void PixBlit(ppgPixel in, OUT_COORDS) const
    {BC(dcPixBlit), in, O_CRDS CB;} //:RGBA PutPixel  (not really a Blit)
	
  void PixBlitRaw(ppgPixel in, OUT_COORDS) const
    {BC(dcPixBlitRaw), in, O_CRDS CB;} //:Raw data PutPixel  (not really a Blit)
	
 
  //FIX: I suppose we should  also have verical span blits (less important).
  //blits a single scanline
  void SpanBlit( const ppgPixel* in, OUT_COORDS, int len) const
    {BC(dcSpanBlit), in, O_CRDS, len CB;} 
	//: Blast a horzontal line of RGBA pixels
	
  void SpanBlitRaw( const void* in, OUT_COORDS, int len) const
    {BC(dcSpanBlitRaw), in, O_CRDS, len CB;}
	//: Blast a horzontal line of raw pixels
	

  //blits rectangles of pixels.
  void Blit( BUFF_INFO(ppgPixel), OUT_COORDS, SIZE_COORDS) const
    {BC(dcBlit), B_INFO, O_CRDS, S_CRDS CB;}
	//: Blast a rectangle line of RGBA pixels
	
  void BlitRaw( BUFF_INFO(void),  OUT_COORDS, SIZE_COORDS) const
    {BC(dcBlitRaw), B_INFO, O_CRDS, S_CRDS CB;}
	//: Blast a rectangle line of Raw pixels

    
  //unpacks "in" to get the BUFF_INFO to do one of the above blits.
  void Blit( const ppgLinearSurface& in, COORDS) const;
    //: Blast a rectangle line of RGBA pixels from a linear surface.
    // Note this is not inline.  In some cases you may want to inline
    // this kind of thing on your own.  In general, don't bother.
  void BlitRaw( const ppgLinearSurface& in, COORDS) const;
    //: Blast a rectangle line of raw pixels from a linear surface.
    //void BlitCI( const ppgLinearSurface& in, COORDS) const;
    //: Blast a rectangle line of indexed pixels from a linear surface.

  //nice freindly (copy this to  this type function)
  void Blit( const ppgLinearSurface& in, _PP_POINT_COORDS)const
    {Blit( in, x, y, 0, 0, in.GetWidth(), in.GetHeight()); }
    //: Blast an RGBA ppgLinearSurface as if it were a sprite
	
  void BlitRaw( const ppgLinearSurface& in, _PP_POINT_COORDS) const
    {BlitRaw( in, x, y, 0, 0, in.GetWidth(), in.GetHeight()); }
    //: Blast a ppgLinearSurface raw, as if it were a sprite

  //blit a spanning sprite, doing rle transparency encoding.
  void TransBlit( const ppgSpanSprite& in, COORDS) const;
  void TransBlitRaw( const ppgSpanSprite& in, COORDS) const;
 

  /*
   * Clearing the screen.
   */

  void Clear() const{BC(dcClear) CB;} //FIX: this'll cause a preproc error.


  /*
   * "Vector Oriented" primitives.
   */
  void DrawPoint(_PP_POINT_COORDS) const;
    //: Draws a point in a given positon.  
    //  Unlike the PixBlitXXXX functions, this one is affected by the
    //  drawing colour, style and other kinds of GC state.
  
  void DrawLine(_PP_BOX_COORDS) const{BC(dcDrawLine), x, y, w, h CB;}
    //: Draws a line of solid colour.
  
  void DrawHLine(ppgCoord x, ppgCoord y, ppgCoord len) const
    {BC(dcDrawHLine), x, y, len CB;}
    //: Draws a horizontal line of solid colour.
	
  void DrawVLine(ppgCoord x, ppgCoord y, ppgCoord len) const
    {BC(dcDrawVLine), x, y, len CB;}
    //: Draws a vertical line of solid colour.
	
  void DrawRect(_PP_BOX_COORDS) const{BC(dcDrawLine), x, y, w, h CB;}
    //: Draws a (possibly filled ) box.
  
  void DrawPoly( const ppgPoint* points, int num_points) const;
    //: Draws a given polygon box.
  
  void DrawArc(
            ppgCoord x, ppgCoord y,     //top corner
            ppgCoord w, ppgCoord h,     //horz and vert [semi?]axes
            float start, float end      //positon of arc.
                                        //any reason not to use radians?
  ) const;
  //: Draws an eliptical arc.


  /************
    These are state change commands, also draw commands of a kind.
    The state can be saved and restored with PushState and PopState.
  ************/

  #define GETSET_CMD_TYPECONV(NAME, VAR, TYPE, TYPE2) \
  void Set##NAME (TYPE n) {                           \
    VAR=n;                                            \
    BC(dcSet##NAME), (TYPE2)n CB ;                    \
  }                                                   \
  TYPE Get##NAME () const{                            \
    BC(dcGet##NAME)    CB;                            \
    return VAR;                                       \
  }                                                   \
  _PP_GETSET(Raw##NAME, VAR, TYPE)

  #define GETSET_CMD(NAME, VAR, TYPE) GETSET_CMD_TYPECONV(NAME, VAR, TYPE, TYPE)
	
  GETSET_CMD_TYPECONV(Foreground, fore_pixel, ppgPixel, ppu32);
  GETSET_CMD_TYPECONV(Background, back_pixel, ppgPixel, ppu32);

  void SetForegroundAbs(ppgAbsPixel pix)
  {
    SetForeground(GetDevice().GetColourMode().Decanonicalize(pix));
  };

  void SetBackgroundAbs(ppgAbsPixel pix)
  {
    SetBackground(GetDevice().GetColourMode().Decanonicalize(pix));
  };
  
  // FIX
  // Think about Transparency handling.  (mode, alpha threshold,
  // value of the "magic transpant colour" etc).  What should go
  // in here, what should go in ppgColourMode.

  // FIX:
  // Think about how exactly we allow the user to set scale and xform matrix.
  // Should we have a special IntegerScale xform mode? I think so.

  GETSET_CMD(TransformMode, transform_mode, TransformMode); 
  GETSET_CMD(AntialiasMode, antialias_mode, AntialiasMode);
 
  #undef GETSET_CMD
  #undef GETSET_CMD_TYPECONV

  #include "ppgCppUndefs.inc"

  #undef BC
  #undef CB
};



#undef _in_ppgGC_h
#endif //_ppgGC_h
