/************************************************************************
    ppgDrawCmd.h:  Defines ppgDrawCmd.


    ppgDrawCmd:
        Wraps up a drawing primitive invocation for buffering
        etc. Also defines types and constants for use with draw
        commands.


    Part of the PenguinPlay 2d library.  Please see LICENSE.TXT supplied
    with this library for copyright details.

    Copyright (C)    Adrian Ratnapala 1998.


    $Author: raka $
    $Date: 1998/11/25 10:06:01 $
    $Revision: 1.3 $
************************************************************************/

#ifndef ppgDrawCmd_h
#define ppgDrawCmd_h

#include "ppgCommon.h"
#include <PenguinPlay/ppgPixel.h>


/**********************
* Some little forward declarations.
**********************/
class ppgGC;
class ppgSurface;
class ppgSpanSprite;


//: These wrap invocation of draw commands up as objects.
//
//  Representation of DrawCommands.   DrawCommands        
//  are executed by calling a handler with the args     
//                                                          
//    void Handler(
//       GC, 
//       X, Y, W, H, 
//       ExtraDatum0, ..., 
//       TypeId);
//    
//  (don't take this too seriously, there is real code below you can read).  
//                                                     
//  ppgDrawCmd is an object type which contains all the args to
//  these handlers,  so that these commands can be saved and
//  and manipulated.
//
//  Feel free to ignore and these args, just so long as if you
//  use one, you use all the  one before it.  i.e you can say
//
//  int MyHandler(ppgGC&);
//
//  But not, of course
//
//  int MyHandler(ppgCoord W);
//
//  When one of these invocations gets "frozen" as an object,
//  you can use the GETSETS provided below to peek at it.
//
//
class ppgDrawCmd {


  public:
  /***
    Some useful types.
  ***/
  enum CmdType{
    dcNull,


    /*************
      These copy from one linear block surface to any
      other surface.  
      FIX: Hmm, this could change to "any surface of currently
      selected GC input type".

      Transparency handling is controlled by the
      GC's internal state.  These different functions
      are for different input pixel formats.

      Raw means input pixel format==output, otherwise, input
      depends on GC state.
    *************/

    //Not really blits, PutPixels really.
    dcPixBlit, 
    dcPixBlitRaw, 
	
    //These copy simple linear spans of pixels
    dcSpanBlit, 
    dcSpanBlitRaw, 
	
    //Rectangular blits.
    dcBlit,      
    dcBlitRaw,

    
    //Like dcBlitXXXX, but input is a transparency encoded sprte
    dcTransBlit,
    dcTransBlitRaw,


    /************
      Clears the whole surface.  Uses the (possibly transparent) background
      colour.
    ************/
    dcClear,

    /************
      Nice shapes and things.
    *************/
    dcDrawPoint,   
    dcDrawLine,
    dcDrawHLine,
    dcDrawVLine,
    dcDrawPoly,
    dcDrawRect,
    dcDrawArc,  /*elliptical arc that is*/

    
    /**************
      State operators.  These only effect the
      GC, and the Surface field is meanigless.

      PushState pushes the current state down, and
      sets the new state to a copy of the old, so
      there is no work for the user.
    *************/
    
    dcPushState,                 /*We keep a nice neat stack of */
    dcPopState,                  /*states for interesting cases*/

    dcSetTransMode,              /*How is transparency to be handled*/
    dcGetTransMode,
    dcSetAlphaThreshold,         /*Min alpha for opacity in all-or-nothing */
    dcGetAlphaThreshold,
    dcSetTransColour,            /*Set value of transparent colour*/
                                 /*FIX Hmmm, something's wrong here*/
    
    dcSetTransformMode,          /*identity, scale, arbitrary*/
    dcGetTransformMode,
    dcSetScale,
    dcGetScale,
    dcSetTransform,
    dcGetTransform,
    dcSetAntialiasMode,
    dcGetAntialiasMode,

    
    dcSetForeground,
    dcGetForeground,
    dcSetBackground,
    dcGetBackground,
    dcSetStipple, 
    dcGetStipple,
      /*etc.  do "man XCreateGC" for more ideas than you want*/ 
     
    /*User defined draw command types (ofcourse)*/
    dcUser0,
    dcUser1,
    dcUser2,
    dcUser3,
    dcUser4,
    dcUser5,
    dcUser6,
    dcUser7,
    dcOnePastUser, 


    /****
      Useful thing to put at the end of all enums
    ****/
    dcNumVals = dcOnePastUser
  };/*end enum CmdType*/
  //: Actuall draw command opcode.

 

  
  typedef void(*Handler)(  //Actuall draw command handlers
    ppgGC&,
    ppgSurface&,
    ...
  );
  //: A function which fields a draw command.


  
  /***************************
    A few "standard handlers"

    FIX: Should these be in their own class?
    I think so.  Acutally we should use namespaces 
    as soon as egcs lets us.
  ****/

#include "ppgCppDefs.inc"
#define HAND(Name) static void Name(ppgGC& context, ppgSurface& target
  //:Pay Attention to this macro.  It means "Define a command handler".
#define DNAH )
  //:Pay Attention to this macro.  It means "Stop defining a command handler".

  HAND(EmptyHandler) ...DNAH; 
  //:no op draw command.

  /*
   * NullHandler means a command that has not been set, even
   * though it should have been.  When debbuging, it generates
   * a warning when it is called.
   */
#ifdef PP_DEBUG
  HAND(NullHandler) ...DNAH; 
#else /*!PP_DEBUG*/
  /*
   * this saves a _teeny_ bit of memory when not debugging.
   *
   * This makes debugging _easier_ because the it makes
   * it segfault instead of silently continuing.  Perhaps the
   * alternative should not be _silent_ though?
   */
  const Handler NullHandler=NULL; 
#endif /*PP_DEBUG*/

 

  //These are SLOW default handlers based on PutPixel.
  //they should be overriden.  

  HAND(SpanBlit), const ppgPixel* in, OUT_COORDS, int len DNAH;
  HAND(SpanBlitRaw), const void* in, OUT_COORDS, int len DNAH;

  HAND(Blit), const BUFF_INFO(ppgPixel), OUT_COORDS, SIZE_COORDS DNAH;
  HAND(BlitRaw), const BUFF_INFO(void), OUT_COORDS, SIZE_COORDS DNAH;

  //blit a spanning sprite, doing rle transparency encoding.
  //HAND(TransBlit), const ppgSpanSprite& in, COORDS DNAH;
  //HAND(TransBlitRaw), const ppgSpanSprite& in, COORDS DNAH;
  //HAND(TransBlitCI), const ppgSpanSprite& in, COORDS DNAH;

  // Clears output either to transparent or background (if there
  // is no transparency).
  HAND(Clear) DNAH;

  HAND(DrawPoint),  _PP_POINT_COORDS                             DNAH;
  HAND(DrawLine),   _PP_BOX_COORDS                               DNAH;
  HAND(DrawHLine),  _PP_POINT_COORDS,  int len                   DNAH;
  HAND(DrawVLine),  _PP_POINT_COORDS,  int len                   DNAH;
  HAND(DrawRect),   _PP_BOX_COORDS                               DNAH;
  HAND(DrawPoly),   const ppgPoint* points, int num_points       DNAH;
  /*forget arc for now.  Wait till we have designed the little guy.*/

  
  #undef HAND
  #undef DNAH
  #include "ppgCppUndefs.inc"

  /************
    Actuall draw command data.
  ***/
  
  private:
  /*
    Actual command info.  Mostly args for
    handlers.
  */
  CmdType           type;           /*What kind of command are we*/

  /*
   * These data a backwards to allow a (possible) optimization hack
   * described below.
   */
  void*             extention;
  long              extra_data[2];  /*Always leave room for more*/

  ppgCoord          w, h;           /*ditto*/
  ppgCoord          x, y;           /*Usefull sundry variables*/




  /**************************
    Oh, and we need an interface
  ****/

  public:

  //_PP_GETSET macro defines GetSomething and SetSomething inline methods.
  //SetXXXX always returns the _new_ value.
  _PP_GETSET(Type, type, CmdType);  //: Command opcoode.
  _PP_GETSET(X, x, ppgCoord);       //: (usually) x coord for op.
  _PP_GETSET(Y, y, ppgCoord);       //: (usually) y coord for op.
  _PP_GETSET(W, w, ppgCoord);       //: (usually) width of op
  _PP_GETSET(H, h, ppgCoord);       //: (usually) height of op.
  _PP_GETSET(ExtraDatum0, extra_data[1], long);  //: In case you need it.
  _PP_GETSET(ExtraDatum1, extra_data[0], long);  //: In case you need it.
  _PP_GETSET(Extention, extention, void*);  //: If you've run out of space.
  /*
   * ExtraDatum 0 and 1 are swapped, since it _might_ provoke a cool
   * optimization if gcc is on the ball.  It could copy (almost) 
   * the entire ppgDrawCmd straight onto the stack in one go.
   * 
   * Don't know if gcc is clever enough, or if it is a real gain.  
   * I shall have to see.
   * 
   * later: OK, I can't remember what I was thinking when I did that, but 
   * I wonder if that is endianness dependent.
   */
  
  /******************
    Actuall real work.
  ***/
  
  //void Execute(ppgGC& context, const Handler hand)const
  //{
  //  hand(
  //   context,
  //    GetX(),  GetY(), GetW(), GetH(),
  //    GetExtraDatum0(), GetExtraDatum1(), GetExtention(),
  //    GetType()
  //  );
  // }
  //: Actually excute a command.
  //  This one actually invokes the handler.  Notice you must
  //  supply the handler explicitly.
  //  FIX: this is all fscked, will write it when I actually implement
  //  this stuff.
};

#endif //ppgDrawCmd_h
