/************************************************************************
    ppgColour.h:  Defines ppgColourMode.                

    ppgColourMode:
        Defines mapping of pixel data to actual colours.

    Part of the PenguinPlay 2d library.  Please see LICENSE.TXT supplied
    with this library for copyright details.

    Copyright (C) Adrian Ratnapala 1998.


    $Author: raka $
    $Date: 1998/11/25 10:05:58 $
    $Revision: 1.4 $
************************************************************************/



#ifndef _ppgColour_h
#define _ppgColour_h

#include "ppgCommon.h"
#include <PenguinPlay/ppgPixel.h>

#ifdef PP_DEBUG
#include <iostream.h>
#endif


//: Maps ppgPixels to real colours.
// ppgColourMode encapsulates the mapping from ppgPixels into colours,
// so it deals with whether something is RGB or indexed, the depth,
// pixel format and colourmaps.
// (FIX:  We also have to worry about insane things like bit ordering
//        and I ain't just talk'n' about msb-lsb first, take a look
//        at the ggi direct access docs!)
//
class ppgColourMode : public ppBase {

  public:
    enum ColourType {
      ctRGBA, /*Includes RGB*/
      ctIndexed,
      ctNumVals
    };
    //: Possible kinds of colour modes.  FIX, rename.  FIX support more?

  private:

    ColourType     type;  //:What kind of mode are we?
    int            depth; //:How many (significant) bits are used per pixel
    int            bytes_pp;
	

  protected:

    ppgColourMode(ColourType _type, int _depth);  


  public:
    
    //HELP should these be setable?
    ColourType GetType(){return type;}
    
    int GetDepth()const{return depth;}
    int GetBytesPP()const{return bytes_pp;}


    /*
     *  Pixel conversions.
     */
    virtual ppgAbsPixel Canonicalize(ppgPixel val) = 0; 
	  //: Canonicalize a pixel into 8r8g8b8a.
    virtual ppgPixel Decanonicalize(ppgAbsPixel rgba_val) = 0; 
	  //: Decanonicalize a pixel from 8r8g8b8a.

     virtual void Decanonicalize(void *out, const ppgAbsPixel *in,
                                 int len)=0;
     //: pack an 8r8b8g8a pixel array into native pixels.
     //: FIX: canonicalize for this.  Unnecassary, but for the
     //: sake of compeleteness...

    #ifdef PP_DEBUG
    virtual char* dbGetClassName()const{return "COLOUR_MODE";}
    virtual void dbDump(FILE *out)const;
    #endif

};






/******************************************************************
*  ppgRGBAMode:
*  
*  This class encapsulates the mapping of RGBA pixels into colours.
*  So it deals with pixel format.
******************************************************************/ 
class ppgRGBAMode : public  ppgColourMode {

  private:

    ppu32      red_mask, green_mask, blue_mask, alpha_mask;
    ppu32      red_shift, green_shift, blue_shift, alpha_shift;


  public:

    ppgRGBAMode(
          int _depth, 
          ppu32 rmask, 
          ppu32 gmask, 
          ppu32 bmask, 
          ppu32 amask
      );

    /*********
      These bitmasks cordon off the red, green, blue and alpha bits.
      Basically the mask values contain the most red, most green, most
      blue and most opaque values.  They must have a contigous set of
      true bits.
    ***/
    _PP_GETSET(RedMask   ,  red_mask   ,  ppu32);
    _PP_GETSET(GreenMask ,  green_mask ,  ppu32);
    _PP_GETSET(BlueMask  ,  blue_mask  ,  ppu32);
    _PP_GETSET(AlphaMask ,  alpha_mask ,  ppu32);


   _PP_GETSET(RedShift, red_shift, int);
         //:How far to shift the red pixels left to canonicalize
   _PP_GETSET(GreenShift, green_shift, int);
         //:How far to shift the green pixels left to canonicalize
   _PP_GETSET(BlueShift, blue_shift, int);
         //:How far to shift the blue pixels left to canonicalize
   _PP_GETSET(AlphaShift, alpha_shift, int);
         //:How far to shift the alpha pixels left to canonicalize

    virtual ppgAbsPixel Canonicalize(ppgPixel val); 
    virtual ppgPixel Decanonicalize(ppgAbsPixel val); 
    virtual void Decanonicalize(void *out, const ppgAbsPixel *in, int len);
	
    #ifdef PP_DEBUG
    virtual void dbDump(FILE* out)const;
    virtual bool dbIntegrityOk()const;
    #endif
};



/***************************************************************
*  ppgIndexedMode:
*
*  Later  
***************************************************************/ 


#endif //_ppgColour_h
