/*
 * some X11 pixmaps rotines
 *
 *   (c) 1996 Gerd Knorr <kraxel@cs.tu-berlin.de>
 *
 * basic usage:
 *  1) call x11_color_init()
 *     this does all the visual checking/colormap handling stuff and returns
 *     TRUECOLOR or PSEUDOCOLOR
 *  2) create/load the image
 *  3) call x11_create_pixmaps()
 *     For TRUECOLOR:   It expects the data in one long (4 byte) per pixel.
 *                      To create the long, run the rgb-values throuth the
 *                      x11_lut_[red|green|blue] tables and or the results
 *     For PSEUDOCOLOR: The data is expected to be one byte per pixel,
 *                      containing the results from dither_line (see dither.c)
 *                      Not required to call init_dither, this is done by
 *                      x11_color_init
 *     returns a pixmap.
 *
 */

#include <stdio.h>
#include <stdlib.h>

#include <X11/Xlib.h>
#include <X11/Intrinsic.h>

#include "x11.h"
#include "dither.h"

/* ------------------------------------------------------------------------ */

static int display_type  = 0;
static int display_depth = 0;

/* PseudoColor: ditherresult => colormap-entry */
unsigned long x11_map[256];
int           x11_colors;

static int try_red[]   = { 6,4,5,4,3 };
static int try_green[] = { 6,8,5,4,3 };
static int try_blue[]  = { 6,4,5,4,3 };

/* TrueColor: r,g,b => X11-color */
unsigned long x11_lut_red[256];
unsigned long x11_lut_green[256];
unsigned long x11_lut_blue[256];

static int
x11_alloc_colorcube(Display *dpy, Colormap cmap, unsigned long *colors,
		    int red, int green, int blue)
{
    XColor       akt_color;
    int          i;
    
    for (i = 0; i < red*green*blue; i++) {
	akt_color.red   = ((i/(green*blue))%red)*  65535/(red-1);
	akt_color.green = ((i/blue)        %green)*65535/(green-1);
	akt_color.blue  = ( i              %blue)* 65535/(blue-1);

	if (!XAllocColor(dpy,cmap,&akt_color)) {
	    /* failed, free them */
	    XFreeColors(dpy,cmap,colors,i,0);
	    return 1;
	}
	colors[i] = akt_color.pixel;
    }
    return 0;
}

static void
x11_create_lut(unsigned long red_mask,
	       unsigned long green_mask,
	       unsigned long blue_mask)
{
    int rgb_red_bits = 0;
    int rgb_red_shift = 0;
    int rgb_green_bits = 0;
    int rgb_green_shift = 0;
    int rgb_blue_bits = 0;
    int rgb_blue_shift = 0;
    int i;
    unsigned long mask;

    for (i = 0; i < 24; i++) {
	mask = (1 << i);
	if (red_mask & mask) rgb_red_bits++;
	else if (!rgb_red_bits) rgb_red_shift++;
	if (green_mask & mask) rgb_green_bits++;
	else if (!rgb_green_bits) rgb_green_shift++;
	if (blue_mask & mask) rgb_blue_bits++;
	else if (!rgb_blue_bits) rgb_blue_shift++;
    }
#if 1
    printf("color: bits shift\n");
    printf("red  : %04i %05i\n",rgb_red_bits,rgb_red_shift);
    printf("green: %04i %05i\n",rgb_green_bits,rgb_green_shift);
    printf("blue : %04i %05i\n",rgb_blue_bits,rgb_blue_shift);
#endif
    
    for (i = 0; i < 256; i++) {
	x11_lut_red[i]   = (i >> (8-rgb_red_bits  )) << rgb_red_shift;
	x11_lut_green[i] = (i >> (8-rgb_green_bits)) << rgb_green_shift;
	x11_lut_blue[i]  = (i >> (8-rgb_blue_bits )) << rgb_blue_shift;
    }
}

int
x11_color_init(Widget shell)
{
    Display      *dpy;
    Screen       *scr;
    XVisualInfo  *info, template;
    int          found,i;

    dpy = XtDisplay(shell);
    scr = XtScreen(shell);

    /* Ask for visual type */
    template.screen   = XDefaultScreen(dpy);
    template.visualid =
	XVisualIDFromVisual(DefaultVisualOfScreen(scr));
    info = XGetVisualInfo(dpy,VisualIDMask | VisualScreenMask, &template,
			  &found);

    display_depth = (info->depth+7)/8;
    fprintf(stderr,"visual depth: %ibit, %ibyte\n",info->depth,display_depth);
    if (info->class == TrueColor) {
	/* TrueColor */
	x11_create_lut(info->red_mask, info->green_mask, info->blue_mask);
	display_type = TRUECOLOR;
    } else if (info->class == PseudoColor && info->depth == 8) {
	/* 8bit PseudoColor */
	for (i = 0; i < sizeof(try_red)/sizeof(int); i++) {
	    fprintf(stderr,"trying %i:%i:%i...",
		    try_red[i],try_green[i],try_blue[i]);
	    if (0 == x11_alloc_colorcube
		(dpy,DefaultColormapOfScreen(scr),x11_map,
		 try_red[i],try_green[i],try_blue[i])) {
		init_dither(try_red[i],try_green[i],try_blue[i]);
		x11_colors = try_red[i]*try_green[i]*try_blue[i];
		fprintf(stderr,"ok, %i colors allocated\n",x11_colors);
		break;
	    } else {
		fprintf(stderr,"failed\n");
	    }
	}
	if (i == sizeof(try_red)/sizeof(int)) {
	    fprintf(stderr,"sorry, can't allocate colors\n");
	    /* own colormap ? */
	    exit(1);
	}
	display_type = PSEUDOCOLOR;
    } else {
	fprintf(stderr,"sorry, can't handle visual\n");
	exit(1);
    }
    XFree(info);
    return display_type;
}

/* ------------------------------------------------------------------------ */

Pixmap
x11_create_pixmap(Widget shell, unsigned char *byte_data,
		  int width, int height)
{
    Pixmap         pixmap;
    XImage         *ximage;
    XGCValues      values;
    GC             gc;
    int            x,y;
    unsigned char  *ximage_data;
    unsigned long  *long_data = (unsigned long*)byte_data;

    Display *dpy = XtDisplay(shell);
    Screen  *scr = XtScreen(shell);

    pixmap = XCreatePixmap(dpy,
			   RootWindowOfScreen(scr),
			   width,height,
			   DefaultDepthOfScreen(scr));
    gc = XCreateGC(dpy,pixmap,0,&values);
    ximage_data = malloc(width*height*display_depth);
    ximage = XCreateImage(dpy,
			  DefaultVisualOfScreen(scr),
			  DefaultDepthOfScreen(scr),
			  ZPixmap,0,ximage_data,
			  width,height,
			  8,0);

    for (y = 0; y < height; y++)
	if (display_type == TRUECOLOR)
	    for (x = 0; x < width; x++)
		XPutPixel(ximage,x,y,*(long_data++));
	else
	    for (x = 0; x < width; x++)
		XPutPixel(ximage,x,y,x11_map[(int)(*(byte_data++))]);

    XPutImage(dpy,pixmap,gc,ximage,0,0,0,0,width,height);

    XDestroyImage(ximage);
    XFreeGC(dpy,gc);

    return pixmap;
}
