/*
 * a PhotoCD viewer, for svgalib and X11
 *
 *   (c) 1996 Gerd Knorr <kraxel@cs.tu-berlin.de>
 *
 */


#define _BSD_SOURCE

#include "config.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#ifdef HAVE_GETOPT_H
# include <getopt.h>
#else
extern int   optind;
extern char  *optarg;
#endif

#include <X11/Intrinsic.h>
#ifdef MOTIF
/* a lot Motif includes and defines */
#include <Xm/Xm.h>
#include <Xm/Form.h>
#include <Xm/PushB.h>
#include <Xm/Label.h>
#include <Xm/ScrolledW.h>
#include <X11/Xmu/Editres.h>
extern void _XEditResCheckMessages();
#define WIDGET_FORM     xmFormWidgetClass
#define WIDGET_BUTTON   xmPushButtonWidgetClass
#define WIDGET_LABEL    xmLabelWidgetClass
#define WIDGET_VIEWPORT xmScrolledWindowWidgetClass
#define CALLBACK_BUTTON XmNactivateCallback
#define RESOURCE_PIXMAP XmNlabelPixmap
#define RESOURCE_WIDTH  XmNwidth
#define RESOURCE_HEIGHT XmNheight
#else
/* the same for Athena */
#include <X11/StringDefs.h>
#include <X11/Xaw/XawInit.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/Viewport.h>
#define WIDGET_FORM     formWidgetClass
#define WIDGET_BUTTON   commandWidgetClass
#define WIDGET_LABEL    labelWidgetClass
#define WIDGET_VIEWPORT viewportWidgetClass
#define CALLBACK_BUTTON XtNcallback
#define RESOURCE_PIXMAP XtNbitmap
#define RESOURCE_WIDTH  XtNwidth
#define RESOURCE_HEIGHT XtNheight
#endif
#include "x11.h"

#ifdef SVGA
#include <vga.h>
#include <vgagl.h>
#include "svga.h"
#endif

#include "pcd.h"
#include "dither.h"

#define TRUE  1
#define FALSE 0

/* ---------------------------------------------------------------------- */

/* variables for read_image */
unsigned long    *lut_red=NULL,*lut_green=NULL,*lut_blue=NULL;
int              type,dither=FALSE,res=3,gray=FALSE;

/* file list */
char             **files;
int              fileanz,filenr;

/* svga font stuff */
int              font_height;
unsigned char    *font;
char             *fontfile = "/usr/lib/kbd/consolefonts/lat1-16.psf";

/* X11 stuff */
int              image_size;
XtAppContext     app_context;
Widget           app_shell,form,label,scroll,draw;

static struct RESDATA {
    char     *res;
} resdata;

static XtResource res_desc[] = {
    {
	"res",
	XtCString,
	XtRString,
	sizeof(char*),
	XtOffset(struct RESDATA*,res),
	XtRString,
	"3"
    }
};

static XrmOptionDescRec opt_desc[] = {
    { "-r", "res", XrmoptionSepArg, NULL },
};

/* ---------------------------------------------------------------------- */

void
usage(char *name)
{
    char *h;

    h = strrchr(name,'/');
    fprintf(stderr,
	    "This program shows PhotoCD images. It can use both X11 and the svgalib for\n"
	    "displaying the images.\n"
	    "  usage: %s [ options ] file ...\n"
	    "  options: (X11: only '-r' works)\n"
	    "    -h      print this text\n"
	    "    -m n    use graphics mode nr n\n"
	    "    -c n    Use a graphics mode with n colors. Allowed here:\n"
	    "            gray, 256, 32k, 64k and 16m\n"
	    "    -f file read font from file for printing text\n"
	    "            default: %s\n"
	    "    -q      be quiet: don't print anything at all\n"
	    "    -t n    timeout: load next image after n sec without any keypress\n"
	    "    -r n    select resolution [1..5]\n"
	    "\n"
	    "Large images can be scrolled using the cursor keys. Use ESC or 'q' to quit.\n"
	    "Space and PgDn show the next, PgUp shows the previous image. Jumping to a\n"
	    "image works with <number>g\n"
	    "\n"
	    "This program tries to find the best graphics mode for displaying the image,\n"
	    "depending on its size. All graphic modes are likely to be switched on.\n"
	    "SO BE SURE YOUR MONITOR CAN DISPLAY ALL MODES YOU HAVE IN YOUR SVGALIB CONFIG\n"
	    "FILE, YOUR MONITOR MAY GET DAMAGED ELSE.\n"
	    "\n"
	    ,h ? h+1 : name, fontfile);
}
    
/* ---------------------------------------------------------------------- */

void
read_image(char *filename, char *image,
	   int res, int *width, int *height)
{
    int               left,top;
    struct PCD_IMAGE  img;
    
    left=top=*width=*height=0;
    if (0 != pcd_open(&img,filename) ||
	-1 == pcd_select(&img,res,0,(type==PCD_TYPE_GRAY),0,
			 pcd_get_rot(&img,0),
			 &left,&top,width,height) ||
	-1 == pcd_decode(&img)) {
	fprintf(stderr,"libpcd: %s\n",pcd_errmsg);
	exit(1);
    }
    if (dither) {
	unsigned char *rgb_line  = malloc((*width)*3);
	int y;
	
	for (y = 0; y < *height; y++) {
	    if (-1 == pcd_get_image_line(&img,y,rgb_line,type,0)) {
		fprintf(stderr,"libpcd: %s\n",pcd_errmsg);
		exit(1);
	    }
	    dither_line (rgb_line, image+y*(*width), y, *width);
	}
	free(rgb_line);
    } else {
	if (type == PCD_TYPE_LUT_SHORT ||
	    type == PCD_TYPE_LUT_LONG)
		pcd_set_lookup(&img,lut_red,lut_green,lut_blue);
	if (-1 == pcd_get_image(&img,image,type,0)) {
	    fprintf(stderr,"libpcd: %s\n",pcd_errmsg);
	    exit(1);
	}
    }
    pcd_close(&img);
}

/* ---------------------------------------------------------------------- */

void
svga_readfont()
{
    FILE *fp = fopen(fontfile,"r");
    if (NULL == fp)
	return;
    if (fgetc(fp) != 0x36 ||
	fgetc(fp) != 0x04 ||
	fgetc(fp) != 0x00)
	return;
    
    font_height = fgetc(fp);
    font = malloc(font_height*256);
    fread(font,256,font_height,fp);
    fclose(fp);
}

/* ---------------------------------------------------------------------- */

void SetText(char *text)
{
#ifdef MOTIF
    XmString str;

    str = XmStringCreate(text,XmFONTLIST_DEFAULT_TAG);
    XtVaSetValues(label,XmNlabelString,str,NULL);
    XmStringFree(str);
#else
    XtVaSetValues(label,XtNlabel,text,NULL);
#endif
}

Boolean LoadImageWP(XtPointer client_data)
{
    char   *filename = files[filenr];
    char   message[512],*basename;
    int    width,height;
    Pixmap pix;
    unsigned char *image;

    image = malloc(image_size);
    read_image(filename,image,res,&width,&height);
    pix = x11_create_pixmap(app_shell,image,width,height);
    XtVaSetValues(draw,RESOURCE_PIXMAP,pix,NULL);
    basename = strrchr(filename,'/');
    basename = basename ? basename+1 : filename;
    sprintf(message,"%s (%i/%i)",basename,filenr+1,fileanz);
    SetText(message);
    free(image);
    return TRUE;
}

void ExitCB(Widget widget, XtPointer client_data, XtPointer calldata)
{
    exit(0);
}

void ChangeImageCB(Widget widget, XtPointer client_data, XtPointer calldata)
{
    char  message[512],*basename;
    int   diff = (int)client_data;

    if (filenr+diff >= 0 && filenr+diff < fileanz) {
	filenr += diff;
	basename = strrchr(files[filenr],'/');
	basename = basename ? basename+1 : files[filenr];
	sprintf(message,"loading %s...",basename);
	SetText(message);
	XtAppAddWorkProc(app_context,LoadImageWP,NULL);
    }
}

/* ---------------------------------------------------------------------- */

int
main(int argc, char *argv[])
{
#ifdef SVGA
    int              mode=0, colors=0, timeout=0, verbose = 1;
    int              width,height;
    unsigned char    *image;
    vga_modeinfo     *vgamode;
    int              c,rc;
    int              need_read;
    char             message[256],*basename;
#endif

    Widget           quit,next,prev;

    /* check for X11 */
#ifdef SVGA
    if (NULL != getenv("DISPLAY")) {
#endif
	if (geteuid() == 0 && getuid() != 0) {
	    /* running setuid root (svgalib), give up permissions */
#ifndef SVGA
	    fprintf(stderr,"pcdview is compiled without svgalib support, "
		    "so there is no need to run it\nsuid root\n");
#endif
	    fprintf(stderr,"running suid root, give up permissions\n");
	    setgid(getgid());
	    setuid(getuid());
	}
	/* open connection to the X-Server, create widgets, etc. */
	app_shell = XtAppInitialize(&app_context,
				    "Pcdview",
				    opt_desc, 1,
				    &argc, argv,
				    NULL,
				    NULL, 0);
	if (argc == 1) {
	    usage(argv[0]);
	    exit(1);
	}
#ifdef MOTIF
	XtAddEventHandler (app_shell, (EventMask)0, True,
			   _XEditResCheckMessages, NULL);
#endif

	XtGetApplicationResources(app_shell,&resdata,
				  res_desc,XtNumber(res_desc),
				  NULL,0);
	res=atoi(resdata.res);

	form = XtVaCreateManagedWidget("form",WIDGET_FORM,app_shell,
				       NULL);
	quit = XtVaCreateManagedWidget("quit",WIDGET_BUTTON,form,
				       NULL);
	prev = XtVaCreateManagedWidget("prev",WIDGET_BUTTON,form,
				       NULL);
	next = XtVaCreateManagedWidget("next",WIDGET_BUTTON,form,
				       NULL);
	label = XtVaCreateManagedWidget("label",WIDGET_LABEL,form,
					RESOURCE_WIDTH,PCD_WIDTH(res,0)+10,
					NULL);
	scroll = XtVaCreateManagedWidget("s",WIDGET_VIEWPORT,form,
					 RESOURCE_WIDTH,PCD_WIDTH(res,0)+10,
					 RESOURCE_HEIGHT,PCD_HEIGHT(res,0)+10,
					 NULL);
	draw = XtVaCreateManagedWidget("draw",WIDGET_LABEL,scroll,
				       NULL);

	XtAddCallback(quit,CALLBACK_BUTTON,ExitCB,NULL);
	XtAddCallback(prev,CALLBACK_BUTTON,ChangeImageCB,(XtPointer)-1);
	XtAddCallback(next,CALLBACK_BUTTON,ChangeImageCB,(XtPointer)1);
	XtRealizeWidget(app_shell);

	switch (x11_color_init(app_shell)) {
	case PSEUDOCOLOR:
	    type       = PCD_TYPE_RGB;
	    dither     = TRUE;
	    image_size = PCD_WIDTH(res,0)*PCD_HEIGHT(res,0);
	    break;
	case TRUECOLOR:
	    type       = PCD_TYPE_LUT_LONG;
	    lut_red    = x11_lut_red;
	    lut_green  = x11_lut_green;
	    lut_blue   = x11_lut_blue;
	    image_size = 4*PCD_WIDTH(res,0)*PCD_HEIGHT(res,0);
	    break;
	default:
	    fprintf(stderr,"Huh? unexpected return value (x11_color_init)\n");
	    exit(1);
	    break;
	}

	files   = argv+1;
	fileanz = argc-1;
	filenr  = 0;
	SetText("one moment please...");
        XtAppAddWorkProc(app_context,LoadImageWP,NULL);

	XtAppMainLoop(app_context);
	exit(0); /* keep compiler happy */
#ifdef SVGA
    }
    
    /* no X11 ==> use SVGA stuff */
    for (;;) {
	c = getopt(argc, argv, "hqm:c:r:t:f:");
	if (c == -1)
	    break;
	switch (c) {
	case 'h':
	    usage(argv[0]);
	    exit(1);
	case 'q':
	    verbose = 0;
	    break;
	case 'm':
	    mode = atoi(optarg);
	    break;
	case 'c':
	    switch (atoi(optarg)) {
	    case 256:
		colors = 256;
		break;
	    case 32:
	    case 32*1024:
		colors = 32*1024;
		break;
	    case 64:
	    case 64*1024:
		colors = 64*1024;
		break;
	    case 16:
	    case 16*1024*1024:
		colors = 16*1024*1024;
		break;
	    default:
		if (strcasecmp(optarg,"gray") == 0 ||
		    strcasecmp(optarg,"grey") == 0) {
		    colors = 256;
		    gray = TRUE;
		}  else {
		    colors = 0;
		}
	    }
	    if (colors == 0)
		fprintf(stderr,"I don't understand `-c %s', ignoring.\n",
			optarg);
	    break;
	case 'r':
	    res = atoi(optarg);
	    break;
	case 'f':
	    fontfile = optarg;
	    break;
	case 't':
	    timeout = atoi(optarg);
	    break;
	default:
	    exit(1);
	}
    }

    if (optind == argc) {
	usage(argv[0]);
	exit(1);
    }
    files   = argv+optind;
    fileanz = argc-optind;
    filenr  = 0;
    need_read = 1;

    vga_init();
    vgamode = svga_guess_mode(PCD_WIDTH(res,0),PCD_HEIGHT(res,0),colors,mode);
    if (NULL == vgamode) {
	if (mode)
	    fprintf(stderr,"Can't use mode %i, sorry.\n",mode);
	else
	    fprintf(stderr,"I found no graphics mode.\n");
	exit(1);
    }
    if (verbose) {
	 /* set default in case readfont failes */
	font=gl_font8x8; font_height=8;
	svga_readfont();
	gl_setwritemode(WRITEMODE_OVERWRITE | FONT_COMPRESSED);
	gl_setfont(8,font_height,font);
	gl_setfontcolors(0,0xffffffff);
    }
    switch (vgamode->bytesperpixel) {
    case 1:
	if (gray) {
	    type=PCD_TYPE_GRAY;
	    svga_gray_palette();
	} else {
	    type=PCD_TYPE_RGB;
	    svga_dither_palette(8,8,4);
	    dither=TRUE;
	    init_dither(8,8,4);
	}
	break;
    case 2:
	type=PCD_TYPE_LUT_SHORT;
	if (vgamode->colors == 32*1024) {
	    lut_red   = LUT_15_red;
	    lut_green = LUT_15_green;
	    lut_blue  = LUT_15_blue;
	} else {
	    lut_red   = LUT_16_red;
	    lut_green = LUT_16_green;
	    lut_blue  = LUT_16_blue;
	}
	break;
    case 3:
	type=PCD_TYPE_RGB;
	break;
    default:
	fprintf(stderr,"Oops: %i bytes/pixel ???\n",
		vgamode->bytesperpixel);
	exit(1);
    }

    if (verbose) {
	sprintf(message,"video mode:   %ix%i, %i colors",
		vgamode->width, vgamode->height, vgamode->colors);
	gl_write(0,0,message);
	gl_write(0,2*font_height,"ESC,Q:        quit");
	gl_write(0,3*font_height,"PgDn,space:   next image");
	gl_write(0,4*font_height,"PgUp:         previous image");
	gl_write(0,5*font_height,"<number>G:    jump to image <number>");
	gl_write(0,6*font_height,"cursor keys:  scroll large images");
    }

    image = malloc(PCD_WIDTH(res,0)*PCD_HEIGHT(res,0)*
		   vgamode->bytesperpixel);

    /* svga main loop */
    for (;;) {
	if (need_read) {
	    need_read = 0;
	    if (verbose) {
		basename = strrchr(files[filenr],'/');
		basename = basename ? basename+1 : files[filenr];
		sprintf(message,"loading %s... ",basename);
		gl_write(0,vgamode->height-font_height,message);
	    }
	    read_image(files[filenr],image,res,&width,&height);
	    if (width < vgamode->width || height < vgamode->height)
		svga_cls(vgamode);
	}
	switch (rc=svga_show(vgamode,image,width,height,timeout)) {
	case KEY_ESC:
	case KEY_Q:
	case KEY_EOF:
	    vga_setmode(TEXT);
	    exit(0);
	    break;
	case KEY_SPACE:
	    if (filenr == fileanz-1) {
		vga_setmode(TEXT);
		exit(0);
	    }
	    /* fall throuth: space exits after the last image, PgDn not */
	case KEY_PGDN:
	    if (filenr < fileanz-1)
		filenr++, need_read = 1;
	    break;
	case KEY_PGUP:
	    if (filenr > 0)
		filenr--, need_read = 1;
	    break;
	case KEY_TIMEOUT:
	    need_read = 1;
	    filenr++;
	    if (filenr == fileanz)
		filenr = 0;
	    break;
	default:
	    if (rc > 0 && rc <= fileanz)
		filenr = rc-1, need_read = 1;
	    break;
	}
    }
#endif
}


