/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * PhotoCD loading file filter for the GIMP
 *   (c) 1996 Gerd Knorr <kraxel@cs.tu-berlin.de>
 *
 * Hint: If you use the high resolutions, you may want to have a look at
 * xpcd. xpcd can load small pieces from a photocd image, so you don't have
 * to load the whole image (and crop after) if you need a small piece only.
 * This reduces memory usage.
 * You can find it at sunsite (and mirrors):
 *   sunsite.unc.edu:/pub/Linux/X11/xapps/graphics/viewer/xpcd-$version.tar.gz
 *
 * Some of the code (main and some dialog stuff)  is just cut-and
 * pasted from the JPEG plug-in
 *    (by Peter Mattis)
 *
 */

/*
 * ChangeLog
 *
 * Version 2.0beta3:
 *
 *  - thumbnail tweaking (load speedup, layout)
 *
 *
 * Version 2.0beta2:
 *
 *  - more load options (orientation)
 *  - 4BASE and 16BASE resolutions appear only in the dialog box
 *    if they are present in the file
 *  - thumbnails work again, and they are numbered now
 *  - finetuning and bugfixes, both plug-in and libpcd
 *
 *
 * Version 2.0beta1:
 *
 *  - it is based on the libpcd now (and used as test application for it)
 *  - faster: Your CD-ROM is the speed limit now, as long as you don't try
 *    to load the 3072x2048 image with only 8 MB RAM.
 *  - 4BASE & 16BASE support added (OK, these are not that fast)
 *  - You can select between RGB and grayscaled image now
 *  - thumbnail loading temporarly removed
 *
 *
 * Version 1.1:
 *
 *  - added thumbnails support: you can load the "overview.pcd" file now.
 *    you get a huge image with them all :-)
 *
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "gimp.h"
#include "pcd.h"

/*--------------------------------------------------------*/

static char *prog_name;

/*--------------------------------------------------------*/

static char *font[] = {     /* stolen from X11's 5x7-font */

    " ##  ",
    "#  # ",
    "#  # ",
    "#  # ",
    "#  # ",
    " ##  ",
    "     ",

    "  #  ",
    " ##  ",
    "  #  ",
    "  #  ",
    "  #  ",
    " ### ",
    "     ",

    " ##  ",
    "#  # ",
    "   # ",
    "  #  ",
    " #   ",
    "#### ",
    "     ",

    "###  ",
    "   # ",
    " ##  ",
    "   # ",
    "#  # ",
    " ##  ",
    "     ",

    "  #  ",
    " ##  ",
    "# #  ",
    "#### ",
    "  #  ",
    "  #  ",
    "     ",

    "#### ",
    "#    ",
    "###  ",
    "   # ",
    "#  # ",
    " ##  ",
    "     ",

    " ##  ",
    "#    ",
    "###  ",
    "#  # ",
    "#  # ",
    " ##  ",
    "     ",

    "#### ",
    "   # ",
    "  #  ",
    "  #  ",
    " #   ",
    " #   ",
    "     ",

    " ##  ",
    "#  # ",
    " ##  ",
    "#  # ",
    "#  # ",
    " ##  ",
    "     ",

    " ##  ",
    "#  # ",
    "#  # ",
    " ### ",
    "   # ",
    " ##  ",
    "     "
};

/*--------------------------------------------------------*/

static int dialog_ID;
static int dialog_result = 3;
static int color = 1;
static int rot,maxres;

static void
item_callback (item_ID, client_data, call_data)
     int item_ID;
     void *client_data;
     void *call_data;
{
    if (*(int*)call_data == 1)
	dialog_result = (int)client_data;
}

static void
color_callback (item_ID, client_data, call_data)
     int item_ID;
     void *client_data;
     void *call_data;
{
    if (*(int*)call_data == 1)
	color = (int)client_data;
}

static void
rotate_callback (item_ID, client_data, call_data)
     int item_ID;
     void *client_data;
     void *call_data;
{
    if (*(int*)call_data == 1)
	rot = (int)client_data;
}

static void
ok_callback (item_ID, client_data, call_data)
     int item_ID;
     void *client_data;
     void *call_data;
{
  gimp_close_dialog (dialog_ID, 1);
}

static void
cancel_callback (item_ID, client_data, call_data)
     int item_ID;
     void *client_data;
     void *call_data;
{
  gimp_close_dialog (dialog_ID, 0);
}

int size_dialog()
{
    static int one = 1;
    int frame,biggrp,group_ID,i[5];
	
    dialog_ID = gimp_new_dialog ("PhotoCD Load Options");
    biggrp  = gimp_new_row_group (dialog_ID, 0, DEFAULT, "fred");

    frame = gimp_new_frame (dialog_ID, biggrp, "channels");
    group_ID  = gimp_new_column_group (dialog_ID, frame, RADIO, "bar");
    i[0] = gimp_new_radio_button (dialog_ID, group_ID, "gray");
    gimp_add_callback (dialog_ID, i[0], color_callback, (void*)0);
    i[1] = gimp_new_radio_button (dialog_ID, group_ID, "color");
    gimp_add_callback (dialog_ID, i[1], color_callback, (void*)1);
    gimp_change_item (dialog_ID, i[1], sizeof(int), &one);

    frame = gimp_new_frame (dialog_ID, biggrp, "rotate");
    group_ID  = gimp_new_column_group (dialog_ID, frame, RADIO, "o");
    i[0] = gimp_new_radio_button (dialog_ID, group_ID, "0");
    gimp_add_callback (dialog_ID, i[0], rotate_callback, (void*)0);
    i[1] = gimp_new_radio_button (dialog_ID, group_ID, "90 ccw");
    gimp_add_callback (dialog_ID, i[1], rotate_callback, (void*)1);
    i[2] = gimp_new_radio_button (dialog_ID, group_ID, "180");
    gimp_add_callback (dialog_ID, i[2], rotate_callback, (void*)2);
    i[3] = gimp_new_radio_button (dialog_ID, group_ID, "90 cw");
    gimp_add_callback (dialog_ID, i[3], rotate_callback, (void*)3);
    gimp_change_item (dialog_ID, i[rot], sizeof(int), &one);
    
    frame = gimp_new_frame (dialog_ID, biggrp, "resolution");
    group_ID  = gimp_new_row_group (dialog_ID, frame, RADIO, "foo");
    i[0] = gimp_new_radio_button (dialog_ID, group_ID, "BASE/16 (196x128)");
    gimp_add_callback (dialog_ID, i[0], item_callback, (void*)1);
    i[1] = gimp_new_radio_button (dialog_ID, group_ID, "BASE/4 (384x256)");
    gimp_add_callback (dialog_ID, i[1], item_callback, (void*)2);
    i[2] = gimp_new_radio_button (dialog_ID, group_ID, "BASE (768x512)");
    gimp_add_callback (dialog_ID, i[2], item_callback, (void*)3);
    if (maxres > 3) {
	i[3] = gimp_new_radio_button (dialog_ID, group_ID,
				      "4BASE (1536x1024)");
	gimp_add_callback (dialog_ID, i[3], item_callback, (void*)4);
	i[4] = gimp_new_radio_button (dialog_ID, group_ID,
				      "16BASE (3072x2048, RGB: 18 MB !)");
	gimp_add_callback (dialog_ID, i[4], item_callback, (void*)5);
    }
    gimp_change_item (dialog_ID, i[2], sizeof(int), &one);

    gimp_add_callback (dialog_ID, gimp_ok_item_id (dialog_ID), ok_callback, 0);
    gimp_add_callback (dialog_ID, gimp_cancel_item_id (dialog_ID), cancel_callback, 0);

    if (gimp_show_dialog (dialog_ID))
	return dialog_result;
    else
	return -1;
}

/*--------------------------------------------------------*/

void load_image(char *filename)
{
    int              left,top,width,height,size,rc;
    int              tx,ty,tcols,trows,x,y,i,c;
    Image            image;
    struct PCD_IMAGE img;
    unsigned char    *temp,nr[5];

    if (-1 == (rc = pcd_open(&img,filename))) {
        gimp_message (pcd_errmsg);
	gimp_quit ();
    }
    if (rc) {
	/* load thumbnails */
	gimp_init_progress ("Loading thumbnails");
	for (tcols=1; tcols*tcols < rc; tcols++)
	    /* nothing */;
	tcols += tcols>>2;
	if (tcols > 8) tcols = 8;
	trows = (rc+tcols-1)/tcols;

	image = gimp_new_image (filename,100*tcols,100*trows, RGB_IMAGE);
	temp  = gimp_image_data (image);

	left=top=width=height=0;
	for (ty = 0; ty < trows; ty++) {
	    for (tx = 0; tx < tcols && tx+ty*tcols < rc; tx++) {
		gimp_do_progress (tx+ty*tcols, rc);

		/* fill background with gray */
		for (y = 0; y < 99; y++) {
		    i = ((ty*100+y)*100*tcols + tx*100)*3;
		    if (y%98)
			memset(temp+i,192,297);
		    else
			memset(temp+i+3,192,297-6);
		}

		/* load image */
		rot = pcd_get_rot(&img,tx+ty*tcols);
		left=top=width=height=0;
		if (-1 == pcd_select(&img,1,tx+ty*tcols,0,0,rot,
				     &left,&top,&width,&height) ||
		    -1 == pcd_decode(&img)) {
		    gimp_message (pcd_errmsg);
		    gimp_quit ();
		}
		
		if (rot & 1) {
		    for (y = 0; y < 96; y++) {
			i = ((ty*100+y+2)*100*tcols + tx*100 + 18)*3;
			if (-1 == pcd_get_image_line(&img,y,temp+i,
						     PCD_TYPE_RGB,1)) {
			    gimp_message (pcd_errmsg);
			    gimp_quit ();
			}
		    }
		} else {
		    for (y = 0; y < 64; y++) {
			i = ((ty*100+y + 18)*100*tcols + tx*100 +2)*3;
			if (-1 == pcd_get_image_line(&img,y,temp+i,
						     PCD_TYPE_RGB,1)) {
			    gimp_message (pcd_errmsg);
			    gimp_quit ();
			}
		    }
		}
		
		/* add the number to each of them:
		 * white, lower left corner */
		sprintf(nr,"%i",tx+ty*tcols+1);
		for (c = 0; c < strlen(nr); c++)
		    for (y = 0; y < 7; y++)
			for (x = 0; x < 5; x++)
			    if (font[(nr[c]-'0')*7+y][x] == '#') {
				i = ((ty*100 + y+5)*100*tcols +
				     tx*100 + c*5 + x+3)*3;
				temp[i  ] = 0;
				temp[i+1] = 0;
				temp[i+2] = 0;
			    }
	    }
	}
	gimp_do_progress (1, 1);
	gimp_display_image (image);
	gimp_update_image (image);
	gimp_free_image (image);
	gimp_quit ();
    }

    /* normal image */
    rot    = pcd_get_rot(&img,0);
    maxres = pcd_get_maxres(&img);
    size = size_dialog();
    if (-1 == size) {
	gimp_quit ();
	return; /* keep compiler happy */
    }
    
    if (size > 3) {
	/* the large resolutions require some time to decode... */
        temp = malloc (strlen (filename) + 11);
        if (!temp)
            gimp_quit ();
	
        sprintf (temp, "Loading %s:", filename);
        gimp_init_progress (temp);
        free (temp);
    }
    
    left=top=width=height=0;
    if (-1 == pcd_select(&img,size,0,!color,0,rot,&left,&top,&width,&height) ||
	-1 == pcd_decode(&img)) {
        gimp_message (pcd_errmsg);
	gimp_quit ();
    }
#if 0
    fprintf(stderr,"DEBUG: rot=%i, size=%i, geo=%ix%i+%i+%i\n",
	    rot,size,width,height,left,top);
#endif
    if (size > 3)
	gimp_do_progress(1,2);
    image = gimp_new_image (filename,width,height,
			    color ? RGB_IMAGE : GRAY_IMAGE);
    if (-1 == pcd_get_image(&img,gimp_image_data (image),
			    color?PCD_TYPE_RGB:PCD_TYPE_GRAY,0)) {
        gimp_message (pcd_errmsg);
	gimp_quit ();
    }
    pcd_close(&img);
    if (size > 3)
	gimp_do_progress(2,2);
    gimp_display_image (image);
    gimp_update_image (image);
    gimp_free_image (image);
    gimp_quit ();

    return;
}


/*--------------------------------------------------------*/

int
main (int argc, char *argv[])
{
    /*
     * Save the program name so we can use it later in reporting errors
     */
    prog_name = argv[0];

    /*
     * Call 'gimp_init' to initialize this filter.
     * 'gimp_init' makes sure that the filter was properly called and
     *  it opens pipes for reading and writing.
     */
    if (gimp_init (argc, argv))
	{
	    /*
	     * This is a file filter so all it needs to know about is loading
	     *  and saving images. So we'll install handlers for those two
	     *  messages.
	     */
	    gimp_install_load_save_handlers (load_image, NULL);

	    /*
	     * Run until something happens. That something could be getting
	     *  a 'QUIT' message or getting a load or save message.
	     */
	    gimp_main_loop ();
	}
    
    /* keep compiler happy */
    return 0;
}
