/****
File: ico2xpm.c
        - A simple program to convert .ico or .icl to .xpm files.
Author: Laura Tweedy <tweedy@umr.edu>
Date: Mon Jan 11 20:44:36 CST 1999
****/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>

#include "ico.h"

/** CONSTANTS **/
const char *program_name = "ico2xpm";
const char *version = "1.1";
const char *author = "Laura Tweedy <tweedy@umr.edu>";

/*
   msg()
	A specialized fprintf that prints the name of the program before
	all error messages.
*/
void msg(char *format, ...) {
	va_list ap;

	va_start(ap, format);
	fprintf(stderr, "%s: ", program_name);
	vfprintf(stderr, format, ap);
	va_end(ap);
	abort;
}

void usage(char *pname) {
	fprintf(stderr, "Usage: %s [-l] [-n image_number] [infile] [outfile]\n"
			"       %s -i [infile]\n"
			"       %s -v\n", pname, pname, pname);
	fprintf(stderr, "-l   input file is a library\n"
			"-i   information about the icon or library\n"
			"-n   specify the icon number you want to convert\n");
	exit(0);
}

/**
   main()
	Handle command line args and filenames.
**/
int main(int argc, char **argv) {
	FILE*	f = NULL;
	int	rv = 0;
	char*	infile = NULL;
	char*	outfile = NULL;
	ICOImage* images = NULL;
	int	num_images = 0;
	char**	image_names = NULL;
	int	image_to_convert = 1;
	int	print_info = 0;
	int	icon_library = 0;
	int	i;
	/* for getopt() */
	int c;
	extern char *optarg;
	extern int optind;

	while ((c = getopt(argc, argv, "hviln:")) != EOF) {
		switch(c) {
		case 'h': /* fall thru */
		case 'v': fprintf(stderr,
				"Icon to Pixmap Converter  "
				"by %s  Version %s\n", author, version);
			  usage(argv[0]);
			  break;
		case 'i': print_info = 1;
			  break;
		case 'l': icon_library = 1;
			  break;
		case 'n': image_to_convert = atoi(optarg);
			  break;
		default:  break;
		}
	}
	/* first argument is input file */
	if (optind < argc) {
		infile = argv[optind];
		optind++;
	}
	/* next argument is output file */
	if (optind < argc) {
		outfile = argv[optind];
	}

	if (infile != NULL) {
        	f = fopen(infile, "r");
		if (f == NULL) {
			msg("Failed to open file %s\n", infile);
			return -1;
		}
	}
	else {
		f = stdin;
	}

	if (icon_library == 0) {
		num_images = read_ICO_file(f, &images);
	}
	else {
		num_images = read_ICL_file(f, &images, &image_names);
	}

        if (num_images < 0) {
		msg("%s\n", get_err_msg());
                msg("Error reading icon file %s\n", infile);
                return -1;
        }
	if (print_info) {
/* for all images? */
		fprintf(stderr, "Number of Images: %d\n", num_images);
		if (icon_library == 1) {
			fprintf(stderr, "Image Names:\n");
			for (i=0; i<num_images; i++) {
				if (image_names != NULL) {
					if (image_names[i] != NULL) {
	   					fprintf(stderr, "%d %s\n", 
							i+1, image_names[i]);
					}
				}
			}
		}
		return 0;
	}

	if ((image_to_convert <= 0) || (image_to_convert > num_images)) {
		msg("Invalid image number to convert.\n");
		return -1;
	}

	if (outfile != NULL) {
        	f = fopen(outfile, "w");
		if (f == NULL) {
			msg("Failed to open file %s\n", outfile);
			return -1;
		}
	}
	else {
		f = stdout;
	}
	rv = write_XPM_file(f, outfile, images[image_to_convert-1]);
	if (rv != 0) {
		msg("Error writing xpm file\n");
	}
              
	return 0;
}

