/****
File: ico.h
        - Structures for holding ICO and image information.
Author: Laura Tweedy <tweedy@umr.edu>
Date: Tue Jan  5 12:11:41 CST 1999

Notes: Much of the structure information comes from
	"Encyclopedia of Graphic File Formats" from O'Reilly.  Quotes from
	that book are noted with "GFF".
       A "DIB" is a "Device Independent Bitmap".
****/

#ifndef INCLUDED_ICO_H
#define INCLUDED_ICO_H

#include "windows_resources.h"

#define LIBNAME "libico"
#define MAJOR_REVISION_NUMBER 1
#define MINOR_REVISION_NUMBER 2
#define LIBICO_AUTHOR "Laura Tweedy <tweedy@umr.edu>"

/** TYPES **/
typedef unsigned char	byte;
typedef unsigned short	word;
typedef unsigned int	dword;

/** STRUCTS **/
/*
An icon-resource file contains image data for icons used by Windows
applications. The file consists of an icon directory identifying the number 
and types of icon images in the file, plus one or more icon images.
--GFF
*/
/* I have changed the structure member names from the published defaults
	to more meaningful names. 
*/
typedef struct _IconDirEntry {
	byte	width;		/* width of the image in pixels */
	byte	height;		/* height of the image in pixels */
	byte	color_count;	/* number of colors in image, 0 if >= 8bpp */
	byte	reserved;	/* put your weed in here, man */
	word	color_planes;	/* color planes */
	word	bits_per_pixel;
	dword	bytes_in_resource; /* length of the DIB image */
	dword	offset_in_file;	/* where in the file is this image? */
} IconDirEntry;

typedef struct _IconDir {
	word	reserved;
	word	resource_type;	/* 1 = icon */
	word	image_count;	/* number of images in the file */
	IconDirEntry* image_entries;
} IconDir;

/* The following two structs are DIB information. */
/*
   long is signed, but the members that had type "long" should never be
   negative.  Since it helps my endian changing, I changed the following
   members' types to dword from long: width, height, x_pels_per_meter,
	y_pels_per_meter
*/
typedef struct _BitmapInfoHeader {
	dword	size;		/* number of bytes required by this struct */
	dword	width;		/* width of bitmap in pixels */
	dword	height;		/* height of bitmap in pixels */
	word	planes;		/* number of planes, must be 1 */
	word	bits_per_pixel;	/* valid values are 1,4,8,24 */
		/* all members below should be set to zero for icons */
	dword	compression;	/* type of compression: BI_RGB,BI_RLE8,BI_RLE4
				   which are 0, 1, 2 respectively.  It's
				   always zero for icons. */
	dword	image_size;	/* size in bytes */
	dword	x_pels_per_meter; /* horizontal resolution in pixels per meter*/
	dword	y_pels_per_meter; /* vertical resolution in pixels per meter */
	dword	colors_used;	/* number of color indexes actually used */
	dword	colors_important; /* number of important colors */
} BitmapInfoHeader;

typedef struct _RGBQuad {
	byte	blue;
	byte	green;
	byte	red;
	byte	reserved;	/* must be set to zero */
} RGBQuad;


/* 
   This structure is a single icon image.  The data here is normalized to
   1 byte per pixel, with a maximum of 256 colors.
*/
typedef struct _ICOImage {
	dword	width;		/* in pixels */
	dword	height;		/* in pixels */
	RGBQuad* colors;
	dword	num_colors;
	byte*	data;
	byte*	mask;
} ICOImage;


/** DLL structures for loading Icon Libraries. **/
typedef struct _NameInfo {
	word	offset;		/* offset to resource data from beginning
				   of file */
	word	length;		/* resource length in bytes */
	word	flags;		/* flags important to executables */
	word	id;		/* specifies or points to the resource 
				   identifier */
	word	handle;		/* reserved */
	word	usage;		/* reserved */
} NameInfo;

typedef struct _TypeInfo {
	word	type_id;	/* type identifier of the resource */
	word	resource_count;	/* number of resources of this type in file */
	dword	reserved;
	NameInfo* name_info;	/* information about individual resources,
				   there are resource_count number of these */
} TypeInfo;

typedef struct _ResourceTable {
	word	alignment_shift;  /* alignment shift count for resource data */
	TypeInfo types[2];  	  /* contains information about a resource,
				     one per resource type in the file */
	/* NOTE: since only icons, changing TypeInfo* to a single TypeInfo */
	word	end_types;	  /* marks end of resource type defns,
				     must be zero */
	byte*	resource_names;	  /* names assoc. with the resources, first
				     byte is number of chars in name */
	byte	end_names;	  /* marks end of resource names */
} ResourceTable;

/* My internal structure for file offsets. 
	All offsets are to the beginning of the named structure.
	All offsets are from the beginning of the file.
*/
typedef struct _TableOffsets {
	dword	win_header_start;
	dword	resource_table;
	dword	resident_name_table;
	dword	first_resource;
	dword	first_DIB_resource;
} TableOffsets;

/** Error codes - internal, use get_err_msg() to get a descriptive error **/
enum {
	icoSuccess,
	icoNoMemory,
	icoFileFormatError,
	icoFileOpenError,
	icoUnknown
};

extern int ico_error_code;

/** PROTOTYPES **/
int read_ICO_file(FILE* f, ICOImage** images);
int read_ICL_file(FILE* f, ICOImage** images, char*** image_names);
int write_XPM_file(FILE* f, char* filename, ICOImage icoimage);
char* get_err_msg();

#endif /* INCLUDED_ICO_H */
