/****
File: ico.c
        - Functions to read icons from .ico and .icl files.
Author: Laura Tweedy <tweedy@umr.edu>
Date: Tue Jan  5 12:18:00 CST 1999
****/

#include <stdio.h>
#include <sys/param.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <math.h>

#include "ico.h"
#include "color_strings.h"

/* internal return codes */
const int OK  = 1;
const int ERR = 0;

/** GLOBAL **/
int  ico_error_code;

/** PROTOTYPES **/
/* external */
int read_ICO_file(FILE* f, ICOImage** images);
int read_ICL_file(FILE* f, ICOImage** images_ptr, char*** names);
int write_XPM_file(FILE* f, char* filename, ICOImage icoimage);
char* get_err_msg();

/* internal */
int read_icon_directory(FILE* f);
int read_icon_dir_entry(FILE* f, IconDirEntry* ide);
int read_DIB_header(FILE* f, BitmapInfoHeader* bmp_header);
int read_DIB_color(FILE* f, RGBQuad* color);
int read_image_data(FILE* f, dword width, dword height, dword bpp, byte* data,
				byte* mask_data);
int read_byte(FILE* f, byte* b);
int read_word(FILE* f, word* w);
int read_double_word(FILE* f, dword* dw);
int remove_unused_colors(ICOImage* image);

dword read_MSDOS_header(FILE* f);
int read_windows_header(FILE* f, TableOffsets* offsets);
int read_resource_table(FILE* f, ResourceTable* resource_table);
int read_type_info(FILE* f, TypeInfo* type_info);
int read_name_info(FILE* f, NameInfo* name_info);
int read_resident_name_table(FILE* f, byte** resource_names, word num_res);

#ifdef DEBUG
/*
   debug()
	More usable error messages for me.
*/
void debug(char *format, ...) {
	va_list ap;

	va_start(ap, format);
	vfprintf(stderr, format, ap);
	fprintf(stderr, "\n");
	va_end(ap);
	abort;
}
#endif

/**
   get_err_msg()
	Return a pointer to the more detailed error message.
**/
char* get_err_msg() {

	switch (ico_error_code) {
	case icoSuccess:
			return LIBNAME ": Success"; break;
	case icoNoMemory:
			return LIBNAME ": No Memory"; break;
	case icoFileFormatError:
			return LIBNAME ": File Format Error"; break;
	case icoFileOpenError:
			return LIBNAME ": File Open Error"; break;
	case icoUnknown: /* fall thru */
	default:	return LIBNAME ": Unspecified Error"; break;
	}
	return LIBNAME ": Unspecified Error";
}

/**
   read_ICO_file()
	Read image from a .ico file into an array of my internal structure
	type.  Returns the number of images read, -1 if an error occured.
	I use the evil goto in this function so that I can free all the
	memory I allocate.  Errors before any mallocing is done just return.
	After that, check and free any of my memory after an error.
**/
int read_ICO_file(FILE* f, ICOImage** images_ptr) {
	IconDir icondir;
	BitmapInfoHeader bmp_header;
	ICOImage* images = NULL;
	int rv = OK;
	int i=0, j=0;

	if (f == NULL) {
		ico_error_code = icoFileOpenError;
		return -1;
	}
	if (images_ptr == NULL) {
		ico_error_code = icoNoMemory;
		return -1;
	}
	images = *images_ptr;

	/* Read Icon Directory */
	icondir.image_count = read_icon_directory(f);
	if (icondir.image_count <= 0) {
		ico_error_code = icoFileFormatError;
		#ifdef DEBUG
		debug("DEBUG: %s: %d: Error reading icon directory.", 
							__FILE__, __LINE__);
		#endif
		return -1;
	}

	/* Read Icon Directory Entries */
	icondir.image_entries = NULL;
	icondir.image_entries = (IconDirEntry *) malloc(icondir.image_count *
							sizeof(IconDirEntry));
	if (icondir.image_entries == NULL) {
		ico_error_code = icoNoMemory;
		return -1;
	}
	for (i=0; i<icondir.image_count; i++) {
		rv = read_icon_dir_entry(f, &(icondir.image_entries[i]));
		if (rv != OK) {
			ico_error_code = icoFileFormatError;
			#ifdef DEBUG
			debug("DEBUG: %s: %d: Error reading icon directory "
						"entry.", __FILE__, __LINE__);
			#endif
			goto error;
		}
	}

	/* Create ICOImages Memory */
	images = (ICOImage *) malloc(icondir.image_count * sizeof(ICOImage));
	if (images == NULL) {
		ico_error_code = icoNoMemory;
		goto error;
	}

	/* Read DIBs into ICOImages */
	for (i=0; i<icondir.image_count; i++) {
		/* Seek To DIB */
		rv = fseek(f,icondir.image_entries[i].offset_in_file, SEEK_SET);
		if (rv == -1) {
			ico_error_code = icoFileFormatError;
			#ifdef DEBUG
			debug("DEBUG: %s: %d: Error seeking to DIB.", 
							__FILE__, __LINE__);
			#endif
			goto error;
		}

		/* Read DIB Header */
		rv = read_DIB_header(f, &bmp_header);
		if (rv != OK) {
			ico_error_code = icoFileFormatError;
			#ifdef DEBUG
			debug("DEBUG: %s: %d: Error reading image header.", 
							__FILE__, __LINE__);
			#endif
			goto error;
		}

		/* Copy important DIB header info into ICOImage */
		images[i].width = bmp_header.width;
		images[i].height = bmp_header.height / 2;
		images[i].num_colors = icondir.image_entries[i].color_count;

		/* Get Colors */
		/* if num_colors is 0, it's 8bpp, which is 256 colors max */
		if (images[i].num_colors == 0) {
			images[i].num_colors = 256;
		}
		images[i].colors = NULL;
		images[i].colors = (RGBQuad *) malloc(images[i].num_colors *
							sizeof(RGBQuad));
		if (images[i].colors == NULL) {
			ico_error_code = icoNoMemory;
			goto error;
		}
		for (j=0; j<images[i].num_colors; j++) {
			rv = read_DIB_color(f, &(images[i].colors[j]));
			if (rv != OK) {
				ico_error_code = icoFileFormatError;
				#ifdef DEBUG
				debug("DEBUG: %s: %d: Error reading "
					"color info.", __FILE__, __LINE__);
				#endif
				goto error;
			}
		}

		/* Get Image and Mask Data */
		images[i].data = NULL;
		images[i].mask = NULL;
		images[i].data = (byte *) malloc((8/bmp_header.bits_per_pixel)*
						 images[i].width * 
						 images[i].height * 
						 sizeof(byte));
		images[i].mask = (byte *) malloc((8/bmp_header.bits_per_pixel)*
						 images[i].width * 
						 images[i].height * 
						 sizeof(byte));
		if ((images[i].data == NULL) || (images[i].mask == NULL)) {
			ico_error_code = icoNoMemory;
			goto error;
		}
		rv = read_image_data(f, images[i].width, images[i].height, 
					bmp_header.bits_per_pixel,
					images[i].data, images[i].mask);
		if (rv != OK) {
			ico_error_code = icoFileFormatError;
			#ifdef DEBUG
			debug("DEBUG: %s: %d: Error reading image data.",
							__FILE__, __LINE__);
			#endif
			goto error;
		}

		rv = remove_unused_colors(&(images[i]));
		if (rv != OK) {
			ico_error_code = icoUnknown;
			#ifdef DEBUG
			debug("DEBUG: %s: %d: Error reducing colors.",
							__FILE__, __LINE__);
			#endif
			goto error;
		}

	} /* end get DIB */

	*images_ptr = images;
	return icondir.image_count;

error:
	if (icondir.image_entries != NULL) free(icondir.image_entries);
	if (images != NULL) {
		for (i=0; i<bmp_header.colors_used; i++) {
			if (images[i].colors != NULL) {
				free(images[i].colors);
			}
			if (images[i].data != NULL) {
				free(images[i].data);
			}
			if (images[i].mask != NULL) {
				free(images[i].mask);
			}
		}
		free(images);
		images = NULL;
	}
	return -1;
}

/**
   read_icon_directory()
	Read icon directory information from the file pointer provided.
	Returns the number of icon images in the file.
**/
int read_icon_directory(FILE* f) {
	word tmp = 0;
	int  rv = 0;

	/* Reserved WORD, 0 is only acceptable value */
	rv = read_word(f, &tmp);
	#ifdef DEBUG
	fprintf(stderr, "reserved word is %u\n", tmp);
	#endif
	if ((rv != OK) || (tmp != 0)) return 0;

	/* Type of file, should be 1, icon type */
	rv = read_word(f, &tmp);
	#ifdef DEBUG
	fprintf(stderr, "type %u should be 1\n", tmp);
	#endif
	if ((rv != OK) || (tmp != 1)) return 0;

	/* Number of images in the file */
	rv = read_word(f, &tmp);
	#ifdef DEBUG
	fprintf(stderr, "number of images %u\n", tmp);
	#endif
	if (rv != OK) return 0;

	return tmp;
}

/**
   read_icon_dir_entry()
	Read the IconDirEntry structure from disk.
**/
int read_icon_dir_entry(FILE* f, IconDirEntry* ide) {
	int rv = OK;

	/* width */
	rv = read_byte(f, &(ide->width));
	#ifdef DEBUG
	fprintf(stderr, "width = %u\n", ide->width);
	#endif
	if (rv != OK) return ERR;

	/* height */
	rv = read_byte(f, &(ide->height));
	if (rv != OK) return ERR;
	#ifdef DEBUG
	fprintf(stderr, "height = %u\n", ide->height);
	#endif

	/* number of colors in the image */
	rv = read_byte(f, &(ide->color_count));
	#ifdef DEBUG
	fprintf(stderr, "color_count = %u\n", ide->color_count);
	#endif
	if (rv != OK) return ERR;

	/* reserved */
	rv = read_byte(f, &(ide->reserved));
	if (rv != OK) return ERR;

	/* number of color planes? */
	rv = read_word(f, &(ide->color_planes));
	#ifdef DEBUG
	fprintf(stderr, "color_planes = %u\n", ide->color_planes);
	#endif
	if (rv != OK) return ERR;

	/* bits per pixel */
	rv = read_word(f, &(ide->bits_per_pixel));
	#ifdef DEBUG
	fprintf(stderr, "bits_per_pixel= %u\n", ide->bits_per_pixel);
	#endif
	if (rv != OK) return ERR;

	/* bytes in resource */
	rv = read_double_word(f, &(ide->bytes_in_resource));
	#ifdef DEBUG
	fprintf(stderr, "bytes_in_resource = %u\n", ide->bytes_in_resource);
	#endif
	if (rv != OK) return ERR;

	/* offset in file where the image is */
	rv = read_double_word(f, &(ide->offset_in_file));
	#ifdef DEBUG
	fprintf(stderr, "offset_in_file = %u\n", ide->offset_in_file);
	#endif
	if (rv != OK) return ERR;

	return OK;
}

/**
   read_DIB_header()
	Read the BitmapInfoHeader structure members from a file.
**/
int read_DIB_header(FILE* f, BitmapInfoHeader* bmp_header) {
	int rv = 0;

	/* size */
	rv = read_double_word(f, &(bmp_header->size));
	#ifdef DEBUG_IMAGE
	fprintf(stderr, "BMP size = %u\n", bmp_header->size);
	#endif
	if (rv == 0) return ERR;

	/* width */
	rv = read_double_word(f, &(bmp_header->width));
	#ifdef DEBUG_IMAGE
	fprintf(stderr, "BMP width = %u\n", bmp_header->width);
	#endif
	if (rv == 0) return ERR;

	/* height */
	rv = read_double_word(f, &(bmp_header->height));
	#ifdef DEBUG_IMAGE
	fprintf(stderr, "BMP height = %u\n", bmp_header->height);
	#endif
	if (rv == 0) return ERR;

	/* number of color planes */
	rv = read_word(f, &(bmp_header->planes));
	#ifdef DEBUG_IMAGE
	fprintf(stderr, "BMP planes = %u\n", bmp_header->planes);
	#endif
	if ((rv == 0) || (bmp_header->planes != 1)) return ERR;

	/* bits per pixel */
	rv = read_word(f, &(bmp_header->bits_per_pixel));
	#ifdef DEBUG_IMAGE
	fprintf(stderr, "BMP bpp = %u\n", bmp_header->bits_per_pixel);
	#endif
	if (rv == 0) return ERR;

	#ifdef DEBUG_IMAGE
	fprintf(stderr, "BMP All the following BMP values should be zero\n");
	fprintf(stderr, "BMP but since some icons aren't zero here\n");
	fprintf(stderr, "BMP ignore the values we get\n");
	#endif

	/* compression */
	rv = read_double_word(f, &(bmp_header->compression));
	#ifdef DEBUG_IMAGE
	fprintf(stderr, "BMP compression = %u\n", bmp_header->compression);
	#endif
	if (rv == 0) return ERR;

	/* image size in bytes */
	rv = read_double_word(f, &(bmp_header->image_size));
	#ifdef DEBUG_IMAGE
	fprintf(stderr, "BMP image size = %u\n", bmp_header->image_size);
	#endif
	if (rv == 0) return ERR;

	/* horiz pixels per meter */
	rv = read_double_word(f, &(bmp_header->x_pels_per_meter));
	#ifdef DEBUG_IMAGE
	fprintf(stderr, "BMP x pels per meter = %u\n", 
				bmp_header->x_pels_per_meter);
	#endif
	if (rv == 0) return ERR;

	/* vert pixels per meter */
	rv = read_double_word(f, &(bmp_header->y_pels_per_meter));
	#ifdef DEBUG_IMAGE
	fprintf(stderr, "BMP y pels per meter = %u\n", 
				bmp_header->y_pels_per_meter);
	#endif
	if (rv == 0) return ERR;

	/* number of color indicies used */
	rv = read_double_word(f, &(bmp_header->colors_used));
	#ifdef DEBUG_IMAGE
	fprintf(stderr, "BMP colors used = %u\n", bmp_header->colors_used);
	#endif
	if (rv == 0) return ERR;

	/* number of important colors */
	rv = read_double_word(f, &(bmp_header->colors_important));
	#ifdef DEBUG_IMAGE
	fprintf(stderr, "BMP colors important = %u\n", 
				bmp_header->colors_important);
	#endif
	if (rv == 0) return ERR;

	return OK;
}

/**
   read_DIB_color()
	Read an RGBQuad structure from a file.
**/
int read_DIB_color(FILE* f, RGBQuad* color) {
	int rv = OK;

	/* RGB and the order is blue, green, red??? */
	/* blue */
	rv = read_byte(f, &(color->blue));
	#ifdef DEBUG_COLORS
	fprintf(stderr, "RGB blue = %2.2x  ", color->blue);
	#endif
	if (rv != OK) return ERR;

	/* green */
	rv = read_byte(f, &(color->green));
	#ifdef DEBUG_COLORS
	fprintf(stderr, "green = %2.2x  ", color->green);
	#endif
	if (rv != OK) return ERR;

	/* red */
	rv = read_byte(f, &(color->red));
	#ifdef DEBUG_COLORS
	fprintf(stderr, "red = %2.2x  ", color->red);
	#endif
	if (rv != OK) return ERR;

	/* reserved */
	rv = read_byte(f, &(color->reserved));
	#ifdef DEBUG_COLORS
	fprintf(stderr, "reserved (should be 0)= %d\n", color->reserved);
	#endif
	if ((rv != OK) || (color->reserved != 0)) return ERR;

	return OK;
}
/**
   read_image_data()
	Read the image data from file.  Image is upside down in the file,
	so, correct it.  Also, normalize to 1 byte per pixel.
	Read the mask and deal with it.
	I know this function looks huge, but there's alot of debug loops
	and nesting which makes it longer than it really is.
**/
int read_image_data(FILE* f, dword width, dword height, dword bpp, byte* data,
				byte* mask_data){
	int	i, j, k;
	int	rv = OK;
	/* cheesy, but I don't want to deal w/ malloc and this mess */
	dword	bytes_per_line = width * bpp /8;
	byte	buf[height*bytes_per_line];
	byte	image[width*height];

	dword	mask_bytes_per_line = width * 1/8;
	byte	mask_buf[height*mask_bytes_per_line];
	byte	mask[width*height];

	/* Get the image data */
	rv = fread(&buf, sizeof(byte), bytes_per_line * height, f);
	if (rv == 0) return ERR;
	#ifdef DEBUG_IMAGE
	j=0;
	for (i=0; i<bytes_per_line*height; i++) {
		fprintf(stderr, "%2.2x", buf[i]);
		j++;
		if (j == bytes_per_line) {
			fprintf(stderr, "\n");
			j=0;
		}
	}
	#endif

	/* Normalize if neccesary */
	if (bpp == 8) {
		for (i=0; i<bytes_per_line*height; i++) {
			image[i] = buf[i];
		}
	}
	else if (bpp == 4) {
		for (i=0; i<bytes_per_line*height; i++) {
			image[i*2]   = buf[i] >> 4;
			image[i*2+1] = buf[i] << 4;
			image[i*2+1] = image[i*2+1] >> 4;
		}
	}
	else {
		ico_error_code = icoFileFormatError;
		#ifdef DEBUG
		debug("DEBUG: %s: %d: Monochrome image.", __FILE__, __LINE__);
		#endif
		return ERR;
	}
	#ifdef DEBUG_IMAGE
	j=0;
	for (i=0; i<width*height; i++) {
		fprintf(stderr, "%2.2x", image[i]);
		j++;
		if (j == width) {
			fprintf(stderr, "\n");
			j=0;
		}
	}
	#endif

	/* Flip the image over */
	k=0;
	for (i=0; i<height; i++) {
		k=(width * height) - (width * (i+1));
		for(j=0; j<width; j++) {
			data[k+j] = image[(i*width)+j];
		}
	}
	#ifdef DEBUG_IMAGE
	j=0;
	for (i=0; i<width*height; i++) {
		fprintf(stderr, "%2.2x", data[i]);
		j++;
		if (j == width) {
			fprintf(stderr, "\n");
			j=0;
		}
	}
	#endif

	/* Read transparency mask */
	rv = fread(&mask_buf, sizeof(byte), mask_bytes_per_line * height, f);
	if (rv == 0) return ERR;

	/* Normalize to 1 bpp */
	for (i=0; i<mask_bytes_per_line * height; i++) {
		byte b = 0;
		for (j=0; j<8; j++) {
			b = mask_buf[i] << j;
			b = b >> 7;
			mask[i*8+j] = b;
		}
	}
	/* Flip the mask over */
	k=0;
	for (i=0; i<height; i++) {
		k=(width * height) - (width * (i+1));
		for(j=0; j<width; j++) {
			mask_data[k+j] = mask[(i*width)+j];
		}
	}
	#ifdef DEBUG_IMAGE
	j=0;
	for (i=0; i<width*height; i++) {
		fprintf(stderr, "%2.2x", mask_data[i]);
		j++;
		if (j == width) {
			fprintf(stderr, "\n");
			j=0;
		}
	}
	#endif

	return OK;
}


/*
   read_byte()
	Wrapper to fread.  Endian-ness is not an issue for a single byte,
	but I wanted to make reading uniform in interface.
*/
int read_byte(FILE* f, byte* b) {
	int rv = OK;

	rv = fread(b, sizeof(byte), 1, f);
	if (rv == 0) return ERR;

	return OK;
}

/*
   read_word()
	Reads a little endian word and stores it in proper big endian.
*/
int read_word(FILE* f, word* w) {
	byte buf[2];
	int  rv = OK;

	#ifndef _BIG_ENDIAN
		rv = fread(w, sizeof(word), 1, f);
		return rv;
	#endif

	/* read LSB */
	rv = fread(&(buf[0]), sizeof(byte), 1, f);
	if (rv == 0) return ERR;

	/* read MSB */
	rv = fread(&(buf[1]), sizeof(byte), 1, f);
	if (rv == 0) return ERR;

	/* put MSB in lowest address position and add LSB */
	*w = (buf[1] << 8) + buf[0];

	return OK;
}

/*
   read_double_word()
	Reads a little endian double word and stores it in proper big endian.
*/
int read_double_word(FILE* f, dword* dw) {
	byte buf[4];
	int  rv = OK;
	int  i;

	#ifndef _BIG_ENDIAN
		rv = fread(dw, sizeof(dword), 1, f);
		return rv;
	#endif

	/* read 4 bytes from LSB to MSB */
	for (i=0; i<4; i++) {
		rv = fread(&(buf[i]), sizeof(byte), 1, f);
		if (rv == 0) return ERR;
	}

	/* put bytes in the target in MSB first ordering */
	*dw = (buf[3] << 24) + (buf[2] << 16) + (buf[1] << 8) + buf[0];

	return OK;
}

/**
   write_XPM_file()
        Construct and write out pixmap.
**/
int write_XPM_file(FILE* f, char* filename, ICOImage icoimage) {
        int i = 0;
        int j = 0;
	int index = 0;
	char *iname = NULL;
	char *s = NULL;
	const char *trans_color_string = "  ";

	if (f == NULL) {
		ico_error_code = icoFileOpenError;
		return -1;
	}

	/* try and make a c-style name */
	if (filename == NULL) {
		iname = strdup("image_name");
	}
	else {
		iname = strdup(filename);
		while ( (s = strchr(iname, '.')) != NULL ) {
			*s = '_';
		}
		while ( (s = strchr(iname, '-')) != NULL ) {
			*s = '_';
		}
		#ifdef DEBUG
		debug("DEBUG: %s: %d: C-style filename is %s", 
						__FILE__, __LINE__, iname);
		#endif
	}

	/* Make xpm header */
	/* add one to number of colors for transparent color */
	/* always 2 chars per pixel b/c we normalized */
	fprintf(f,
		"/* XPM */\n"
		"static char * %s[] = {\n"
		"\"%d %d %d %d\",\n",
		iname, icoimage.width, icoimage.height, 
		icoimage.num_colors + 1, 2);

	/* first, transparent color */
	fprintf(f, "\"%s c None\",\n", trans_color_string);

	/* then, the other colors */
	for (i=0; i<icoimage.num_colors; i++) {
		fprintf(f,
			"\"%s c #%2.2x%2.2x%2.2x\",\n",
			color_strings[icoimage.colors[i].reserved], 
			icoimage.colors[i].red, icoimage.colors[i].green, 
			icoimage.colors[i].blue);
	}

	/* Make the image */
	for (i=0; i<icoimage.height; i++) {
		fprintf(f, "\"");
		for (j=0; j<icoimage.width; j++) {
			index = i*icoimage.width + j;

			/* check for transparent pixel */
			if (icoimage.mask[index] == 1) {
				fprintf(f, "%s", trans_color_string);
			}
			else {
				fprintf(f, "%s", 
					color_strings[icoimage.data[index]]);
			}
		}
		/* last line? */
		if (j == icoimage.height - 1) {
			fprintf(f, "\"\n");
		}
		else {
			fprintf(f, "\",\n");
		}
	}
        fprintf(f, "};\n");

        return 0;
}
/**
   remove_unused_colors()
	Remove colors from the color list that aren't used in the picture.
**/
int remove_unused_colors(ICOImage* image) {
	int i = 0;
	int j = 0;
	int num_used_colors = 0;
	RGBQuad* used_colors = NULL;

	/* flag used colors in original color map */
	for (i=0; i<image->width*image->height; i++) {
		if (image->colors[image->data[i]].reserved != 1) {
			image->colors[image->data[i]].reserved = 1;
			num_used_colors++;
		}
	}

	/* make space for used colors array */
	used_colors = (RGBQuad *) malloc(num_used_colors * sizeof(RGBQuad));
	if (used_colors == NULL) {
		return ERR;
	}

	/* if color is flagged, copy into used color array */
	j = 0;
	for (i=0; i<image->num_colors; i++) {
		if (image->colors[i].reserved == 1) {
			used_colors[j] = image->colors[i];
			used_colors[j].reserved = i;
			j++;
		}
	}
	/* remove old colors array and use used colors array instead */
	free(image->colors);
	image->colors = used_colors;
	image->num_colors = num_used_colors;

	return OK;
}

/**
   read_ICL_file
	Read icon images from an Icon Library (.icl files) into an array of
	my internal structure type. .icl files are really just renamed .dll 
	files with only a resource part and no code.
	Returns the number of images read, -1 if an error occured.
**/
int read_ICL_file(FILE* f, ICOImage** images_ptr, char*** names) {
	BitmapInfoHeader bmp_header;
	TableOffsets  offsets;
	ResourceTable resource_table;
	byte** 	      resource_names = NULL;
	word	      resource_count = 0;
	IconDir	      icondir;
	ICOImage*     images = NULL;
	int rv = OK;
	int i=0, j=0;

	if (f == NULL) {
		ico_error_code = icoFileOpenError;
		return -1;
	}
	if (images_ptr == NULL) {
		ico_error_code = icoNoMemory;
		return -1;
	}
	images = *images_ptr;

	/* Read MS-DOS file header */
	offsets.win_header_start = read_MSDOS_header(f);
	if (offsets. win_header_start == 0) {
		ico_error_code = icoFileFormatError;
		#ifdef DEBUG
		debug("DEBUG: %s: %d: Error reading MS-DOS file header.",
						__FILE__, __LINE__);
		#endif
		return -1;
	}

	/* Read Windows file header */
	rv = fseek(f, offsets.win_header_start, SEEK_SET);
	if (rv == -1) {
		ico_error_code = icoFileFormatError;
		#ifdef DEBUG
		debug("DEBUG: %s: %d: Error seeking to windows header.",
						__FILE__, __LINE__);
		#endif
		return -1;
	}
	rv = read_windows_header(f, &offsets);
	if (rv != OK) {
		ico_error_code = icoFileFormatError;
		#ifdef DEBUG
		debug("DEBUG: %s: %d: Error reading windows header.",
						__FILE__, __LINE__);
		#endif
		return -1;
	}

	/* Read Resource table */
	rv = fseek(f, offsets.resource_table, SEEK_SET);
	if (rv == -1) {
		ico_error_code = icoFileFormatError;
		#ifdef DEBUG
		debug("DEBUG: %s: %d: Error seeking to resource table.",
						__FILE__, __LINE__);
		#endif
		return -1;
	}
	rv = read_resource_table(f, &resource_table);
	if (rv != OK) {
		ico_error_code = icoFileFormatError;
		#ifdef DEBUG
		debug("DEBUG: %s: %d: Error reading resource table.",
						__FILE__, __LINE__);
		#endif
		return -1;
	}
	resource_count = resource_table.types[0].resource_count;

	/* Check for a resident name table and read it */
	/* NOTE: there is probably a better way to do this.  I check
		if the claimed resident name table offset is smaller
		than the first resource data offset.  If so, there is a
		name table, otherwise, assume no name table. */

	offsets.first_resource = resource_table.types[0].name_info[0].offset * 
					resource_table.alignment_shift;
	if (offsets.resident_name_table < offsets.first_resource) {

		rv = fseek(f, offsets.resident_name_table, SEEK_SET);
		if (rv == -1) {
			ico_error_code = icoFileFormatError;
			#ifdef DEBUG
			debug("DEBUG: %s: %d: Error seeking to resident "
					"name table.", __FILE__, __LINE__);
			#endif
			return -1;
		}

		resource_table.resource_names = NULL;
		resource_names = NULL;
		resource_names = (byte **) 
				 malloc(resource_count * sizeof(byte *));
		if (resource_names == NULL) {
			ico_error_code = icoNoMemory;
			return -1;
		}
		rv = read_resident_name_table(f,resource_names, resource_count);
		if (rv != OK) {
			ico_error_code = icoFileFormatError;
			#ifdef DEBUG
			debug("DEBUG: %s: %d: Error reading resident "
					"name table.", __FILE__, __LINE__);
			#endif
			goto error;
		}
		/* read end of name table flag */
		rv = read_byte(f, &(resource_table.end_names));
		if (rv != OK) {
			ico_error_code = icoFileFormatError;
			#ifdef DEBUG
			debug("DEBUG: %s: %d: Error end name table flag.",
					__FILE__, __LINE__);
			#endif
			goto error;
		}
	}

	/* now, we should be ready to read in the actual resources */
	/* if not, just ignore it since we will seek to images */
	#ifdef DEBUG
	if (ftell(f) != offsets.first_resource) {
		debug("DEBUG: %s: %d: File not naturally aligned at "
				"resource data.", __FILE__, __LINE__);
		debug("DEBUG: FILE* is:  %lx", ftell(f));
		debug("DEBUG: should be: %x", offsets.first_resource);
	}
	#endif

	/* Seek to beginning of icon dir entries */
	rv = fseek(f, offsets.first_resource, SEEK_SET);
	if (rv == -1) {
		ico_error_code = icoFileFormatError;
		#ifdef DEBUG
		debug("DEBUG: %s: %d: Error seeking to icon dir entries.",
						__FILE__, __LINE__);
		#endif
		goto error;
	}

#ifdef DEBUG
debug("FIXME!! %s: %d: FIXME: icondirs are being faked",__FILE__, __LINE__);
#endif
	/* Read Icon Directories and Entries */
	icondir.image_entries = NULL;
	icondir.image_entries = (IconDirEntry *) malloc(resource_count *
							sizeof(IconDirEntry));
	if (icondir.image_entries == NULL) {
		ico_error_code = icoNoMemory;
		goto error;
	}
	for (i=0; i<resource_count; i++) {
		rv = fseek(f, resource_table.types[0].name_info[i].offset *
				resource_table.alignment_shift, SEEK_SET);
		if (rv == -1) {
			ico_error_code = icoFileFormatError;
			#ifdef DEBUG
			debug("DEBUG: %s: %d: Error seeking to icon directory.",
							__FILE__, __LINE__);
			#endif
			goto error;
		}
		icondir.image_count = read_icon_directory(f);
		if (icondir.image_count <= 0) {
			ico_error_code = icoFileFormatError;
			#ifdef DEBUG
			debug("DEBUG: %s: %d: Error reading icon directory.", 
							__FILE__, __LINE__);
			#endif
			goto error;
		}
		rv = read_icon_dir_entry(f, &(icondir.image_entries[i]));
		if (rv != OK) {
			ico_error_code = icoFileFormatError;
			#ifdef DEBUG
			debug("DEBUG: %s: %d: Error reading icon directory "
						"entry.", __FILE__, __LINE__);
			#endif
			goto error;
		}
	}

	/* Create ICOImages Memory */
	images = (ICOImage *) malloc(resource_count * sizeof(ICOImage));
	if (images == NULL) {
		ico_error_code = icoNoMemory;
		goto error;
	}

	/* Read DIBs into ICOImages */
	for (i=0; i<resource_count; i++) {
		/* Seek To DIB */
		rv = fseek(f, resource_table.types[1].name_info[i].offset *
				resource_table.alignment_shift, SEEK_SET);
		if (rv == -1) {
			ico_error_code = icoFileFormatError;
			#ifdef DEBUG
			debug("DEBUG: %s: %d: Error seeking to DIB.",
							__FILE__, __LINE__);
			#endif
			goto error;
		}

		/* Read DIB Header */
		rv = read_DIB_header(f, &bmp_header);
		if (rv != OK) {
			ico_error_code = icoFileFormatError;
			#ifdef DEBUG
			debug("DEBUG: %s: %d: Error reading DIB.",
							__FILE__, __LINE__);
			#endif
			goto error;
		}

		/* Copy important DIB header info into ICOImage */
		images[i].width = bmp_header.width;
		images[i].height = bmp_header.height / 2;
		images[i].num_colors = icondir.image_entries[i].color_count;

		/* Get Colors */
		/* if num_colors is 0, it's 8bpp, which is 256 colors max */
		if (images[i].num_colors == 0) {
			images[i].num_colors = 256;
		}
		images[i].colors = NULL;
		images[i].colors = (RGBQuad *) malloc(images[i].num_colors *
							sizeof(RGBQuad));
		if (images[i].colors == NULL) {
			ico_error_code = icoNoMemory;
			goto error;
		}
		for (j=0; j<images[i].num_colors; j++) {
			rv = read_DIB_color(f, &(images[i].colors[j]));
			if (rv != OK) {
				ico_error_code = icoFileFormatError;
				#ifdef DEBUG
				debug("DEBUG: %s: %d: Error reading colors.",
							__FILE__, __LINE__);
				#endif
				goto error;
			}
		}

		/* Get Image and Mask Data */
		images[i].data = NULL;
		images[i].mask = NULL;
		images[i].data = (byte *) malloc((8/bmp_header.bits_per_pixel)*
						 images[i].width * 
						 images[i].height * 
						 sizeof(byte));
		images[i].mask = (byte *) malloc((8/bmp_header.bits_per_pixel)*
						 images[i].width * 
						 images[i].height * 
						 sizeof(byte));
		if ((images[i].data == NULL) || (images[i].mask == NULL)) {
			ico_error_code = icoNoMemory;
			goto error;
		}
		rv = read_image_data(f, images[i].width, images[i].height, 
					bmp_header.bits_per_pixel,
					images[i].data, images[i].mask);
		if (rv != OK) {
			ico_error_code = icoFileFormatError;
			#ifdef DEBUG
			debug("DEBUG: %s: %d: Error reading image data.",
							__FILE__, __LINE__);
			#endif
			goto error;
		}

		rv = remove_unused_colors(&(images[i]));
		if (rv != OK) {
			ico_error_code = icoUnknown;
			#ifdef DEBUG
			debug("DEBUG: %s: %d: Error reducing colors.",
							__FILE__, __LINE__);
			#endif
			goto error;
		}

	} /* end get DIB */

	*images_ptr = images;
	*names = (char **) resource_names;
	return resource_count;

error:
	if (icondir.image_entries != NULL) free(icondir.image_entries);
	if (resource_names != NULL) {
		for (i=0; i<resource_count; i++) {
			if (resource_names[i] != NULL) {
				free(resource_names[i]);
			}
		}
		free(resource_names);
	}
	if (images != NULL) {
		for (i=0; i<bmp_header.colors_used; i++) {
			if (images[i].colors != NULL) {
				free(images[i].colors);
			}
			if (images[i].data != NULL) {
				free(images[i].data);
			}
			if (images[i].mask != NULL) {
				free(images[i].mask);
			}
		}
		free(images);
		images = NULL;
	}
	return -1;
}

/**
   read_MSDOS_header
	Reads the MS-DOS header from the icon library.
	Returns the offset from the beginning of the file to the windows
	header.
**/
dword read_MSDOS_header(FILE* f) {
	dword offset = 0;
	word  tmp = 0;
	int   rv = 0;

	/* Signature word, should be equal to 0x5a4d "ZM" */
	rv = read_word(f, &tmp);
	#ifdef DEBUG
	fprintf(stderr, "signature word is %x\n", tmp);
	#endif
	if ((rv != OK) || (tmp != 0x5a4d)) return 0;

	/* Skip other header info since we don't need it */
	rv = fseek(f, 0x3C, SEEK_SET);
	if (rv == -1) {
		return 0;
	}

	/* Get offset of windows header */
	rv = read_double_word(f, &offset);
	#ifdef DEBUG
	fprintf(stderr, "offset to windows header is %x\n", offset);
	#endif
	if (rv != OK) return 0;

	return offset;
}

/**
   read_windows_header
	Read the Windows library header.
**/
int read_windows_header(FILE* f, TableOffsets* offsets) {
	word  tmp = 0;
	int   rv = 0;

	/* Signature word, should be equal to 0x454e "EN" */
	rv = read_word(f, &tmp);
	#ifdef DEBUG
	fprintf(stderr, "windows signature word is %x\n", tmp);
	#endif
	if ((rv != OK) || (tmp != 0x454e)) return ERR;

	/* Skip to the next useful information */
	rv = fseek(f, offsets->win_header_start + 0x24, SEEK_SET);
	if (rv == -1) return ERR;

	/* Read offset to resource table */
	rv = read_word(f, &tmp);
	offsets->resource_table = tmp + offsets->win_header_start;
	#ifdef DEBUG
	fprintf(stderr, "resource table at %x from win_header start\n", 
					offsets->resource_table);
	#endif
	if (rv != OK) return ERR;

	/* Read offset to resident-name table */
	rv = read_word(f, &tmp);
	offsets->resident_name_table = tmp + offsets->win_header_start;
	#ifdef DEBUG
	fprintf(stderr, "resident-name table at %x from win_header start\n", 
					offsets->resident_name_table);
	#endif
	if (rv != OK) return ERR;

	return OK;
}

/**
   read_resource_table()
	Fill out resource table.  The resource table holds offsets, names
	and more about the icons contained in the file.
**/
int read_resource_table(FILE* f, ResourceTable* resource_table) {
	word tmp_word = 0;
	byte tmp_byte = 0;
	int  rv = OK;

	/* Read alignment shift */
	rv = read_word(f, &tmp_word);
	resource_table->alignment_shift = pow(2, tmp_word);
	#ifdef DEBUG
	fprintf(stderr, "alignment shift is %x\n", tmp_word);
	fprintf(stderr, "usable offset modifier is %x\n", 
				resource_table->alignment_shift);
	#endif
	if (rv != OK) return ERR;

	/* Read resource type information structures */
/* FIXME keep reading type info structures until done?? */
	/* group icon typeinfo */
	rv = read_type_info(f, &(resource_table->types[0]));
	if (rv != OK) return ERR;

	/* icon typeinfo */
	rv = read_type_info(f, &(resource_table->types[1]));
	if (rv != OK) return ERR;

	/* read end names flag */
	rv = read_byte(f, &tmp_byte);
	if (rv != OK) return ERR;

	return OK;
}

/**
   read_type_info()
	Read type info structures.  The type info structures contain 
	information about the type of resource we're looking at.
**/
int read_type_info(FILE* f, TypeInfo* type_info) {
	dword dtmp = 0;
	word  tmp  = 0;
	int rv = 0;
	int i;

	/* read type id */
	rv = read_word(f, &(type_info->type_id));
	#ifdef DEBUG
	fprintf(stderr, "type id is %x\n", type_info->type_id);
	#endif
	if (rv != OK) return ERR;

	/* if highest bit is set, it's a resource number */
	if (type_info->type_id >> 15 == 1) {
		/* clear highest bit, leaving resource id number */
		tmp = type_info->type_id << 1;
		type_info->type_id = tmp >> 1;
		#ifdef DEBUG
		fprintf(stderr, "resource ID is %d\n", type_info->type_id);
		#endif
	}

	/* read resource count */
	rv = read_word(f, &(type_info->resource_count));
	#ifdef DEBUG
	fprintf(stderr, "resource count is %d\n", type_info->resource_count);
	#endif
	if (rv != OK) return ERR;

	/* read reserved area */
	rv = read_double_word(f, &dtmp);
	if (rv != OK) return ERR;

	/* make space for name info structures */
	type_info->name_info = NULL;
	type_info->name_info = (NameInfo *) malloc(type_info->resource_count *
							sizeof(NameInfo));
	if (type_info->name_info == NULL) {
		ico_error_code = icoNoMemory;
		return ERR;
	}

	/* read name info */
	for (i=0; i<type_info->resource_count; i++) {
		rv = read_name_info(f, &(type_info->name_info[i]));
		if (rv != OK) return ERR;
	}

	return OK;
}

/**
   read_name_info()
**/
int read_name_info(FILE* f, NameInfo* name_info) {
	word tmp = 0;
	int rv = 0;

	/* read offset to resource data from beginning of file */
	rv = read_word(f, &(name_info->offset));
	#ifdef DEBUG_RESOURCE
	fprintf(stderr, "offset to resource data from file start %x\n",
					name_info->offset);
	#endif
	if (rv != OK) return ERR;

	/* read length of resource in bytes */
	rv = read_word(f, &(name_info->length));
	#ifdef DEBUG_RESOURCE
	fprintf(stderr, "length of resource in bytes %d\n", name_info->length);
	#endif
	if (rv != OK) return ERR;

	/* read flags, unused in icon libraries */
	rv = read_word(f, &(name_info->length));
	if (rv != OK) return ERR;

	/* read resource id pointer */
	rv = read_word(f, &(name_info->id));
	if (rv != OK) return ERR;

	/* if highest bit is set, it's a resource counter/index number */
	if (name_info->id >> 15 == 1) {
		/* clear highest bit, leaving resource id number */
		tmp = name_info->id << 1;
		name_info->id = tmp >> 1;
		#ifdef DEBUG_RESOURCE
		fprintf(stderr, "resource index is %d\n", name_info->id);
		#endif
	}

	/* read handle, unused in icon libraries */
	rv = read_word(f, &(name_info->handle));
	if (rv != OK) return ERR;

	/* read usage, unused in icon libraries */
	rv = read_word(f, &(name_info->usage));
	if (rv != OK) return ERR;

	return OK;
}

/**
   read_resident_name_table()
	Read names associated with resources.
**/
int read_resident_name_table(FILE* f, byte** resource_names, word num_res) {
	byte* tmp = NULL;
	byte len = 0;
	int  rv  = 0;
	int i;

	/* get and discard module name */
	rv = read_byte(f, &len);
	if (rv != OK) return ERR;

	tmp = (byte *) malloc((len + 1) * sizeof(byte));
	if (tmp == NULL) return ERR;

	rv = fread(tmp, sizeof(byte), len, f);
	if (rv == 0) {
		free(tmp);
		return ERR;
	}
	#ifdef DEBUG_RESOURCE
	tmp[len] = '\0';
	fprintf(stderr, "Module name is %s\n", tmp);
	#endif

	free(tmp);

	/* get other resource names */
	for (i=0; i<num_res; i++) {
		rv = read_byte(f, &len);
		if (rv != OK) return ERR;

		resource_names[i] = NULL;
		resource_names[i] = (byte *) malloc((len + 1) * sizeof(byte));
		if (resource_names[i] == NULL) {
			ico_error_code = icoNoMemory;
			goto error;
		}

		rv = fread(resource_names[i], sizeof(byte), len, f);
		if (rv == 0) {
			ico_error_code = icoFileFormatError;
			goto error;
		}
		resource_names[i][len] = '\0';

		#ifdef DEBUG_RESOURCE
		fprintf(stderr, "resource name is %s\n", resource_names[i]);
		#endif

	}

	return OK;

error:
	for (i=0; i<num_res; i++) {
		if (resource_names[i] != NULL) {
			free(resource_names[i]);
		}
	}

	return ERR;
}
