/**
 ** LNXMODE.C  --  linux mode setting
 **
 **  Copyright (C) 1992, Csaba Biegl
 **    820 Stirrup Dr, Nashville, TN, 37221
 **    csaba@vuse.vanderbilt.edu
 **
 **  Copyright (C) 1993, Hartmut Schirmer,
 **    Feldstr. 118, 2300 Kiel 1, Germany
 **    PHC27@rz.uni-kiel.dbp.de
 **
 **  This file is distributed under the terms listed in the document
 **  "copying.cb", available from the author at the address above.
 **  A copy of "copying.cb" should accompany this file; if not, a copy
 **  should be available from where this file was obtained.  This file
 **  may not be distributed without a verbatim copy of "copying.cb".
 **  You should also have received a copy of the GNU General Public
 **  License along with this program (it is in the file "copying");
 **  if not, write to the Free Software Foundation, Inc., 675 Mass Ave,
 **  Cambridge, MA 02139, USA.
 **
 **  This program is distributed in the hope that it will be useful,
 **  but WITHOUT ANY WARRANTY; without even the implied warranty of
 **  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 **  GNU General Public License for more details.
 **/

#ifndef __linux__
#error lnxmode.c for Linux use only !
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <termios.h>
#include <sys/ioctl.h>

#include "vga.h"
#include "grx.h"
#include "libgrx.h"
#include "grdriver.h"
#include "grxfile.h"
/* #include "gppconio.h" */
#include "reg8514a.h"
#define ALLBLITOPS (HAVE_BITBLIT | HAVE_FILLBLIT | HAVE_IMAGEBLIT | HAVE_HLINELISTBLIT)
#define swap(a,b) ({typeof(a) temp; temp = (a); (a) = (b); (b) = temp; })
	
/* structure used in screen updates */
struct dirty {
	unsigned char	flag;
	int		starting;
	int		ending;
};
static int                  ModeXflag;
static GrDriverHeader       drv;
static GR_DRIVER_MODE_ENTRY *GM;		/* graphics mode table */
static GR_DRIVER_MODE_ENTRY TM[2];		/* text mode table */
static unsigned short       *driver_flags;	/* driver flags table */
static unsigned int	    *mode_flags;	/* SVGALIB mode flags */
static int                  Text_Mode = TRUE;
static int                  LastMode = 0;
static int                  RW_page  = 0;	/* != 0 -> have seperate RD/WR pages */
static int		    start,end;
static struct dirty         *Dirty = (struct dirty *)NULL; 
static vga_modeinfo         *current = (vga_modeinfo *)NULL; 
static int		    Accel = FALSE;


static void SetupMemory()
{
    int i;

    Dirty = (struct dirty *) malloc(sizeof(struct dirty) * _GrScreenY); 
    start = 0; end = 0;
    for (i = 0; i < _GrScreenY; i++)
    {
        Dirty[i].flag= FALSE;
        Dirty[i].starting = 0;
        Dirty[i].ending   = 0;
    }
}



static void  AccelSetup(int mode)
{
    vga_modeinfo *VgaInfo;    
    int size, i;
    int data, VideoChip;
    char *AccEnv;

    
    AccEnv = getenv("GRXACCEL");

    if (AccEnv != NULL)
        for (i = 0; AccEnv[i] != 0; i++)
            AccEnv[i] = toupper(AccEnv[i]);
    else
    {
                printf("Acceletator Functions Disabled.\n");
                Accel = FALSE;
                drv.driver_flags &= ~(GRD_TYPE_MASK);
                drv.driver_flags |= GRD_NEW_DRIVER | GRD_VGA;
                return;
            }
    	
    VgaInfo = vga_getmodeinfo(mode);
    VideoChip = vga_getcurrentchipset();
    switch (VideoChip)
    {
    	case MACH32:
	    printf("Grx Detected ATI Mach32 video card\n");

            if ((AccEnv == NULL))
            {
                printf("Acceletator Functions Disabled Useing Vga\n");
                Accel = FALSE;
                drv.driver_flags &= ~(GRD_TYPE_MASK);
                drv.driver_flags |= GRD_NEW_DRIVER | GRD_VGA;
                return;
            }
            if (strcmp(AccEnv,"FALSE") == 0)
            {
                printf("GRXACCEL = False Disabled Accel functions.\n");
                Accel = FALSE;
                drv.driver_flags &= ~(GRD_TYPE_MASK);
                drv.driver_flags |= GRD_NEW_DRIVER | GRD_VGA;
                return;
            }

	    if ((VgaInfo->haveblit & ALLBLITOPS) == ALLBLITOPS)
            {
	        printf("    Video Card Allows Blit Operations\n");
                if (VgaInfo->flags & CAPABLE_LINEAR)
                {
		     GrSetMemCfg(LINEAR_MODE);
		     GrSetFlush(MANUALFLUSH);
		     vga_setlinearaddressing();
                }
	        printf("    Video Card Set for %d\n",VgaInfo->colors);
	        switch (VgaInfo->colors)
                {
                     case 256:
                         Accel = TRUE;
                         drv.driver_flags &= ~(GRD_TYPE_MASK);
                         drv.driver_flags |= GRD_NEW_DRIVER | GRD_8514A;
                         vga_fillblt(0,VgaInfo->width, VgaInfo->height, VgaInfo->linewidth, 0);
                         break;
                     case 32768:
                     case 65536:
                         Accel = TRUE;
                         /* We do the following in order to ensure that */
                         /* the Mach32 is set up correctly.             */
                         drv.driver_flags &= ~(GRD_TYPE_MASK);
                         drv.driver_flags |= GRD_NEW_DRIVER | GRD_MACH32;
                         vga_fillblt(0,VgaInfo->width, VgaInfo->height, VgaInfo->linewidth, 0);
                         break;
                     default:
                         Accel = FALSE;
                         drv.driver_flags &= ~(GRD_TYPE_MASK);
                         drv.driver_flags |= GRD_NEW_DRIVER | GRD_VGA;
                }
                return;
            }
            Accel = FALSE;
            drv.driver_flags &= ~(GRD_TYPE_MASK);
            drv.driver_flags |= GRD_NEW_DRIVER | GRD_VGA;
            return;

        case VGA:
        case ET4000:
        case CIRRUS:
        case TVGA8900:
        case OAK:
        case EGA:
        case S3:
        case ET3000:
        case GVGA6400:
        case ARK:
        case ATI:
        default:
            Accel = FALSE;
            drv.driver_flags &= ~(GRD_TYPE_MASK);
            drv.driver_flags |= GRD_NEW_DRIVER | GRD_VGA;
     }
}




void GrFlush()
{
    int len, i, StartAt;

    if (Accel == TRUE) return;
    if (GrGetMemCfg() == LINEAR_MODE)
        return;
    if (start == 0 && end == 0)
        return;


    for ( i = start; i <= end; i++)
    {
        if (Dirty[i].flag == TRUE)
        {
            Dirty[i].flag = FALSE;
            if (_GrNumColors == 256 )
            {
                SORT2(Dirty[i].starting, Dirty[i].ending);
                len = Dirty[i].ending - Dirty[i].starting;
                StartAt = Dirty[i].starting;
                vga_drawscansegment(&_GrVidPage.gc_baseaddr[(i * _GrVidPage.gc_lineoffset)+StartAt],Dirty[i].starting,i,len);
                Dirty[i].starting = Dirty[i].ending = 0;
            }
            else
            {
                SORT2(Dirty[i].starting, Dirty[i].ending);
                len = (Dirty[i].ending - Dirty[i].starting)<<1;
                StartAt = (Dirty[i].starting<<1);
                vga_drawscansegment(&_GrVidPage.gc_baseaddr[(i * _GrVidPage.gc_lineoffset)+StartAt],Dirty[i].starting,i,len);
                Dirty[i].starting = Dirty[i].ending = 0;
            }
        }
    }
    start = 0; end = 0;
}

char *GrGetVideoScreen()
{
    return(_GrVidPage.gc_baseaddr);
}

void GrFlushScreen()  {
    int i;
  
    if (Accel == TRUE) return;
    for (i = 0; i <= _GrScreenY; i++)
    {
        Dirty[i].flag =TRUE;
        Dirty[i].starting = 0;
        Dirty[i].ending = _GrScreenX;
    }
    start = 0;
    end = _GrScreenY ;
    GrFlush();
}


#define CHECK_CONTEXT(cxt)  ((cxt)->gc_baseaddr != SCREEN->gc_baseaddr)

void GrDirtyCheck( GrContext *c, int x1, int y1, int x2, int y2) {
    int i;
  
    /* Added this section so that we can do an automatic screen update */
    /* whenever we do a operation that is to the screen */

    if ((Accel == TRUE) || (GrGetMemCfg() == LINEAR_MODE))  return;
    if (CHECK_CONTEXT(c)) return;

    x1 += c->gc_xoffset; y1 += c->gc_yoffset;
    x2 += c->gc_xoffset; y2 += c->gc_yoffset;

    SORT2(x1,x2); --x1; ++x2;
    SORT2(y1,y2); --y1; ++y2;
  
    if (x1 < 0) x1 = 0;
    if (x2 < 0) x2 = 0;
    if (y1 < 0) y1 = 0;
    if (y2 < 0) y2 = 0;

    if (x1 >= _GrScreenX)  x1 = _GrScreenX - 1;
    if (x2 >= _GrScreenX)  x2 = _GrScreenX - 1;
    if (y1 >= _GrScreenY)  y1 = _GrScreenY - 1;
    if (y2 >= _GrScreenY)  y2 = _GrScreenY - 1;


    if (start == 0 && end == 0)
    {
        start = y1;
        end = y2;
    }
    else
    {
        if (y1 < start) start = y1;
        if (y2 > end)   end   = y2;
    }
      
    for ( i = y1; i <= y2; i++)
    {
        Dirty[i].flag = TRUE;
        if (Dirty[i].starting == 0 && Dirty[i].ending == 0)
        {
            Dirty[i].starting = x1;
            Dirty[i].ending = x2 + 1;
        }
        else
        {
            if (Dirty[i].starting > x1) Dirty[i].starting = x1;
            if (Dirty[i].ending < x2)   Dirty[i].ending = x2;
        }
    }
}

static int biggest_mode(int interl) {
    int i, w, h, c, m, p, pix;

    w = h = c = p = m = 0;
    for (i=LastMode; i >= 0; --i) {
        if (GM[i].number_of_colors >= 256)  {
            pix = ((int)GM[i].width) * ((int)GM[i].height);
            if (pix > p || GM[i].number_of_colors > c ) {
                w = GM[i].width;
                h = GM[i].height;
                c = GM[i].number_of_colors;
                m = i;
                p = pix;
            }
        }
    }
      return m;
}
  

static int Compare(GR_DRIVER_MODE_ENTRY *fst, GR_DRIVER_MODE_ENTRY *scd)
{
    int p1, p2;

    if (fst->number_of_colors > scd->number_of_colors)
        return  1;
    if (fst->number_of_colors < scd->number_of_colors)
        return -1;
    p1 = ((int) fst->width) * ((int) fst->height);
    p2 = ((int) scd->width) * ((int) scd->height);
    return (int)(p1-p2);
}

static vga_modeinfo *mode_usable(int mode)
{
    vga_modeinfo *info;    
    int size;

    if (!vga_hasmode(mode) || !(info = vga_getmodeinfo(mode)))
        return NULL; 

/*
    switch (mode)
    {
        case G320x240x256:
        case G320x400x256:
        case G360x480x256: 
        return NULL;
    }
*/

    size = info->linewidth * info->height;
    switch (info->colors)
    {
        case   256:
        case 32768:
        case 65536: break;
        case    16: 
            switch (mode)
            {
    	        case G320x200x16:
    	        case G640x200x16:
    	        case G640x350x16:
    	        case G640x480x16:
                    return info;
	        default:
                    return NULL;
            }

        default   : return NULL;
    }
    return info;
}
   
static void set_up_modes(void)
{
      static int   modes_ok = FALSE;
             int   i, j, modes;
    vga_modeinfo   *info;
  struct winsize   winsiz;
             int   memory = 0;


    if (modes_ok == TRUE) return;
    vga_init();
     vga_disabledriverreport();

    /* First count all available modes */
    modes = 0;
    for (i=TEXT+1; i <= GLASTMODE; i++)
            if (mode_usable(i) != NULL) ++modes;

    GM = (GR_DRIVER_MODE_ENTRY *) malloc((modes+1) * sizeof(GR_DRIVER_MODE_ENTRY));
    driver_flags = (unsigned short *) malloc((modes+1) * sizeof(unsigned short));
    mode_flags = (unsigned int *) malloc((modes+1) * sizeof(unsigned int));

    if (modes < 1 || GM == NULL || driver_flags == NULL || mode_flags == NULL)
    {
        fprintf(stderr, "modes Found          = %d\n",modes);
        fprintf(stderr, "GM Pointer           = 0x%x\n",GM);
        fprintf(stderr, "Driver_flags Pointer = 0x%x\n",driver_flags);
        fprintf(stderr, "Mode Flags Pointer   = 0x%x\n",mode_flags);
        fprintf(stderr, "Couldn't set up GRX graphics modes\n");
        exit(1);
    }

    LastMode = modes - 1;

    /* Second get all available modes into the table */
    j = 0;
    for (i=TEXT+1; i <= GLASTMODE; ++i)
    {
        vga_modeinfo *info_tmp;
        if ((info_tmp=mode_usable(i)) != NULL)
        {
            GM[j].width              = info_tmp->width;
            GM[j].height             = info_tmp->height;
            GM[j].number_of_colors   = info_tmp->colors;
             GM[j].mode.grn.BIOS_mode = i;
            mode_flags[j]             = info_tmp->flags;
            ++j;
            info = info_tmp;
        }
    }
    memset(&GM[modes+1], 0, sizeof(GR_DRIVER_MODE_ENTRY));

    /* Third sort table */
    for (i=modes-1; i>0; --i)
    {
        for (j=0; j < i; ++j)
        {
            if ( Compare( &GM[j], &GM[j+1]) > 0)
            {
                swap(GM[j],GM[j+1]);
                swap(mode_flags[j],mode_flags[j+1]);
            }
        }
    }

    /* Last set the correct flags */
    for (i=0; i < modes; ++i)
    {
        switch (GM[i].number_of_colors)
        {
            case 2:
                driver_flags[i] = GRD_1_PLANE;
                break;

            case 16:
                driver_flags[i] = GRD_4_PLANES;
                break;

            case 256:
                switch (GM[i].mode.grn.BIOS_mode)
                {
                    case G320x240x256:
                    case G320x400x256:
                    case G360x480x256: 
                        driver_flags[i] = GRD_8_X_PLANES;
                        break;
                    default :
                        driver_flags[i] = GRD_8_PLANES;
                }
                break;

            case 65535:
            case 32768:
                driver_flags[i] = GRD_16_PLANES;
                break; 
        }
        if ((mode_flags[i] & HAVE_RWPAGE) != 0)
            driver_flags[i] |= GRD_RW_64K;
    }  

      drv.modeset_routine     = 0x0000;  /* not used with LINUX */
      drv.paging_routine      = 0x0000;  /* not used with LINUX */
    

    drv.driver_flags  = GRD_NEW_DRIVER | GRD_NO_RW;
      
    switch (memory)
    {
            case 64:   drv.driver_flags |= GRD_64K;
                       printf("Grx Driver reflects 64Kb of video memory\n");
               break;
            case 128:  drv.driver_flags |= GRD_128K;
                       printf("Grx Driver reflects 128Kb of video memory\n");
                           break;
            case 192:  drv.driver_flags |= GRD_192K;
                       printf("Grx Driver reflects 192Kb of video memory\n");
                           break;
            case 256:  drv.driver_flags |= GRD_256K;
                       printf("Grx Driver reflects 256Kb of video memory\n");
                           break;
            case 512:  drv.driver_flags |= GRD_512K;
                       printf("Grx Driver reflects 512Kb of video memory\n");
                           break;
            case 1024: drv.driver_flags |= GRD_1024K;
                       printf("Grx Driver reflects 1024Kb of video memory\n");
                           break;
            case 1536: drv.driver_flags |= GRD_1536K;
                       printf("Grx Driver reflects 1636Kb of video memory\n");
                           break;
            case 2048: drv.driver_flags |= GRD_2048K;
                       printf("Grx Driver reflects 2048Kb of video memory\n");
                           break;
            case 3072: drv.driver_flags |= GRD_3072K;
                       printf("Grx Driver reflects 3072Kb of video memory\n");
                           break;
            case 4096: drv.driver_flags |= GRD_4096K;
                       printf("Grx Driver reflects 4096Kb of video memory\n");
                           break;
      }
      ioctl(fileno(stdout), TIOCGWINSZ, &winsiz);
      drv.def_th = winsiz.ws_row;
      drv.def_tw = winsiz.ws_col;

      i = -1;
      j = vga_getdefaultmode();
      if ( j >= 0) 
            for (i = 0; i < modes; ++i) 
                  if (GM[i].mode.grn.BIOS_mode == j)
                break;

      if (i < 0 || i >= modes) 
            i = biggest_mode(1);

      drv.def_gw       = GM[i].width;
      drv.def_gh       = GM[i].height;
      drv.def_numcolor = GM[i].number_of_colors;

      drv.driver_init_routine = 0x0000;  /* not used with LINUX */
      drv.text_table          = 0x0000;  /* no way to fit 32 bit into 16 bit storage ! */
      drv.graphics_table      = 0x0000;  /* no way to fit 32 bit into 16 bit storage ! */
      TM[0].width            = drv.def_tw;
      TM[0].height           = drv.def_th;
      TM[0].number_of_colors = 8;
      TM[0].mode.grn.BIOS_mode = 0;
      modes_ok = TRUE;
}

long _GrLowSetMode(int mode, int width, int height, int colors)
{
    int i;

    switch (colors)
    {
    	case 1: colors = 16; break;
    	case 8: colors = 256; break;
    	case 15: colors = 32768; break;
    	case 16: colors = 65536; break;
    }
    
    set_up_modes();
    switch (mode)
    {
            case GR_320_200_graphics:
                width  = 320; height = 200; colors = 256; 
                Accel = FALSE;
                break;
 
          case GR_default_graphics:
            width  = 640; height = 480; colors = 256;
            break; 

        case GR_width_height_graphics:
            colors = 256;
            break;

        case GR_width_height_color_graphics:
            break;

        case GR_biggest_graphics:
        case GR_biggest_noninterlaced_graphics:
            i = biggest_mode(mode != GR_biggest_graphics);
            width = GM[i].width;
            height= GM[i].height;
            colors= GM[i].number_of_colors;
            break;
        case GR_default_text:
        default:
           Accel = FALSE;
           GrSetKeyMode(GR_NORMAL);
           vga_setmode(TEXT);
           vga_screenon();
           Text_Mode = TRUE;
           _GrScreenX = drv.def_tw;
           _GrScreenY = drv.def_th;
          drv.driver_flags &= ~(GRD_TYPE_MASK);
          return ( drv.driver_flags |= GRD_NEW_DRIVER | GR_VGA);
    }
    
    i = LastMode;
    while (i >= 0 && (GM[i].width > width || GM[i].height > height || GM[i].number_of_colors > colors ))
        --i;
    if (i<0) i = 0;
      mode = GM[i].mode.grn.BIOS_mode;
      vga_screenoff();
      vga_setmode(mode);
      AccelSetup(mode);
      Text_Mode = FALSE;
      _GrScreenX = vga_getxdim();
      _GrScreenY = vga_getydim();
      vga_screenon();
      SetupMemory();
      if ((driver_flags[i] & GRD_RW_64K) != 0) RW_page = FALSE;
      return (drv.driver_flags |= driver_flags[i]);
}

void _GrSetVideoPage( int page) {
  if (RW_page) {
    int rdp, wrp;
    
    rdp = page & 0x0FF;
    wrp = (page >> 8) & 0x0FF;
    if (rdp != wrp) { 
    vga_setreadpage(rdp);
    vga_setwritepage(wrp);
    return;
    }    
  }
  vga_setpage(page & 0x0FF);
}

long GrGetDriverModes(GR_DRIVER_MODE_ENTRY **t, GR_DRIVER_MODE_ENTRY **g)
{
  set_up_modes();
  *t = TM;
  *g = GM;
  return (drv.driver_flags);
}

