/**********************************************************************
 *  
 *  NAME:           lnxconio.c
 *  
 *  DESCRIPTION:    simulate Borland text video funcs for GNU C++/Linux
 *
 * 
 *  M O D I F I C A T I O N   H I S T O R Y
 *
 *  when        who                 what
 *  -------------------------------------------------------------------
 *  09/11/93    Hartmut Schirmer    created
 *  
 *********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdarg.h>
#include <ncurses/ncurses.h>
#include <signal.h>
#include <sys/time.h>
#define __WINCONIO
#define __GPPCONIO_C
#include <Grx/gppconio.h>

#ifndef FALSE
#  define FALSE (1==0)
#endif
#ifndef TRUE
#  define TRUE (1==1)
#endif

#define UPDATETIME 200000
#define LIMIT      250
#define NOTIMPL(func) do {								\
			fprintf(stderr,"lnxconio.c: %s() not implemented!\n",(func));	\
			exit(1); 							\
		      } while (FALSE)

#define NO_COL    0
#define FG_COL	  1
#define FG_BG_COL 2
static int colors       = NO_COL;

static int initialized  = FALSE;
static WINDOW *win      = NULL;
static int cur_attr     = 0x07;    /* Gray on Black */
static unsigned changes = 0;
static struct sigaction sa;
static int inside_refresh = 0;
static int timer_on     = FALSE;

static void do_refresh(void)
{
  if (inside_refresh++) return;
  changes = 0;
  wrefresh(win);
  inside_refresh = 0;
}

static __inline void restart_timer(void) {
  timer_on = TRUE;
  sigaction(SIGALRM, &sa, NULL);
}

static __inline void change(void)
{
  ++changes;
  if (timer_on) return; 
  if (changes>LIMIT) 
    do_refresh();
  else
    restart_timer();
}

static void update_tm(int v)
{
  if (changes>0) 
    do_refresh();
  timer_on = FALSE; 
}

static void lnxconio_exit(void)
{
  if (initialized) {
    do_refresh();
    endwin();
  }
}

static int ColTrans(int c)
{
  switch (c) {
    case BLACK	   : return COLOR_BLACK;
    case BLUE	   : return COLOR_BLUE;
    case GREEN	   : return COLOR_GREEN;
    case CYAN	   : return COLOR_CYAN;
    case RED	   : return COLOR_RED;
    case MAGENTA   : return COLOR_MAGENTA;
    case BROWN     : return COLOR_YELLOW;
    case LIGHTGRAY :
    default	   : return COLOR_WHITE;
  }
}

static void lnxconio_init(void)
{
  struct itimerval t;
  int maxx, maxy, fg, bg;

  if (initialized) return;
  initscr();
  start_color();
  if (COLORS >= 8) {
    if (COLOR_PAIRS >= 64) {
      for (fg=0; fg < 8; ++fg) 
        for (bg = 0; bg < 8; ++bg)
          init_pair( (bg<<3) + fg, ColTrans(fg), ColTrans(bg));
      colors = FG_BG_COL;
    } else 
    if (COLOR_PAIRS >= 8) {
      for (fg = 0; fg < 8; ++fg)
        init_pair(fg, ColTrans(fg), COLOR_BLACK);
      colors = FG_COL;
    }
  }
  atexit(lnxconio_exit);
  getmaxyx(stdscr, maxy, maxx);
  win = subwin(stdscr, maxy,maxx,0,0);  /* Full screen window */
  initialized = TRUE;
  idlok(win,TRUE);
  scrollok(win,TRUE);
  cbreak();
  noecho();
  normvideo();
  do_refresh();
  sa.sa_handler = update_tm;
  sa.sa_flags   = SA_RESTART;
  sigemptyset(&sa.sa_mask);
  sigaddset(&sa.sa_mask, SIGALRM);
  sigaction(SIGALRM, &sa, NULL);
  t.it_interval.tv_sec  = UPDATETIME / 1000000;
  t.it_interval.tv_usec = UPDATETIME % 1000000;
  t.it_value.tv_sec     = UPDATETIME / 1000000;
  t.it_value.tv_usec    = UPDATETIME % 1000000;
}

static __inline void init(void) 
{
  if (!initialized)
	lnxconio_init();
}

void clreol(void)
{
  int x, y, mx, my;

  init();
  getyx(win,y,x);
  getmaxyx(win,my,mx);  
  wclrtoeol(win);
  mvwdelch(win,y,mx-1);
  wmove(win,y,x);
  change();
}

void clrscr(void)
{
  init();
  werase(win);
  gotoxy(1,1);
}

void gotoxy(int x, int y)
{
  init();
  wmove(win,y-1,x-1);
  change();
}

int wherex(void)
{
  int x, y;

  init();
  getyx(win,y,x);
  return x+1;  
}

int wherey(void)
{
  int x, y;

  init();
  getyx(win,y,x);
  return y+1;
}

int putch(int c)
{
  init();
  waddch(win,c);
  change();
  return c;
}

void delline(void)
{
  init();
  wdeleteln(win);
  change();
}

void gettextinfo(struct text_info *r)
{
  init(); 
  getbegyx(win, r->wintop, r->winleft);
  getmaxyx(win, r->winbottom, r->winright);
  r->winright  += (r->winleft++);
  r->winbottom += (r->wintop++);
  r->attribute = cur_attr;
  r->normattr  = 0x07;
  r->currmode  = 0;
  getmaxyx(stdscr, r->screenheight, r->screenwidth);
  getyx(win, r->cury, r->curx);
  ++(r->curx);
  ++(r->cury);
}

void insline(void)
{
  init();
  winsertln(win);
  change();
}

static void order(int *small, int *big)
{
  int temp;

  if (*small > *big) {
    temp   = *small;
    *small = *big;
    *big   = temp;
  }
}
 
void window(int left, int top, int right, int bottom)
{
  int sx, sy, mx, my;

  init();

  order(&left, &right);
  order(&top, &bottom);

  getbegyx(win, sy, sx);
  getmaxyx(win, my, mx);
  touchline(stdscr, sy, my);
  delwin(win);
  win = subwin(stdscr, bottom-top+1, right-left+1, top-1, left-1);
  idlok(win,TRUE);
  scrollok(win,TRUE);
  refresh();
  gotoxy(1,1);
}

void _setcursortype(int type) 
{
  switch (type) {
    case _NOCURSOR    : type = 0; break;
    case _SOLIDCURSOR : type = 2; break;
    default           : type = 1; break;
  }
  init();
  curs_set(type);
}

char *cgets(char *str)
{
  init();
  refresh();
  echo();
  wgetstr(win, str);
  noecho();
  return str;
}

int cprintf(const char *fmt, ...)
{
  va_list args;
  char    str[2000];
  int     cnt;

  va_start( args, fmt);                 /* Initialize va_ functions     */
  cnt = vsprintf( str, fmt, args );     /* prints string to buffer      */
  va_end( args );                       /* Close va_ functions          */

  cputs(str);                           /* send string to screen	*/
  return cnt;
}


int cputs(const char *str)
{
  if (*str == '\0') 
    return '\0';

  init();
  while (*str != '\0') {
    if (*str != '\r' || *(str+1) != '\n')
      waddch(win,*str);
    ++str;
  }
  change();
  return *(--str);
}

int cscanf(const char *format, ...)
{
  va_list args;
  int     res;

  init();
  refresh();
  va_start(args, format);
  echo();
  res = vwscanw(win, format, args);
  noecho();
  va_end(args);
 
  return res;
}

int _lnxconio_system(const char *cmd)
{
  int res;

  if (initialized)
    do_refresh();
  res = system(cmd);
  if (initialized) 
    doupdate();
  return res;
}

int kbhit(void)
{
  int ch;

  init();
  do_refresh();
  nodelay(win, TRUE);
  ch = wgetch(win);
  if (ch == ERR) return FALSE;
  ungetch(ch);
  return TRUE;  
}

int _lnxconio_ungetch(int ch)
{
  ungetch(ch);
  return ch;
}

int _lnxconio_getch(void)
{
  int ch;

  init();
  do_refresh();
  nodelay(win,FALSE);
  ch = wgetch(win);
  nodelay(win,TRUE);
  return ch;
}

int getche(void)
{
  int ch;

  ch = _lnxconio_getch();
  putch(ch);
  do_refresh();
  return ch;
}

int _lnxconio_fprintf(FILE *f, const char *fmt, ...)
{
  va_list args;
  char    str[2000];
  int     cnt;

  va_start( args, fmt);                 /* Initialize va_ functions     */
  if (f == stdout) {
    cnt = vsprintf( str, fmt, args );   /* prints string to buffer      */
    cputs(str);                         /* send string to screen	*/
  } else 
    cnt = vfprintf(f,fmt,args);		/* normal operation		*/
  va_end( args );                       /* Close va_ functions          */

  return cnt;
}

int _lnxconio_fputc(int c, FILE *stream)
{
  if (stream == stdout) 
    return putch(c);
  return putc(c,stream);
}

int _lnxconio_fputs(const char *strg, FILE *stream)
{
  if (stream == stdout)
    return cputs(strg);
  return fputs(strg,stream);
}

/* ----------------------------- special routines affecting the whole screen */

void _lnxconio_clearall(void)
{
  init();
  changes = 0;
  werase(win);
  wclear(stdscr);
  wnoutrefresh(win);
  wnoutrefresh(stdscr);
  doupdate();
  gotoxy(1,1);
}

/* --------------------------------------------------------- text attributes */

static chtype attr2curses(int attr)
{
  chtype at;

  at = A_NORMAL;
  if ( (attr & BLINK) != 0) at |= A_BLINK;
  if ( (attr & 0x08)  != 0) at |= A_BOLD;
  switch (colors) {
    case NO_COL    : break;
    case FG_COL    : at |= COLOR_PAIR(attr&0x07);
		     break;
    case FG_BG_COL : at |= COLOR_PAIR( ((attr&0x07) | ((attr&0x70)>>1)));
		     break;
  }
  return at;
}

static int curses2attr( chtype c)
{
  int attr;
  int colp;

  colp = PAIR_NUMBER(c);
  switch (colors) {
    case FG_COL    : attr = (colp&0x07);
		     break;
    case FG_BG_COL : attr = (colp&0x07) | ((colp&0x38)<<1); 
		     break;
    default        : attr = 0;
		     break;
  }
  if ( (c&A_BLINK) != 0 ) attr |= BLINK;  /* Blink  */
  if ( (c&A_BOLD)  != 0 ) attr |= 0x08;   /* Highlightning */
 
  return attr;
} 

void textattr(int attr)
{
  init(); 
  wattrset(win, attr2curses(attr));
  cur_attr = attr;
}

void highvideo(void)
{
  init();
  textattr(cur_attr|0x08);
}

void lowvideo(void)
{
  init();
  textattr(cur_attr&0xF7);
}

void normvideo(void)
{
  textattr(0x07);
}

void textbackground(int color)
{
  init(); 
  textattr( (cur_attr&0x8F) | ((color & 0x07) << 4) );
}

void textcolor(int color)
{
  init();
  textattr( (cur_attr&0xF0) | (color&0x0F) );
}

/* ------------------------------------------------------------- put/gettext() -- */

int gettext(int left, int top, int right, int bottom, void *destin)
{
  unsigned short *sp;
  int x, y;
  chtype ch;

  init(); 
  getmaxyx(stdscr,y,x);
  if ( top<=0 || left<=0 || top>bottom || left>right || bottom>y || right>x)
    return 0;
  sp = (unsigned short *) destin;
  for (y = top-1; y < bottom; ++y)
    for (x = left-1; x < right; ++x) {
      ch = mvwinch(stdscr,y,x);
      *(sp++) = (ch & A_CHARTEXT) | (curses2attr(ch)<<8);
    }
  return 1;
}

int movetext(int left, int top, int right, int bottom, int destleft, int desttop)
{
  int mx, my, rows, cols, x, y, dx, dy;
  chtype ch;

  if (left == destleft && top == desttop)
    return 1;
  init(); 
  getmaxyx(stdscr,my,mx);
  rows = bottom-top+1;
  cols = right-left+1;
  if (   top<=0            || left<=0 
      || top>bottom        || left>right 
      || bottom>my         || right>mx
      || desttop+rows>my+1 || destleft+cols>mx+1)
    return 0;
  if (desttop < top || (desttop == top && destleft < left)) {
    for (y=top-1, dy = desttop-1; y<bottom; ++y, ++dy) 
      for (x=left-1, dx = destleft-1; x<right; ++x, ++dx) {
        ch = mvwinch(stdscr, y, x);
        mvwaddch(stdscr, dy, dx, ch);
      }
  } else {
    for (y=top+rows-2, dy = desttop+rows-2; y>=top-1; --y, --dy) 
      for (x=left+cols-2, dx = destleft+cols-2; x>=left-1; --x, --dx) {
        ch = mvwinch(stdscr, y, x);
        mvwaddch(stdscr, dy, dx, ch);
      }
  }
  wrefresh(stdscr);
  changes = 0;
  return 1;
}

int puttext(int left, int top, int right, int bottom, void *source)
{
  unsigned short *sp;
  int x, y;
  int ch;
  chtype wch;

  init(); 
  getmaxyx(stdscr,y,x);
  if ( top < 0 || left < 0 || top > bottom || left > right || bottom > y || right > x)
    return 0;
  sp = (unsigned short *) source;
  for (y = top-1; y < bottom; ++y)
    for (x = left-1; x < right; ++x) {
      ch = *(sp++);
      wch = (ch&0xff) | attr2curses(ch>>8);      
      mvwaddch(stdscr,y,x,wch);
    }

  changes = 0;
  touchwin(win);
  touchwin(stdscr);
  wnoutrefresh(win);
  wnoutrefresh(stdscr);
  doupdate();
  return 1;
}


/* --------------------------------------------------------------- delay() */

void delay(unsigned ms)
{
  if (ms > 0)
    usleep(ms*1000);
}

