/*
 * scaled image source
 *
 * provides on-the-fly scaling, shamelessly leached from gdk
 *
 * $Id: scaled.c,v 1.9 2000/11/03 16:04:32 opencare Exp $
 *
 * Copyright (C) 2000 OpenCare (www.ocare.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  
 */

#include <math.h>
#include <glib.h>
#include "gfxtrans.h"

#define SUBSAMPLE_BITS 4
#define SUBSAMPLE (1 << SUBSAMPLE_BITS)
#define SUBSAMPLE_MASK ((1 << SUBSAMPLE_BITS)-1)
#define SCALE_SHIFT 16

static int
get_check_shift (int check_size)
{
  int check_shift = 0;
  g_return_val_if_fail (check_size >= 0, 4);

  while (!(check_size & 1)) {
    check_shift++;
    check_size >>= 1;
  }

  return check_shift;
}

static void
scale_pixel (guchar *dest, int dest_x, int dest_channels, int dest_has_alpha,
	     int src_has_alpha, int check_size, guint32 color1, guint32 color2,
	     int r, int g, int b, int a)
{
  if (src_has_alpha) {
    if (a) {
      dest[0] = r / a;
      dest[1] = g / a;
      dest[2] = b / a;
      dest[3] = a >> 16;
    } else {
      dest[0] = 0;
      dest[1] = 0;
      dest[2] = 0;
      dest[3] = 0;
    }
  } else {
    dest[0] = (r + 0xffffff) >> 24;
    dest[1] = (g + 0xffffff) >> 24;
    dest[2] = (b + 0xffffff) >> 24;
      
    if (dest_has_alpha)
      dest[3] = 0xff;
  }
}

static double
bilinear_quadrant (double bx0, double bx1, double by0, double by1) {
  double ax0, ax1, ay0, ay1;
  double x0, x1, y0, y1;

  ax0 = 0.;
  ax1 = 1.;
  ay0 = 0.;
  ay1 = 1.;

  if (ax0 < bx0) {
    if (ax1 > bx0) {
      x0 = bx0;
      x1 = MIN (ax1, bx1);
    } else {
      return 0;
    }
  } else {
    if (bx1 > ax0) {
      x0 = ax0;
      x1 = MIN (ax1, bx1);
    } else {
      return 0;
    }
  }

  if (ay0 < by0) {
    if (ay1 > by0) {
      y0 = by0;
      y1 = MIN (ay1, by1);
    } else {
      return 0;
    }
  } else {
    if (by1 > ay0) {
      y0 = ay0;
      y1 = MIN (ay1, by1);
    } else {
      return 0;
    }
  }

  return 0.25 * (x1*x1 - x0*x0) * (y1*y1 - y0*y0);
}

static void
process_pixel (int *weights, int n_x, int n_y,
	       guchar *dest, int dest_x, int dest_channels, int dest_has_alpha,
	       guchar **src, int src_channels, gboolean src_has_alpha,
	       int x_start, int src_width,
	       int check_size, guint32 color1, guint32 color2)
{
  unsigned int r = 0, g = 0, b = 0, a = 0;
  int i, j;
  
  for (i=0; i<n_y; i++) {
    int *line_weights  = weights + n_x * i;

    for (j=0; j<n_x; j++) {
      unsigned int ta;
      guchar *q;

      if (x_start + j < 0)
	q = src[i];
      else if (x_start + j < src_width)
	q = src[i] + (x_start + j) * src_channels;
      else
	q = src[i] + (src_width - 1) * src_channels;

      if (src_has_alpha)
	ta = q[3] * line_weights[j];
      else
	ta = 0xff * line_weights[j];

      r += ta * q[0];
      g += ta * q[1];
      b += ta * q[2];
      a += ta;
    }
  }

  scale_pixel (dest, dest_x, dest_channels, dest_has_alpha, src_has_alpha, check_size, color1, color2, r, g, b, a);
}

static guchar *
scale_line (int *weights, int n_x, int n_y,
	    guchar *dest, int dest_x, guchar *dest_end, int dest_channels, int dest_has_alpha,
	    guchar **src, int src_channels, gboolean src_has_alpha,
	    int x_init, int x_step, int src_width,
	    int check_size, guint32 color1, guint32 color2)
{
  int x = x_init;
  int i, j;

  while (dest < dest_end) {
    int x_scaled = x >> SCALE_SHIFT;
    int *pixel_weights;

    pixel_weights = weights + ((x >> (SCALE_SHIFT - SUBSAMPLE_BITS)) & SUBSAMPLE_MASK) * n_x * n_y;

    if (src_has_alpha) {
      unsigned int r = 0, g = 0, b = 0, a = 0;
      for (i=0; i<n_y; i++) {
	guchar *q = src[i] + x_scaled * src_channels;
	int *line_weights  = pixel_weights + n_x * i;
	      
	for (j=0; j<n_x; j++) {
	  unsigned int ta;
		  
	  ta = q[3] * line_weights[j];
	  r += ta * q[0];
	  g += ta * q[1];
	  b += ta * q[2];
	  a += ta;
		  
	  q += src_channels;
	}
      }

      if (a) {
	dest[0] = r / a;
	dest[1] = g / a;
	dest[2] = b / a;
	dest[3] = a >> 16;
      } else {
	dest[0] = 0;
	dest[1] = 0;
	dest[2] = 0;
	dest[3] = 0;
      }
    } else {
      unsigned int r = 0, g = 0, b = 0;
      for (i=0; i<n_y; i++) {
	guchar *q = src[i] + x_scaled * src_channels;
	int *line_weights  = pixel_weights + n_x * i;
	      
	for (j=0; j<n_x; j++) {
	  unsigned int ta = line_weights[j];
		  
	  r += ta * q[0];
	  g += ta * q[1];
	  b += ta * q[2];

	  q += src_channels;
	}
      }

      dest[0] = (r + 0xffff) >> 16;
      dest[1] = (g + 0xffff) >> 16;
      dest[2] = (b + 0xffff) >> 16;
	  
      if (dest_has_alpha)
	dest[3] = 0xff;
    }

    dest += dest_channels;
      
    x += x_step;
  }

  return dest;
}

static void
pixops_process (imgsource_t img_source, gfxtrans_display_t *fb) {
  guchar         *dest_buf;
  int             render_x0;
  int             render_y0;
  int             render_x1;
  int             render_y1;
  int             dest_rowstride;
  int             dest_channels;
  const gboolean        dest_has_alpha = 0;
  const guchar   *src_buf;
  int             src_width;
  int             src_height;
  int             src_rowstride;
  int             src_channels;
  const gboolean        src_has_alpha = 0;
  double          scale_x;
  double          scale_y;
  const int             check_x = 0;
  const int             check_y = 0;
  const int             check_size = 0;
  const guint32         color1 = 0;
  const guint32         color2 = 0;

  int i, j;
  int x, y;
  guchar **line_bufs = g_new (guchar *, GFXTRANS_SCALED_T(img_source).n_y);

  int x_step;
  int y_step;

  int dest_x;
  int scaled_x_offset;

  int run_end_index;
  int check_shift = 1;

  /* select a display to render */
  if (fb) {
    dest_buf = gfxtrans_get_write_fb(fb);
  } else {
    dest_buf = gfxtrans_get_write_fb(img_source->display); 
  }

  render_x0 = 0;
  render_y0 = 0;
  render_x1 = img_source->width;
  render_y1 = img_source->height;
  dest_channels = 4;
  dest_rowstride = (render_x1 - render_x0) * dest_channels;

  src_width = GFXTRANS_SCALED_T(img_source).img_source->width;
  src_height = GFXTRANS_SCALED_T(img_source).img_source->height;
  src_buf = gfxtrans_get_read_fb(GFXTRANS_SCALED_T(img_source).img_source->display);
  src_channels = 4;
  src_rowstride = src_width * src_channels;

  scale_x = (double)render_x1 / src_width;
  scale_y = (double)render_y1 / src_height;

  x_step = (1 << SCALE_SHIFT) / scale_x;
  y_step = (1 << SCALE_SHIFT) / scale_y;

  scaled_x_offset = floor (GFXTRANS_SCALED_T(img_source).x_offset * (1 << SCALE_SHIFT));

  check_shift = check_size ? get_check_shift (check_size) : 0;


  run_end_index = (((src_width - GFXTRANS_SCALED_T(img_source).n_x + 1) << SCALE_SHIFT) - scaled_x_offset - 1) / x_step + 1 - render_x0;

  y = render_y0 * y_step + floor (GFXTRANS_SCALED_T(img_source).y_offset * (1 << SCALE_SHIFT));

  for (i = 0; i < (render_y1 - render_y0); i++) {
    int y_start = y >> SCALE_SHIFT;
    int x_start;
    int *run_weights = (GFXTRANS_SCALED_T(img_source).weights
			+ ((y >> (SCALE_SHIFT - SUBSAMPLE_BITS)) & SUBSAMPLE_MASK)
			* GFXTRANS_SCALED_T(img_source).n_x * GFXTRANS_SCALED_T(img_source).n_y * SUBSAMPLE);
    guchar *new_outbuf;
    guint32 tcolor1, tcolor2;
      
    guchar *outbuf = dest_buf + dest_rowstride * i;
    guchar *outbuf_end = outbuf + dest_channels * (render_x1 - render_x0);

    if (((i + check_y) >> check_shift) & 1) {
      tcolor1 = color2;
      tcolor2 = color1;
    } else {
      tcolor1 = color1;
      tcolor2 = color2;
    }

    for (j=0; j<GFXTRANS_SCALED_T(img_source).n_y; j++) {
      if (y_start <  0)
	line_bufs[j] = (guchar *)src_buf;
      else if (y_start < src_height)
	line_bufs[j] = (guchar *)src_buf + src_rowstride * y_start;
      else
	line_bufs[j] = (guchar *)src_buf + src_rowstride * (src_height - 1);

      y_start++;
    }

    dest_x = check_x;
    x = render_x0 * x_step + scaled_x_offset;
    x_start = x >> SCALE_SHIFT;

    while (x_start < 0 && outbuf < outbuf_end) {
      process_pixel (run_weights + ((x >> (SCALE_SHIFT - SUBSAMPLE_BITS)) & SUBSAMPLE_MASK) 
		     * (GFXTRANS_SCALED_T(img_source).n_x * GFXTRANS_SCALED_T(img_source).n_y),
		     GFXTRANS_SCALED_T(img_source).n_x, GFXTRANS_SCALED_T(img_source).n_y,
		     outbuf, dest_x, dest_channels, dest_has_alpha,
		     line_bufs, src_channels, src_has_alpha,
		     x >> SCALE_SHIFT, src_width,
		     check_size, tcolor1, tcolor2);
	  
      x += x_step;
      x_start = x >> SCALE_SHIFT;
      dest_x++;
      outbuf += dest_channels;
    }

    new_outbuf = scale_line(run_weights, GFXTRANS_SCALED_T(img_source).n_x, GFXTRANS_SCALED_T(img_source).n_y,
			    outbuf, dest_x,
			    MIN (outbuf_end, dest_buf + dest_rowstride * i + run_end_index * dest_channels),
			    dest_channels, dest_has_alpha,
			    line_bufs, src_channels, src_has_alpha,
			    x, x_step, src_width, check_size, tcolor1, tcolor2);

    dest_x += (new_outbuf - outbuf) / dest_channels;

    x = dest_x * x_step + scaled_x_offset;
    outbuf = new_outbuf;

    while (outbuf < outbuf_end) {
      process_pixel (run_weights + ((x >> (SCALE_SHIFT - SUBSAMPLE_BITS)) & SUBSAMPLE_MASK)
		     * (GFXTRANS_SCALED_T(img_source).n_x * GFXTRANS_SCALED_T(img_source).n_y),
		     GFXTRANS_SCALED_T(img_source).n_x, GFXTRANS_SCALED_T(img_source).n_y,
		     outbuf, dest_x, dest_channels, dest_has_alpha,
		     line_bufs, src_channels, src_has_alpha,
		     x >> SCALE_SHIFT, src_width,
		     check_size, tcolor1, tcolor2);
	  
      x += x_step;
      dest_x++;
      outbuf += dest_channels;
    }

    y += y_step;
  }

  g_free (line_bufs);
}


static void
bilinear_make_weights (imgsource_t img_source, double x_scale, double y_scale, double overall_alpha)
{
  int i_offset, j_offset;

  int n_x = ceil(1/x_scale + 2.0);
  int n_y = ceil(1/y_scale + 2.0);

  GFXTRANS_SCALED_T(img_source).x_offset = -1.0;
  GFXTRANS_SCALED_T(img_source).y_offset = -1.0;
  GFXTRANS_SCALED_T(img_source).n_x = n_x;
  GFXTRANS_SCALED_T(img_source).n_y = n_y;
  
  GFXTRANS_SCALED_T(img_source).weights = g_new (int, SUBSAMPLE * SUBSAMPLE * n_x * n_y);

  for (i_offset=0; i_offset<SUBSAMPLE; i_offset++) {
    for (j_offset=0; j_offset<SUBSAMPLE; j_offset++) {
      int *pixel_weights = GFXTRANS_SCALED_T(img_source).weights + ((i_offset*SUBSAMPLE) + j_offset) * n_x * n_y;
      double x = (double)j_offset / 16;
      double y = (double)i_offset / 16;
      int i,j;
	  
      for (i = 0; i < n_y; i++) {
	for (j = 0; j < n_x; j++) {
	  double w;

	  w = bilinear_quadrant  (0.5 + j - (x + 1 / x_scale), 0.5 + j - x, 0.5 + i - (y + 1 / y_scale), 0.5 + i - y);
	  w += bilinear_quadrant (1.5 + x - j, 1.5 + (x + 1 / x_scale) - j, 0.5 + i - (y + 1 / y_scale), 0.5 + i - y);
	  w += bilinear_quadrant (0.5 + j - (x + 1 / x_scale), 0.5 + j - x, 1.5 + y - i, 1.5 + (y + 1 / y_scale) - i);
	  w += bilinear_quadrant (1.5 + x - j, 1.5 + (x + 1 / x_scale) - j, 1.5 + y - i, 1.5 + (y + 1 / y_scale) - i);
	  
	  *(pixel_weights + n_x * i + j) = 65536 * w * x_scale * y_scale * overall_alpha;
	}
      }
    }
  }
}

/*
 * create a scaled source
 *
 * arg1 : reference to the image to scale
 * arg2 : wanted width
 * arg3 : wanted height
 * ...  : ignored
 */
imgsource_t 
gfxtrans_scaled_create(va_list ap) {
  imgsource_t scaled_source, img_source;
  int dest_width, dest_height;

  /* consume our arguments, trailing will be ignored */
  img_source = va_arg(ap, imgsource_t);
  if (img_source) {
    dest_width = va_arg(ap, int);
    if (dest_width) {
      dest_height = va_arg(ap, int);
      if (dest_height) {

	gfx_log(LOG_DEBUG, "gfxtrans_scaled_create(%p) : %dx%d",
		img_source, dest_width, dest_height);

	/* create scaled source */
	scaled_source = (imgsource_t) malloc ( sizeof(struct _imgsource_t) );
	if (scaled_source) {

	  /* initialize member functions */
	  scaled_source->create_f = gfxtrans_scaled_create;
	  scaled_source->display_f = gfxtrans_scaled_display;
	  scaled_source->destroy_f = gfxtrans_scaled_destroy;

	  /* initialize member datas */
	  GFXTRANS_SCALED_T(scaled_source).img_source = img_source;
	  scaled_source->width = dest_width;
	  scaled_source->height = dest_height;
	  bilinear_make_weights (scaled_source,
				 ((double)img_source->width) / dest_width,
				 ((double)img_source->height) / dest_height,
				 1.0);
	  scaled_source->display =  gfxtrans_display_allocate(GFXTRANS_DISPLAY_MEMORY,
							      dest_width,
							      dest_height);
	  if (scaled_source->display) {
	    /* exit successfully */
	    return scaled_source;

	  } else {
	    gfx_log(LOG_CRIT,"can't create display");
	  }
	} else {
	  gfx_log(LOG_CRIT,"can't create scaled source");
	}
      } else {
	gfx_log(LOG_CRIT,"invalid image height : %d", dest_height);
      }
    } else {
      gfx_log(LOG_CRIT,"invalid image width : %d", dest_width);
    }
  } else {
    gfx_log(LOG_CRIT,"NULL image source given");
  }

  return NULL;
}

gfxtrans_display_t *
gfxtrans_scaled_display(imgsource_t img_source,
			gfxtrans_display_t *display) {
  
  gfx_log(LOG_DEBUG, "gfxtrans_scaled_display(%p, %p)",
	  img_source, display);

  if (display) {
    gfxtrans_lock_read_display(display);
/*      img_source->display = display; */
  }

  pixops_process(img_source, display);
  
  if (display) {
    gfxtrans_unlock_display(display);
    gfxtrans_flush_display(display);
  }
  return NULL;
}

void
gfxtrans_scaled_destroy(imgsource_t imgsource) {
  gfx_log(LOG_DEBUG, "gfxtrans_scaled_destroy(%p)",
	  imgsource);

  if (imgsource) {
    if (imgsource->display) {
      gfxtrans_display_destroy(imgsource->display);
      g_free (GFXTRANS_SCALED_T(imgsource).weights);
    }
    free(imgsource);
  }
}
