/*
 * jpeg image source
 *
 * provides jpeg decompression
 *
 * $Id: jpeg.c,v 1.10 2000/11/03 16:04:32 opencare Exp $
 *
 * Copyright (C) 2000 OpenCare (www.ocare.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  
 */

#include "gfxtrans.h"

#include <jpeglib.h>
#include <string.h>
#include <errno.h>

/*
 * open and decompress a jpeg image file to an image source
 *
 * img_source: source to fill
 * filename: obvious
 *
 */
static int
gfxtrans_jpeg_decompress(imgsource_t img_source, char *filename) {
  FILE * infile;
  int jpeg_error = 0;

  infile = fopen(filename, "rb");
  if (infile) {

    struct jpeg_decompress_struct cinfo;
    struct jpeg_error_mgr jerr;

    cinfo.err = jpeg_std_error(&jerr);
    jpeg_create_decompress(&cinfo);

    jpeg_stdio_src(&cinfo, infile);
    jpeg_read_header(&cinfo, TRUE);

    gfx_log(LOG_DEBUG, "gfxtrans_jpeg_decompress: jpeg is %dx%dx%dbpp\n", 
	    cinfo.image_width, cinfo.image_height,
	    cinfo.num_components * 8);

    if (cinfo.image_width && cinfo.image_height && cinfo.num_components) {

      /* init jpeg_source members */
      img_source->width = cinfo.image_width;
      img_source->height = cinfo.image_height;
      img_source->display = gfxtrans_display_allocate(GFXTRANS_DISPLAY_MEMORY,
						      cinfo.image_width,
						      cinfo.image_height);

      if (img_source->display) {
	JSAMPLE *raw_line;

	raw_line = (JSAMPLE*)malloc(cinfo.image_width * cinfo.num_components);

	if (raw_line) {
	  unsigned char *fbptr;
	  int row_stride;

	  /* assure rbg output */
	  cinfo.out_color_space = JCS_RGB;

	  jpeg_start_decompress(&cinfo);

	  /* init decompression pointer */
	  fbptr = gfxtrans_get_read_fb(img_source->display);

	  row_stride = cinfo.output_width * cinfo.num_components;

	  /* decompress scanlines */
	  while(cinfo.output_scanline < cinfo.output_height)
	    {
	      int i;
	      unsigned char *rbuf;

	      jpeg_read_scanlines(&cinfo, &raw_line, 1);

	      /* convert to 32 bpp */
	      rbuf = (unsigned char*)raw_line;

	      /* BGRa -> RGBa   */
	      for (i = 0; i < cinfo.output_width; i++) {
		*(fbptr++) = rbuf[3*i + 2];
		*(fbptr++) = rbuf[3*i + 1];
		*(fbptr++) = rbuf[3*i];
		*(fbptr++) = 0;	/* skip alpha channel */
	      }
	    }

	  jpeg_finish_decompress(&cinfo);	
	  jpeg_destroy_decompress(&cinfo);	

	  free(raw_line);
	  fclose (infile);

	  /* exit successfully */
	  return 0;

	} else {
	  gfx_log(LOG_CRIT, "can't allocate line buffer\n");
	  jpeg_error = ENOMEM;
	}
	gfxtrans_display_destroy(img_source->display);
      } else {
	gfx_log(LOG_CRIT, "can't create raw picture\n");
	jpeg_error = ENOMEM;
      }
    } else {
      gfx_log(LOG_CRIT, "invalid size: %dx%dx%d\n", 
	      cinfo.image_width, cinfo.image_height,
	      cinfo.num_components * 8);
      jpeg_error = EINVAL;
    }
    jpeg_destroy_decompress(&cinfo);
    fclose (infile);
  } else {
    gfx_log(LOG_CRIT, "can't open %s : %s\n", filename, strerror(errno)); 
    jpeg_error = errno;
  }
  return jpeg_error;
}

/*
 * create a jpeg source
 *
 * arg1: char* file name
 * ... : ignored
 */
imgsource_t 
gfxtrans_jpeg_create(va_list ap) {
  imgsource_t jpeg_source;
  char *filename;

  /* consume our arguments, trailing will be ignored */
  filename = va_arg(ap, char*);

  if (filename) {

    gfx_log(LOG_DEBUG, "gfxtrans_jpeg_create(\"%s\")", filename);

    /* create jpeg source */
    jpeg_source = (imgsource_t) malloc ( sizeof(struct _imgsource_t) );
    if (jpeg_source) {
      if (!gfxtrans_jpeg_decompress(jpeg_source, filename)) {

	/* initialize member functions */
	jpeg_source->create_f = gfxtrans_jpeg_create;
	jpeg_source->display_f = gfxtrans_jpeg_display;
	jpeg_source->destroy_f = gfxtrans_jpeg_destroy;

	/* initialize member datas */
	GFXTRANS_JPEG_T(jpeg_source).filename = filename;

	/* return successfully */
	return jpeg_source;

      } else {
	gfx_log(LOG_CRIT,"error reading jpeg");
      }

      /*      gfxtrans_jpeg_dispose(jpeg_source); */

    } else {
      gfx_log(LOG_CRIT,"can't malloc jpeg");
    }
    free(jpeg_source);
  } else {
    gfx_log(LOG_CRIT,"NULL filename given");
  }


  /* reference last frames of previous image source */
  /* reference(prev); */

  /* prepare destination image */
/*    dest->create(); */
  return NULL;
}


gfxtrans_display_t *
gfxtrans_jpeg_display(imgsource_t img_source, gfxtrans_display_t *display) {
  return NULL;
}

/*
 * destroy a jpeg source
 */
void
gfxtrans_jpeg_destroy(imgsource_t img_source) {
  if (img_source) {
    if (img_source->display) {
      gfxtrans_display_destroy(img_source->display);
    }
    free(img_source);
  }
}
