/*
 * display abstraction
 *
 * provides a readable/writable 32-bit framebuffer
 *
 * $Id: display.c,v 1.5 2000/11/03 16:02:52 opencare Exp $
 *
 * Copyright (C) 2000 OpenCare (www.ocare.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  
 */

#include <pthread.h>
#include <string.h>
#include "display.h"
#include "gfxlog.h"

static const char * ident="libgfxtrans";

/* this serialize access to pools */
static pthread_mutex_t onboard_visual_pool_lock = PTHREAD_MUTEX_INITIALIZER;

/* mallocated array of available displays in video ram */
static gfxtrans_display_t *onboard_visual_pool;

/* bit-map for free slots in onboard_visual_pool, 1 means free */
static unsigned long onboard_visual_pool_map;

/*
 * create a display
 *
 * device: ggi device ("display-memory", "display-fbdev" ...)
 * width: requested width
 * heigth: requested height
 *
 * return new display_t on success, NULL on error
 */
static gfxtrans_display_t *
gfxtrans_display_create(char *device, int width, int height) {
  gfxtrans_display_t *disp;

  gfx_log(LOG_DEBUG, "gfxtrans_display_create: dev=\"%s\", %dx%d",
      device, width, height);

  disp = (gfxtrans_display_t*) malloc( sizeof(gfxtrans_display_t) );
  if (disp) {

    /* open ggi visual */
    disp->visual = ggiOpen(device, NULL);
    if (disp->visual) {

      /* try to set mode */
      disp->mode.frames = 1;
      disp->mode.visible.x = width;
      disp->mode.visible.y = height;
      disp->mode.virt.x = width;
      disp->mode.virt.y = height;
      disp->mode.graphtype = GT_32BIT;
      disp->mode.dpp.x = disp->mode.dpp.y = 1;

      if (ggiSetMode(disp->visual, &disp->mode) == 0) {

	disp->fb = ggiDBGetBuffer (disp->visual, 0);

	/* exit successfully */
	return disp;

      } else {
	gfx_log(LOG_CRIT, "can't set requested mode");
      }
      ggiClose(disp->visual);
    } else {
      gfx_log(LOG_CRIT, "can't open ggi display");
    }
    free(disp);
  } else {
    gfx_log(LOG_CRIT, "can't allocate display_t");
  }
  return NULL;
}

/*
 * initialize the display engine
 * width, height: resolution wanted for screen
 *
 * return 0 on success, otherwise an error code
 */
int
gfxtrans_display_init(int width, int height) {
  int result;
  unsigned long numbufs = 8;
  ggi_visual_t visual;
  ggi_mode mode;


  openlog(ident, 0, LOG_USER );
  result = ggiInit();

  if (!result) {
    visual = ggiOpen(NULL); /* open visual for default target */    

    if (visual) {

      mode.frames = numbufs;	/* try to get 8 frames */
      mode.visible.x = width;
      mode.visible.y = height;
      mode.virt.x = width;
      mode.virt.y = height;
      mode.graphtype = GT_32BIT;
      mode.dpp.x = mode.dpp.y = 1;

      /* open available display  */
      result = ggiSetMode(visual, &mode);
      if ( !result ) {
	int i;

	/* allocate available buffers */
	numbufs = ggiDBGetNumBuffers(visual);

	gfx_log(LOG_DEBUG, "gfxtrans_display_init: init %d on-board displays",
		numbufs);

	if (numbufs > 0) {
	  onboard_visual_pool = malloc ( numbufs * sizeof(gfxtrans_display_t) );

	  if (onboard_visual_pool) {

	    /* register displays in free pool */
	    for(i = 0; i < numbufs; i++) {

	      onboard_visual_pool[i].location = GFXTRANS_DISPLAY_ONBOARD;
	      onboard_visual_pool[i].id = i;
	      onboard_visual_pool[i].visual = visual;
	      onboard_visual_pool[i].fb = ggiDBGetBuffer(visual, i);

	      gfx_log(LOG_DEBUG, "gfxtrans_display_init creates fb at %p", onboard_visual_pool[i].fb->write);
	

	      /* save mode in display */
	      onboard_visual_pool[i].mode = mode; /* <=> memcpy */
	    }

	    /* compute free slots bitmap */
	    onboard_visual_pool_map = (1UL << numbufs) - 1;

	    /* switches to asynchronous mode */
	    ggiAddFlags(visual, GGIFLAG_ASYNC);

	    return 0;
	  } else {
	    gfx_log(LOG_CRIT, "can't allocate visual pool");
	  }
	} else {
	  gfx_log(LOG_CRIT, "can't open direct buffers");
	}
      } else {
	gfx_log(LOG_CRIT, "can't set mode : %d", result);
      }
      ggiClose(visual);
    } else {
      gfx_log(LOG_CRIT, "can't open ggi display");
    }
  }

  return result;
}

/*
 * done with display fucntion
 */
void
gfxtrans_display_exit() {

  pthread_mutex_lock(&onboard_visual_pool_lock);

  /* free everybody */
  closelog();
  ggiClose(onboard_visual_pool[0].visual);
  ggiExit();
  free(onboard_visual_pool);

  pthread_mutex_destroy(&onboard_visual_pool_lock);
}

/*
 * allocate a display
 * 
 * where: what kind of display (GFXTRANS_DISPLAY_ONBOARD, GFXTRANS_DISPLAY_MEMORY)
 * width, height: dimensions of the display
 *
 * return new display_t on success, NULL otherwise
 */
gfxtrans_display_t *
gfxtrans_display_allocate(int where, int width, int height) {
  int slot;
  gfxtrans_display_t *disp;

  switch (where) {
  case GFXTRANS_DISPLAY_ONBOARD:

    pthread_mutex_lock(&onboard_visual_pool_lock);

    /* get a free slot - find_last_set should be faster */
    slot = ffs(onboard_visual_pool_map);
    if (slot--) {
      
      /* reference slot */
      onboard_visual_pool_map &= ~(1UL << slot);
      
      gfx_log(LOG_DEBUG, "gfxtrans_display_allocate allocates vram-display %d [%p] for %dx%d display, map: %08x",
	      slot, &onboard_visual_pool[slot], width, height, onboard_visual_pool_map);

      return &onboard_visual_pool[slot];
    }
    /* no more free buffers, go on DISPLAY_MEMORY */
    
    pthread_mutex_unlock(&onboard_visual_pool_lock);


  case GFXTRANS_DISPLAY_MEMORY:

    gfx_log(LOG_DEBUG, "gfxtrans_display_allocate allocates ram-display for %dx%d display",
	    width, height);

    disp = gfxtrans_display_create("display-memory", width, height);
    disp->location = GFXTRANS_DISPLAY_MEMORY;
    disp->id = 0;
    return disp;
  }
  return NULL; 

}

/*
 * destroy a display (not really :)
 *
 * disp: previously allocated display to destroy
 */
void
gfxtrans_display_destroy(gfxtrans_display_t *disp) {
  int slot;

  if (disp) {
    switch (disp->location) {

    case GFXTRANS_DISPLAY_ONBOARD:
      pthread_mutex_lock(&onboard_visual_pool_lock);

      slot = (disp - &onboard_visual_pool[0]);

      /* unreference slot */
      onboard_visual_pool_map |= (1UL << slot);

      pthread_mutex_unlock(&onboard_visual_pool_lock);

      gfx_log(LOG_DEBUG, "destroying vram-display %d [%p], map: %08x", slot, disp, onboard_visual_pool_map);
      break;

    case GFXTRANS_DISPLAY_MEMORY:
      gfx_log(LOG_DEBUG, "destroying ram-display [%p]", disp);

      ggiClose(disp->visual);
      free(disp);
    }
  } else {
    gfx_log(LOG_CRIT, "trying to destroy NULL display");
  }
}

/*
 * render a display to screen
 */
void
gfxtrans_display_render(gfxtrans_display_t *disp) {
  if (disp && disp->location == GFXTRANS_DISPLAY_ONBOARD) {
    gfx_log(LOG_DEBUG, "gfxtrans_display_render %p (frame %d)", disp->fb->read, disp->id);

    gfxtrans_lock_write_display(disp);
    ggiSetDisplayFrame(disp->visual, disp->id);
    gfxtrans_unlock_display(disp);

    gfxtrans_flush_display(disp);

  } else {
      gfx_log(LOG_CRIT, "can't render non-onboard display");

  }
}
