/*
 * blocky transitions
 *
 * provides blocky transformations
 *
 * $Id: blocky.c,v 1.8 2000/11/03 16:05:14 opencare Exp $
 *
 * Copyright (C) 2000 OpenCare (www.ocare.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  
 */

#include <stdlib.h>
#include "gfxtrans.h"
#include "blocky.h"

gfxtrans_display_t *
gfxtrans_blocky_display_hbars(imgsource_t imgsource, gfxtrans_display_t *fb) {
  int i, xstep;
  int stride_height = imgsource->height / GFXTRANS_BLOCKY_T(imgsource).strides;
  gfxtrans_display_t *disp;
  struct timespec requested, remaining;

  /* select a display to render */
  if (fb) {
    disp = fb;
  } else {
    disp = imgsource->display;
  }

  requested.tv_sec = 0;
  requested.tv_nsec = 10000000;
  
  /* acquire locks */
  gfxtrans_lock_read_display( GFXTRANS_BLOCKY_T(imgsource).prev->display );
  gfxtrans_lock_read_display( GFXTRANS_BLOCKY_T(imgsource).next->display );
  gfxtrans_lock_write_display( disp );

  xstep = GFXTRANS_ANIMATED_T(imgsource).tick * GFXTRANS_BLOCKY_T(imgsource).step; 
  stride_height = imgsource->height / ( 2 * GFXTRANS_BLOCKY_T(imgsource).strides );

  gfx_log(LOG_DEBUG, "gfxtrans_blocky_display_hbars(step = %d, tick = %d, disp to %p (frame %d))",
	  GFXTRANS_BLOCKY_T(imgsource).step, GFXTRANS_ANIMATED_T(imgsource).tick, disp->fb->write, disp->id);

  if (GFXTRANS_ANIMATED_T(imgsource).tick == GFXTRANS_ANIMATED_T(imgsource).frames) {
    gfxtrans_blit(disp,
		  GFXTRANS_BLOCKY_T(imgsource).next->display,
		  0, 0,
		  0, 0,
		  imgsource->width,
		  imgsource->height);
  } else {

  /* compute each stride */
  for (i = 0; i < GFXTRANS_BLOCKY_T(imgsource).strides; i++) {
    /* blit left-right */
    gfxtrans_blit(disp,
		  GFXTRANS_BLOCKY_T(imgsource).prev->display,
		  0,		/* sx */
		  stride_height * 2 * i, /* sy */
		  xstep,	/* dx */
		  stride_height * 2 * i, /* dy */
		  imgsource->width - xstep, /* width */
		  stride_height); /* height */
    gfxtrans_blit(disp,
		  GFXTRANS_BLOCKY_T(imgsource).next->display,
		  imgsource->width - xstep, /* sx */
		  stride_height * 2 * i, /* sy */
		  0,		/* dx */
		  stride_height * 2 * i, /* dy */
		  xstep,	/* width */
		  stride_height); /* height */

    /* blit right-left */
    gfxtrans_blit(disp,
		  GFXTRANS_BLOCKY_T(imgsource).prev->display,
		  xstep,	/* sx */
		  stride_height * (2 * i + 1), /* sy */
		  0,		/* dx */
		  stride_height * (2 * i + 1), /* dy */
		  imgsource->width - xstep, /* width */
		  stride_height); /* height */

    gfxtrans_blit(disp,
		  GFXTRANS_BLOCKY_T(imgsource).next->display,
		  0,		/* sx */
		  stride_height * (2 * i + 1), /* sy */
		  imgsource->width - xstep, /* dx */
		  stride_height * (2 * i + 1), /* dy */
		  xstep,	/* width */
		  stride_height); /* height */
  }
  }
  /* release locks */
  gfxtrans_unlock_display( disp );
  gfxtrans_unlock_display( GFXTRANS_BLOCKY_T(imgsource).prev->display );
  gfxtrans_unlock_display( GFXTRANS_BLOCKY_T(imgsource).next->display );
  gfxtrans_flush_display( disp );

  GFXTRANS_ANIMATED_T(imgsource).tick ++;
  nanosleep(&requested, &remaining);
  return NULL;
}

gfxtrans_display_t *
gfxtrans_blocky_display_vbars(imgsource_t img_source, gfxtrans_display_t *fb) {
  return NULL;
}

/*
 * create a blocky transition
 * prev and next sources must be the same size
 *
 * arg1: 'prev' image source
 * arg2: 'next' image source
 * arg3: transition type, 0 for random
 * ... : ignored
 */
imgsource_t 
gfxtrans_blocky_create(va_list ap) {
  imgsource_t prev_source, next_source, blocky_source;
  gfxtrans_blocky_transition_t transition_type;

  prev_source = va_arg(ap, imgsource_t);
  next_source = va_arg(ap, imgsource_t);
  transition_type = va_arg(ap, gfxtrans_blocky_transition_t);
  

  gfx_log(LOG_DEBUG, "gfxtrans_blocky_create(%p,%p,%d)",
	  prev_source, next_source, transition_type);

  if (prev_source && next_source) {

    if ( (prev_source->width == next_source->width)
	 && (prev_source->height == next_source->height) ) {
      /* create blocky source */
      blocky_source = (imgsource_t) malloc ( sizeof(struct _imgsource_t) );

      if (blocky_source) {
	blocky_source->display = gfxtrans_display_allocate(GFXTRANS_DISPLAY_ONBOARD,
							   prev_source->width,
							   prev_source->height);

	if (blocky_source->display) {

	  if ( !ggiAddFlags(blocky_source->display->visual, GGIFLAG_ASYNC) ) {

	  /* initialize member functions */
	  blocky_source->create_f = gfxtrans_blocky_create;
	  blocky_source->destroy_f = gfxtrans_blocky_destroy;

	  /* initialize member datas */
	  blocky_source->width = prev_source->width;
	  blocky_source->height = prev_source->height;
	  GFXTRANS_ANIMATED_T(blocky_source).framerate = 50;
	  GFXTRANS_ANIMATED_T(blocky_source).timelength = 1;
	  GFXTRANS_ANIMATED_T(blocky_source).frames = 
	    GFXTRANS_ANIMATED_T(blocky_source).framerate
	    * GFXTRANS_ANIMATED_T(blocky_source).timelength ; 
	  GFXTRANS_ANIMATED_T(blocky_source).tick = 0;
	  GFXTRANS_ANIMATED_T(blocky_source).still = 0;
	  GFXTRANS_BLOCKY_T(blocky_source).prev = prev_source;
	  GFXTRANS_BLOCKY_T(blocky_source).next = next_source;

	  /* pick up now number of strides */
	  GFXTRANS_BLOCKY_T(blocky_source).strides = 2 + rand() % 4 + rand() % 4;

	  switch (transition_type) {

	  case BLOCKY_RANDOM:
	    /* randomly pick-up BLOCKY_HBARS :) */

	  case BLOCKY_HBARS:
	    /* strides go left/right */

	    GFXTRANS_BLOCKY_T(blocky_source).step =
	      blocky_source->width / GFXTRANS_ANIMATED_T(blocky_source).frames;
	    blocky_source->display_f = gfxtrans_blocky_display_hbars;

	    return blocky_source;

	  case BLOCKY_VBARS:
	    /* strides go up/down */
	    GFXTRANS_BLOCKY_T(blocky_source).step =
	      blocky_source->height / GFXTRANS_ANIMATED_T(blocky_source).frames;
	    blocky_source->display_f = gfxtrans_blocky_display_vbars;

	    return blocky_source;

	  default:
	    gfx_log(LOG_CRIT, "invalid transition given: %d", transition_type);
	  }
	  gfxtrans_display_destroy(blocky_source->display);
	  } else {
	    gfx_log(LOG_CRIT, "can't set async flag");
	  }
	} else {
	  gfx_log(LOG_CRIT, "can't open display");
	}
	free(blocky_source);
      } else {
	gfx_log(LOG_CRIT, "can't mallocate imgsource object");
      }

    } else {
      gfx_log(LOG_CRIT, "prev and next are not the same size: %dx%d / %dx%d", 
	      prev_source->width, prev_source->height, 
	      next_source->width, next_source->height);
    }
  } else {
    gfx_log(LOG_CRIT, "NULL image source given");
  }

  /* reference last frames of previous image source */
  /* reference(prev); */

  /* prepare destination image */
/*    dest->create(); */

  return NULL;
}

void
gfxtrans_blocky_destroy(imgsource_t imgsource) {

  gfx_log(LOG_DEBUG, "gfxtrans_blocky_destroy(%p)",
	  imgsource);
  gfxtrans_display_destroy(imgsource->display);
  free(imgsource);

}
