/*****************************************************************************/
/* File: header.c                                                            */
/* Author: David Chatenay                                                    */
/* Last Modified: Mon Jan 20 1997                                            */
/*                                                                           */
/* Routines to manipulate file headers containing vital informations.        */
/*****************************************************************************/
#include <stdio.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include <sys/stat.h>
#include "common.h"
#include "crunch.h"
#include "buffer.h"
#include "header.h"



/* The real size of the header */
#define HEADER_REAL_SIZE 24
/* The offset to lengths of files */
#define LENGTH_OFFSET    4


int ReadHeader(int fd, CrunchHeader *h)
{
    byte buffer[HEADER_REAL_SIZE];
    word crc;

    TORI(Seek(fd, 0, SEEK_SET));
    TORI(ReadAbs(fd, buffer, HEADER_REAL_SIZE));
    crc = CRC16(buffer, 22);
    (void)memcpy(h->Signature, buffer, 4);
    h->OriginalLength = (buffer[4] << 24) + (buffer[5] << 16) +
                        (buffer[6] << 8) + buffer[7];
    h->CrunchedLength = (buffer[8] << 24) + (buffer[9] << 16) +
                        (buffer[10] << 8) + buffer[11];
    h->CRC = (buffer[12] << 24) + (buffer[13] << 16) +
             (buffer[14] << 8) + (buffer[15]);
    h->Method = buffer[16];
    h->V.Major = buffer[17];
    h->V.Minor = buffer[18];
    h->V.Revision = buffer[19];
    h->ExtensionLength = (buffer[20] << 8) + buffer[21];
    h->HeaderCrc = (buffer[22] << 8) + buffer[23];
    if (crc != h->HeaderCrc)
        return CHECK_CRC;

    return 0;
}



int WriteHeader(int fd, CrunchHeader *h)
{
    byte buffer[HEADER_REAL_SIZE];

    /* Go to beginning of file */
    TORI(Seek(fd, 0, SEEK_SET));
    /* Fill the buffer */
    (void)memcpy(buffer, LIBCRUNCH_SIGNATURE, 4);
    buffer[4]  = (h->OriginalLength >> 24) & 0xff;
    buffer[5]  = (h->OriginalLength >> 16) & 0xff;
    buffer[6]  = (h->OriginalLength >> 8) & 0xff;
    buffer[7]  = (h->OriginalLength) & 0xff;
    buffer[8]  = (h->CrunchedLength >> 24) & 0xff;
    buffer[9]  = (h->CrunchedLength >> 16) & 0xff;
    buffer[10] = (h->CrunchedLength >> 8) & 0xff;
    buffer[11] = (h->CrunchedLength) & 0xff;
    buffer[12] = (h->CRC >> 24) & 0xff;
    buffer[13] = (h->CRC >> 16) & 0xff;
    buffer[14] = (h->CRC >> 8) & 0xff;
    buffer[15] = (h->CRC) & 0xff;
    buffer[16] = (h->Method) & 0xff;
    buffer[17] = (h->V.Major) & 0xff;
    buffer[18] = (h->V.Minor) & 0xff;
    buffer[19] = (h->V.Revision) & 0xff;
    buffer[20] = (h->ExtensionLength >> 8) & 0xff;
    buffer[21] = (h->ExtensionLength) & 0xff;
    /* Compute CRC */
    h->HeaderCrc = CRC16(buffer, 22);
    buffer[22] = (h->HeaderCrc >> 8) & 0xff;
    buffer[23] = (h->HeaderCrc) & 0xff;
    /* Write the header */
    return Write(fd, buffer, HEADER_REAL_SIZE);
}



int ComputeHeader(int fd, CrunchHeader *h, byte method)
{
    (void)memcpy(h->Signature, LIBCRUNCH_SIGNATURE, 4);
    h->OriginalLength = 0; /* Temporary: wrote at the end of the process */
    h->CrunchedLength = 0; /* Temporary: wrote at the end of the process */
    h->CRC = 0; /* For future versions */
    h->Method = method; 
    h->V.Major = VERSION_MAJOR;
    h->V.Minor = VERSION_MINOR;
    h->V.Revision = VERSION_REVISION;
    h->ExtensionLength = 0; /* For future versions */
    h->HeaderCrc = 0; /* Temporary: wrote at the end of process */

    return 0;
}



int SeekRealPosition(int fd, CrunchHeader *h, off_t offset)
{
    TORI(Seek(fd, HEADER_REAL_SIZE + h->ExtensionLength + offset, SEEK_SET));

    return 0;
}


int CheckFileHeader(int fd, CrunchHeader *h)
{
    struct stat s;

    /* Check the signature of file */
    if (memcmp(h->Signature, LIBCRUNCH_SIGNATURE, 4) != 0) {
	ERROR("This is not a crunched file.");
	return CHECK_INVALID;
    }
    /* Check method index */
    if (h->Method == 0 || h->Method >= METHOD_INVALID)
        return CHECK_INVALID;
    /* Check version number */
    if (h->V.Major > VERSION_MAJOR || h->V.Minor > VERSION_MINOR)
        return CHECK_VERSION;
    /* Check size (real and expected) */
    if (fstat(fd, &s) == -1) {
	ERROR("Cannot stat file!");
	perror("Reason");
	return CHECK_ERROR;
    }
    if ((dword)s.st_size > h->CrunchedLength) {
	ERROR("Garbaged file!");
	return CHECK_CORRUPT;
    }
    if ((dword)s.st_size < h->CrunchedLength) {
	ERROR("Truncated file!");
	return CHECK_CORRUPT;
    }

    return h->Method;
}


int CheckFile(int fd)
{
    CrunchHeader h;
    int r;

    /* Read the header */
    if ((r = ReadHeader(fd, &h)) < 0)
        return r;

    /* Check it */
    return CheckFileHeader(fd, &h);
}


int WriteLength(CrunchHeader *h, Buffer *in, Buffer *out)
{
    TORI(Seek(out->fd, LENGTH_OFFSET, SEEK_SET));
    out->total += HEADER_REAL_SIZE;
    h->OriginalLength = in->total;
    h->CrunchedLength = out->total;

    return WriteHeader(out->fd, h);
}


byte IdentifyMethod(int fd)
{
    byte buffer[HEADER_REAL_SIZE];

    TORI(Seek(fd, 0, SEEK_SET));
    TORI(ReadAbs(fd, buffer, HEADER_REAL_SIZE));
    if (buffer[12] > 0 && buffer[12] < METHOD_INVALID)
        return buffer[12];
    else
        return METHOD_UNKNOWN;
}

