/*****************************************************************************/
/* File: decrunch.c                                                          */
/* Author: David Chatenay                                                    */
/* Last Modified: Thu Dec 05 1996                                            */
/*                                                                           */
/* The front-end to the crunch library, used to decrunch files.              */
/*****************************************************************************/
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include "crunch.h"


#define DECRUNCHER_VERSION "1.1"


static void Banner()
{
    (void)fprintf(stderr, "Decruncher v" DECRUNCHER_VERSION "\n");
    (void)fprintf(stderr, FULL_NAME " v" FULL_VERSION "\n");
    (void)fprintf(stderr, "By " AUTHOR_NAME "\n\n");
}


static void Usage(char *n)
{
    (void)fprintf(stderr, "Usage: %s file\n", n);
    exit(-1);
}


int main(int argc, char *argv[])
{
    int r=0, in, out, method;
    char FileName[256];
    dword l;

    /* First check */
    Banner();
    switch (argc) {
    case 1:
	Usage(argv[0]);
	break;
    case 2:
	if (strcmp(argv[1], "-h") == 0 || strcmp(argv[1], "--help") == 0)
	    Usage(argv[0]);
	break;
    default:
        Usage(argv[0]);
    }
    /* Compute out file name */
    l = strlen(argv[1]);
    if (l < 3) {
	(void)fprintf(stderr, "File name is too short!\n");
	Usage(argv[0]);
    }
    /* Regular suffix? */
    if (argv[1][l-3] != '.' || argv[1][l-2] != 'j' || argv[1][l-1] != 'z') {
	(void)fprintf(stderr, "Not a regular suffix!\n");
	Usage(argv[0]);
    }
    /* Build filename */
    (void)strncpy(FileName, argv[1], l-3);
    FileName[l-3] = '\0';
    (void)fprintf(stderr, "Original file: %s\n", FileName);
    /* Open in and out file */
    if ((in = OpenR(argv[1], 0644)) == -1)
        return -1;
    if ((out = OpenW(FileName, 0644)) == -1)
        return -1;
    /* Check file and find method */
    method = CheckFile(in);

    /* Switch on compression method */
    switch (method) {
    case CHECK_ERROR:
	(void)fprintf(stderr, "File I/O error\n");
	return -1;
    case CHECK_INVALID:
	(void)fprintf(stderr, "Not a recognized file.\n");
	return -1;
    case CHECK_CORRUPT:
	(void)fprintf(stderr, "Corrupted file.\n");
	return -1;
    case CHECK_CRC:
	(void)fprintf(stderr, "Header CRC check error.\n");
	return -1;
    case METHOD_UNKNOWN:
	(void)fprintf(stderr, "Unknow method.\n");
	return -1;
    case METHOD_AD_HUFF:
	(void)fprintf(stderr, "Method: Adaptative Huffman\n");
	r = AdaptativeHuffmanDecode(in, out);
	break;
    case METHOD_HUFFMAN:
	(void)fprintf(stderr, "Method: Huffman\n");
	r = HuffmanDecode(in, out);
	break;
    case METHOD_LZW:
	(void)fprintf(stderr, "Method: Lzw\n");
	r = LzwDecode(in, out);
	break;
    case METHOD_LZSS:
	(void)fprintf(stderr, "Method: Lzss\n");
	r = LzssDecode(in, out);
	break;
    default:
	(void)fprintf(stderr, "Invalid suffix!\n");
	Usage(argv[0]);
    }

    /* Test return value */
    if (r == -1) {
        (void)fprintf(stderr, "There was an error during decoding.\n");
	(void)close(out);
	if (unlink(FileName) == -1)
	    perror(FileName);
	return -1;
    }
	  
    /* Close files */
    (void)close(in);
    (void)close(out);

    return r;
}
